(function() {
	'use strict';

	// Check if wp is available
	if (typeof wp === 'undefined') {
		return;
	}

	const { registerBlockType } = wp.blocks;
	const { createElement: el, useState, useEffect } = wp.element;
	const { __ } = wp.i18n;
	const { 
		PanelBody, 
		SelectControl, 
		TextControl, 
		ToggleControl, 
		RangeControl,
		TextareaControl,
		Button,
		Notice,
		Spinner
	} = wp.components;
	const { InspectorControls, useBlockProps } = wp.blockEditor;

	// Get the TokenMultiSelectControl from global scope
	const TokenMultiSelectControl = window.TokenMultiSelectControl;
	
	// Fallback component if TokenMultiSelectControl is not available
	const FallbackFieldControl = ({ label, help, value, onChange }) => {
		const stringValue = Array.isArray(value) ? value.join(',') : value;
		return el(TextControl, {
			label: label + ' (Fallback)',
			value: stringValue,
			onChange: (newValue) => {
				const arrayValue = newValue ? newValue.split(',').map(v => v.trim()).filter(v => v) : [];
				onChange(arrayValue);
			},
			help: help + ' ' + __('(Using fallback - comma-separated IDs)', 'displayformentries')
		});
	};

	// Helper function to fetch form fields
	function useFormFields(formId) {
		const [fields, setFields] = useState([]);
		const [loading, setLoading] = useState(false);

		useEffect(() => {
			if (!formId || !window.dfeBlocks) {
				setFields([]);
				return;
			}

			setLoading(true);
			
			const formData = new FormData();
			formData.append('action', 'dfe_get_form_fields');
			formData.append('form_id', formId);
			formData.append('nonce', window.dfeBlocks.nonce);

			fetch(window.dfeBlocks.ajaxUrl, {
				method: 'POST',
				body: formData
			})
			.then(response => response.json())
			.then(data => {
				if (data.success) {
					setFields(data.data.fields || []);
				} else {
					console.error('DFE: Failed to fetch form fields:', data.data?.message || 'Unknown error');
					setFields([]);
				}
			})
			.catch(error => {
				console.error('DFE: Error fetching form fields:', error);
				setFields([]);
			})
			.finally(() => {
				setLoading(false);
			});
		}, [formId]);

		return { fields, loading };
	}

	// Entries Block
	registerBlockType('displayformentries/entries', {
		title: __('Display Form Entries', 'displayformentries'),
		icon: 'list-view',
		category: 'widgets',
		attributes: {
			formId: {
				type: 'number',
				default: 0
			},
			template: {
				type: 'string',
				default: 'list'
			},
			limit: {
				type: 'number',
				default: 10
			},
			fields: {
				type: 'string',
				default: ''
			},
			hideEmpty: {
				type: 'boolean',
				default: true
			},
			showLabels: {
				type: 'boolean',
				default: true
			},
			linkField: {
				type: 'string',
				default: ''
			},
			showEdit: {
				type: 'boolean',
				default: true
			},
			editText: {
				type: 'string',
				default: 'Edit'
			}
		},

		edit: function(props) {
			const { attributes, setAttributes } = props;
			const { formId, template, limit, fields, hideEmpty, showLabels, linkField, showEdit, editText } = attributes;
			const blockProps = useBlockProps();

			// Get forms from localized data
			const forms = window.dfeBlocks ? window.dfeBlocks.forms : [];
			const formLabel = forms.find(f => f.value == formId)?.label || '';
			
			// Get form fields
			const { fields: formFields, loading: fieldsLoading } = useFormFields(formId);

			return el('div', blockProps,
				el(InspectorControls, {},
					el(PanelBody, { title: __('Form Settings', 'displayformentries'), initialOpen: true },
						el(SelectControl, {
							label: __('Select Form', 'displayformentries'),
							value: formId,
							options: [
								{ value: 0, label: __('-- Select a Form --', 'displayformentries') },
								...forms
							],
							onChange: (value) => setAttributes({ formId: parseInt(value) })
						})
					),
					el(PanelBody, { title: __('Display Options', 'displayformentries'), initialOpen: false },
						el(SelectControl, {
							label: __('Template', 'displayformentries'),
							value: template,
							options: [
								{ value: 'list', label: __('List', 'displayformentries') },
								{ value: 'table', label: __('Table', 'displayformentries') },
								{ value: 'cards', label: __('Cards', 'displayformentries') },
								{ value: 'linked-list', label: __('Linked List', 'displayformentries') }
							],
							onChange: (value) => setAttributes({ template: value })
						}),
						el(RangeControl, {
							label: __('Number of Entries', 'displayformentries'),
							value: limit,
							onChange: (value) => setAttributes({ limit: value }),
							min: 1,
							max: 100
						}),
						// Show spinner while loading fields, or the control when ready
						formId ? (
							fieldsLoading ? 
								el('div', { style: { textAlign: 'center', padding: '20px' } },
									el(Spinner)
								) :
								TokenMultiSelectControl ? 
									el(TokenMultiSelectControl, {
										label: __('Select Fields to Display', 'displayformentries'),
										help: __('Leave empty to show all fields. Click tokens to remove them.', 'displayformentries'),
										value: fields || [],
										options: formFields,
										onChange: (value) => setAttributes({ fields: value })
									}) :
									el(FallbackFieldControl, {
										label: __('Select Fields to Display', 'displayformentries'),
										help: __('Leave empty to show all fields.', 'displayformentries'),
										value: fields || [],
										onChange: (value) => setAttributes({ fields: value })
									})
						) : el(Notice, { status: 'warning', isDismissible: false },
							__('Please select a form first to choose fields.', 'displayformentries')
						),
						el(ToggleControl, {
							label: __('Hide Empty Fields', 'displayformentries'),
							checked: hideEmpty,
							onChange: (value) => setAttributes({ hideEmpty: value })
						}),
						el(ToggleControl, {
							label: __('Show Field Labels', 'displayformentries'),
							checked: showLabels,
							onChange: (value) => setAttributes({ showLabels: value })
						}),
						el(ToggleControl, {
							label: __('Show Edit Links', 'displayformentries'),
							checked: showEdit,
							onChange: (value) => setAttributes({ showEdit: value })
						})
					),
					template === 'linked-list' && el(PanelBody, { title: __('Linked List Options', 'displayformentries'), initialOpen: false },
						// Show spinner while loading fields, or the control when ready
						formId ? (
							fieldsLoading ? 
								el('div', { style: { textAlign: 'center', padding: '20px' } },
									el(Spinner)
								) :
								TokenMultiSelectControl ? 
									el(TokenMultiSelectControl, {
										label: __('Select Link Field', 'displayformentries'),
										help: __('Choose which field should be clickable in the linked list view.', 'displayformentries'),
										value: linkField || [],
										options: formFields,
										onChange: (value) => setAttributes({ linkField: value })
									}) :
									el(FallbackFieldControl, {
										label: __('Select Link Field', 'displayformentries'),
										help: __('Choose which field should be clickable.', 'displayformentries'),
										value: linkField || [],
										onChange: (value) => setAttributes({ linkField: value })
									})
						) : el(Notice, { status: 'warning', isDismissible: false },
							__('Please select a form first to choose link field.', 'displayformentries')
						)
					)
				),
				el('div', { className: 'dfe-block-preview' },
					formId ?
						el('div', { className: 'dfe-block-static-preview' },
							el('strong', {}, __('Display Form Entries', 'displayformentries')),
							el('ul', {},
								el('li', {}, __('Form:', 'displayformentries') + ' ' + (formLabel || formId)),
								el('li', {}, __('Template:', 'displayformentries') + ' ' + template),
								el('li', {}, __('Entries:', 'displayformentries') + ' ' + limit),
								fields && fields.length > 0 && el('li', {}, __('Fields:', 'displayformentries') + ' ' + 
									fields.map(fieldId => {
										const field = formFields.find(f => f.value === fieldId);
										return field ? field.label : fieldId;
									}).join(', ')
								),
								template === 'linked-list' && linkField && linkField.length > 0 && el('li', {}, __('Link Field:', 'displayformentries') + ' ' + 
									linkField.map(fieldId => {
										const field = formFields.find(f => f.value === fieldId);
										return field ? field.label : fieldId;
									}).join(', ')
								),
								el('li', {}, __('Hide Empty Fields:', 'displayformentries') + ' ' + (hideEmpty ? __('Yes', 'displayformentries') : __('No', 'displayformentries'))),
								el('li', {}, __('Show Field Labels:', 'displayformentries') + ' ' + (showLabels ? __('Yes', 'displayformentries') : __('No', 'displayformentries'))),
								el('li', {}, __('Show Edit Links:', 'displayformentries') + ' ' + (showEdit ? __('Yes', 'displayformentries') : __('No', 'displayformentries')))
							)
						)
					:
						el('div', { className: 'dfe-block-static-preview' },
							el('strong', {}, __('Display Form Entries', 'displayformentries')),
							el('p', {}, __('Please select a form to display entries.', 'displayformentries'))
						)
				)
			);
		},

		save: function() {
			return null; // Dynamic block
		}
	});

	// Entry Block
	registerBlockType('displayformentries/entry', {
		title: __('Display Single Entry', 'displayformentries'),
		icon: 'admin-post',
		category: 'widgets',
		attributes: {
			entryId: {
				type: 'number',
				default: 0
			},
			formId: {
				type: 'number',
				default: 0
			},
			fields: {
				type: 'array',
				default: []
			},
			hideEmpty: {
				type: 'boolean',
				default: true
			},
			showLabels: {
				type: 'boolean',
				default: true
			}
		},

		edit: function(props) {
			const { attributes, setAttributes } = props;
			const { entryId, formId, fields, hideEmpty, showLabels } = attributes;

			const blockProps = useBlockProps();
			
			// Get forms from localized data
			const forms = window.dfeBlocks ? window.dfeBlocks.forms : [];
			
			// Get form fields
			const { fields: formFields, loading: fieldsLoading } = useFormFields(formId);

			const shortcode = `[dfe_entry entry_id="${entryId}" form_id="${formId}" fields="${Array.isArray(fields) ? fields.join(',') : fields}" hide_empty="${hideEmpty}" show_labels="${showLabels}"]`;

			return el('div', blockProps,
				el(InspectorControls, {},
					el(PanelBody, { title: __('Entry Settings', 'displayformentries'), initialOpen: true },
						el(SelectControl, {
							label: __('Select Form', 'displayformentries'),
							value: formId,
							options: [
								{ value: 0, label: __('-- Select a Form --', 'displayformentries') },
								...forms
							],
							onChange: (value) => setAttributes({ formId: parseInt(value) })
						}),
						el(TextControl, {
							label: __('Entry ID', 'displayformentries'),
							value: entryId,
							onChange: (value) => setAttributes({ entryId: parseInt(value) }),
							type: 'number'
						}),
						// Show spinner while loading fields, or the control when ready
						formId ? (
							fieldsLoading ? 
								el('div', { style: { textAlign: 'center', padding: '20px' } },
									el(Spinner)
								) :
								TokenMultiSelectControl ? 
									el(TokenMultiSelectControl, {
										label: __('Select Fields to Display', 'displayformentries'),
										help: __('Leave empty to show all fields. Click tokens to remove them.', 'displayformentries'),
										value: fields || [],
										options: formFields,
										onChange: (value) => setAttributes({ fields: value })
									}) :
									el(FallbackFieldControl, {
										label: __('Select Fields to Display', 'displayformentries'),
										help: __('Leave empty to show all fields.', 'displayformentries'),
										value: fields || [],
										onChange: (value) => setAttributes({ fields: value })
									})
						) : el(Notice, { status: 'warning', isDismissible: false },
							__('Please select a form first to choose fields.', 'displayformentries')
						),
						el(ToggleControl, {
							label: __('Hide Empty Fields', 'displayformentries'),
							checked: hideEmpty,
							onChange: (value) => setAttributes({ hideEmpty: value })
						}),
						el(ToggleControl, {
							label: __('Show Field Labels', 'displayformentries'),
							checked: showLabels,
							onChange: (value) => setAttributes({ showLabels: value })
						})
					)
				),
				el('div', { className: 'dfe-block-preview' },
					entryId ? 
						el('div', { className: 'dfe-shortcode-preview' },
							el('strong', {}, __('Preview:', 'displayformentries')),
							el('code', {}, shortcode)
						) :
						el(Notice, { status: 'warning' },
							__('Please enter an entry ID to display.', 'displayformentries')
						)
				)
			);
		},

		save: function() {
			return null; // Dynamic block
		}
	});

	// Form Block
	registerBlockType('displayformentries/form', {
		title: __('Display Form', 'displayformentries'),
		icon: 'forms',
		category: 'widgets',
		attributes: {
			formId: {
				type: 'number',
				default: 0
			},
			title: {
				type: 'boolean',
				default: true
			},
			description: {
				type: 'boolean',
				default: true
			}
		},

		edit: function(props) {
			const { attributes, setAttributes } = props;
			const { formId, title, description } = attributes;

			const blockProps = useBlockProps();

			const forms = window.dfeBlocks ? window.dfeBlocks.forms : [];

			const shortcode = `[dfe_form form_id="${formId}" title="${title}" description="${description}"]`;

			return el('div', blockProps,
				el(InspectorControls, {},
					el(PanelBody, { title: __('Form Settings', 'displayformentries'), initialOpen: true },
						el(SelectControl, {
							label: __('Select Form', 'displayformentries'),
							value: formId,
							options: [
								{ value: 0, label: __('-- Select a Form --', 'displayformentries') },
								...forms
							],
							onChange: (value) => setAttributes({ formId: parseInt(value) })
						}),
						el(ToggleControl, {
							label: __('Show Title', 'displayformentries'),
							checked: title,
							onChange: (value) => setAttributes({ title: value })
						}),
						el(ToggleControl, {
							label: __('Show Description', 'displayformentries'),
							checked: description,
							onChange: (value) => setAttributes({ description: value })
						})
					)
				),
				el('div', { className: 'dfe-block-preview' },
					formId ? 
						el('div', { className: 'dfe-shortcode-preview' },
							el('strong', {}, __('Preview:', 'displayformentries')),
							el('code', {}, shortcode)
						) :
						el(Notice, { status: 'warning' },
							__('Please select a form to display.', 'displayformentries')
						)
				)
			);
		},

		save: function() {
			return null; // Dynamic block
		}
	});

})(); 