<?php
/**
 * API functions for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Ristretto Apps
 * @link      https://ristrettoapps.com
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_API {

	/**
	 * Get field label
	 *
	 * @param array  $field Field array
	 * @param array  $entry Entry array
	 * @param bool   $force_show_label Whether to force show label
	 *
	 * @return string Field label
	 */
	public static function field_label( $field, $entry = array(), $force_show_label = false ) {
		if ( empty( $field ) ) {
			return '';
		}

		$label = isset( $field['label'] ) ? $field['label'] : '';

		// Handle entry meta fields
		if ( isset( $field['type'] ) && $field['type'] === 'entry_meta' ) {
			return $label; // Already set in the virtual field creation
		}

		// Handle choice-based fields
		if ( ! empty( $entry ) && isset( $field['choices'] ) && is_array( $field['choices'] ) ) {
			$field_value = isset( $entry[ $field['id'] ] ) ? $entry[ $field['id'] ] : '';
			
			if ( ! empty( $field_value ) ) {
				foreach ( $field['choices'] as $choice ) {
					if ( isset( $choice['value'] ) && $choice['value'] === $field_value ) {
						$label = isset( $choice['text'] ) ? $choice['text'] : $label;
						break;
					}
				}
			}
		}

		return $label;
	}

	/**
	 * Replace merge tags in text
	 *
	 * @param string $text Text with merge tags
	 * @param array  $form Form array
	 * @param array  $entry Entry array
	 * @param bool   $url_encode Whether to URL encode
	 * @param bool   $esc_html Whether to escape HTML
	 * @param bool   $nl2br Whether to convert newlines to <br>
	 * @param string $format Output format
	 * @param array  $aux_data Additional data
	 *
	 * @return string Processed text
	 */
	public static function replace_variables( $text, $form = array(), $entry = array(), $url_encode = false, $esc_html = true, $nl2br = true, $format = 'html', $aux_data = array() ) {
		if ( empty( $text ) ) {
			return '';
		}

		// Simple merge tag replacement
		$text = self::replace_simple_merge_tags( $text, $entry );

		if ( $url_encode ) {
			$text = urlencode( $text );
		}

		if ( $esc_html ) {
			$text = esc_html( $text );
		}

		if ( $nl2br ) {
			$text = nl2br( $text );
		}

		return $text;
	}

	/**
	 * Replace simple merge tags
	 *
	 * @param string $text Text with merge tags
	 * @param array  $entry Entry array
	 *
	 * @return string Processed text
	 */
	private static function replace_simple_merge_tags( $text, $entry ) {
		if ( empty( $entry ) || ! is_array( $entry ) ) {
			return $text;
		}

		// Replace {all_fields} with all field values
		if ( strpos( $text, '{all_fields}' ) !== false ) {
			$all_fields = self::get_all_fields_text( $entry );
			$text = str_replace( '{all_fields}', $all_fields, $text );
		}

		// Replace {entry_id} with entry ID
		if ( strpos( $text, '{entry_id}' ) !== false ) {
			$entry_id = isset( $entry['id'] ) ? $entry['id'] : '';
			$text = str_replace( '{entry_id}', $entry_id, $text );
		}

		// Replace {entry_date} with entry date
		if ( strpos( $text, '{entry_date}' ) !== false ) {
			$entry_date = isset( $entry['date_created'] ) ? DFE_Common::format_date( $entry['date_created'] ) : '';
			$text = str_replace( '{entry_date}', $entry_date, $text );
		}

		return $text;
	}

	/**
	 * Get all fields as text
	 *
	 * @param array $entry Entry array
	 *
	 * @return string All fields text
	 */
	private static function get_all_fields_text( $entry ) {
		if ( empty( $entry ) || ! is_array( $entry ) ) {
			return '';
		}

		$output = '';
		
		foreach ( $entry as $key => $value ) {
			// Skip system fields
			if ( in_array( $key, array( 'id', 'form_id', 'date_created', 'date_updated', 'is_starred', 'is_read', 'ip', 'source_url', 'user_agent', 'currency' ) ) ) {
				continue;
			}

			if ( ! empty( $value ) ) {
				$output .= $key . ': ' . $value . "\n";
			}
		}

		return trim( $output );
	}

	/**
	 * Get field width
	 *
	 * @param array  $field Field array
	 * @param string $format Format string
	 *
	 * @return string Field width
	 */
	public static function field_width( $field, $format = '%d%%' ) {
		if ( empty( $field ) || ! isset( $field['width'] ) ) {
			return '';
		}

		return sprintf( $format, $field['width'] );
	}

	/**
	 * Get field CSS class
	 *
	 * @param array $field Field array
	 * @param array $form Form array
	 * @param array $entry Entry array
	 *
	 * @return string Field CSS class
	 */
	public static function field_class( $field, $form = null, $entry = null ) {
		if ( empty( $field ) ) {
			return '';
		}

		$classes = array();

		// Add field type class
		if ( isset( $field['type'] ) ) {
			$classes[] = 'dfe-field-' . sanitize_html_class( $field['type'] );
		}

		// Add field ID class
		if ( isset( $field['id'] ) ) {
			$classes[] = 'dfe-field-' . intval( $field['id'] );
		}

		// Add required class
		if ( isset( $field['isRequired'] ) && $field['isRequired'] ) {
			$classes[] = 'dfe-field-required';
		}

		// Add admin only class
		if ( isset( $field['adminOnly'] ) && $field['adminOnly'] ) {
			$classes[] = 'dfe-field-admin-only';
		}

		return implode( ' ', $classes );
	}

	/**
	 * Get field HTML ID
	 *
	 * @param array $field Field array
	 * @param array $form Form array
	 * @param array $entry Entry array
	 *
	 * @return string Field HTML ID
	 */
	public static function field_html_attr_id( $field, $form = array(), $entry = array() ) {
		if ( empty( $field ) || ! isset( $field['id'] ) ) {
			return '';
		}

		$id = 'dfe-field-' . intval( $field['id'] );

		if ( ! empty( $entry ) && isset( $entry['id'] ) ) {
			$id .= '-entry-' . intval( $entry['id'] );
		}

		return $id;
	}

	/**
	 * Get field value
	 *
	 * @param array  $entry Entry array
	 * @param array  $field_settings Field settings
	 * @param string $format Output format
	 *
	 * @return string Field value
	 */
	public static function field_value( $entry, $field_settings, $format = 'html' ) {
		if ( empty( $entry ) || empty( $field_settings ) ) {
			return '';
		}

		$field_id = isset( $field_settings['id'] ) ? $field_settings['id'] : '';
		
		if ( empty( $field_id ) ) {
			return '';
		}

		// Handle entry meta fields
		$entry_meta_fields = array( 'id', 'date_created', 'date_updated', 'is_starred', 'is_read', 'ip', 'source_url', 'user_agent', 'payment_date', 'payment_amount', 'payment_status', 'transaction_id', 'created_by', 'post_id' );
		
		if ( in_array( $field_id, $entry_meta_fields ) ) {
			$value = isset( $entry[ $field_id ] ) ? $entry[ $field_id ] : '';
			
			// Format specific entry meta fields
			switch ( $field_id ) {
				case 'date_created':
				case 'date_updated':
				case 'payment_date':
					$value = DFE_Common::format_date( $value );
					break;
				case 'is_starred':
				case 'is_read':
					$value = $value ? __( 'Yes', 'displayformentries' ) : __( 'No', 'displayformentries' );
					break;
				case 'payment_amount':
					if ( ! empty( $value ) && is_numeric( $value ) ) {
						$value = '$' . number_format( floatval( $value ), 2 );
					}
					break;
				case 'created_by':
					if ( ! empty( $value ) && is_numeric( $value ) ) {
						$user = get_user_by( 'id', $value );
						if ( $user ) {
							$value = $user->display_name . ' (' . $value . ')';
						}
					}
					break;
				default:
					if ( $format === 'html' ) {
						$value = esc_html( $value );
					}
					break;
			}
			
			return $value;
		}

		// Check if specific subfields are selected for this field
		$selected_subfields = isset( $GLOBALS['dfe_selected_subfields'][ $field_id ] ) ? 
			$GLOBALS['dfe_selected_subfields'][ $field_id ] : array();

		if ( ! empty( $selected_subfields ) ) {
			// Handle selected subfields
			$parts = array();
			foreach ( $selected_subfields as $subfield_id ) {
				$subfield_value = isset( $entry[ $subfield_id ] ) ? trim( $entry[ $subfield_id ] ) : '';
				if ( ! empty( $subfield_value ) ) {
					$parts[] = $subfield_value;
				}
			}
			$value = implode( ' ', $parts );
		} else {
			// Handle complete field
			// Special handling for Gravity Forms "Name" field
			if ( isset($field_settings['type']) && $field_settings['type'] === 'name' ) {
				// Gravity Forms subfield keys
				$parts = array();
				$prefix = isset($entry["{$field_id}.2"]) ? $entry["{$field_id}.2"] : '';
				$first  = isset($entry["{$field_id}.3"]) ? $entry["{$field_id}.3"] : '';
				$middle = isset($entry["{$field_id}.4"]) ? $entry["{$field_id}.4"] : '';
				$last   = isset($entry["{$field_id}.6"]) ? $entry["{$field_id}.6"] : '';
				$suffix = isset($entry["{$field_id}.8"]) ? $entry["{$field_id}.8"] : '';

				if ( $prefix ) $parts[] = $prefix;
				if ( $first )  $parts[] = $first;
				if ( $middle ) $parts[] = $middle;
				if ( $last )   $parts[] = $last;
				if ( $suffix ) $parts[] = $suffix;

				$value = implode(' ', array_filter($parts));
			} elseif ( isset($field_settings['type']) && $field_settings['type'] === 'date' ) {
				// Handle date fields with subfields
				$parts = array();
				$month = isset($entry["{$field_id}.1"]) ? $entry["{$field_id}.1"] : '';
				$day   = isset($entry["{$field_id}.2"]) ? $entry["{$field_id}.2"] : '';
				$year  = isset($entry["{$field_id}.3"]) ? $entry["{$field_id}.3"] : '';

				if ( $month || $day || $year ) {
					// Reconstruct date if parts exist
					$date_parts = array_filter( array( $month, $day, $year ) );
					if ( count( $date_parts ) === 3 ) {
						$value = $month . '/' . $day . '/' . $year;
					} else {
						$value = implode( ' ', $date_parts );
					}
				} else {
					$value = isset( $entry[ $field_id ] ) ? $entry[ $field_id ] : '';
				}
			} else {
				$value = isset( $entry[ $field_id ] ) ? $entry[ $field_id ] : '';
			}
		}

		// Handle empty values
		if ( DFE_Common::is_empty( $value ) ) {
			return '';
		}

		// Format based on field type
		$field_type = isset( $field_settings['type'] ) ? $field_settings['type'] : '';
		$value = self::format_field_value( $value, $field_type, $field_settings, $format );

		return $value;
	}

	/**
	 * Format field value based on type
	 *
	 * @param mixed  $value Field value
	 * @param string $field_type Field type
	 * @param array  $field_settings Field settings
	 * @param string $format Output format
	 *
	 * @return string Formatted value
	 */
	private static function format_field_value( $value, $field_type, $field_settings, $format ) {
		switch ( $field_type ) {
			case 'name':

				// Gravity Forms "Name" field is an array: ['first' => ..., 'last' => ...]
				if ( is_array( $value ) ) {
					$parts = array();
					if ( !empty( $value['prefix'] ) ) $parts[] = $value['prefix'];
					if ( !empty( $value['first'] ) ) $parts[] = $value['first'];
					if ( !empty( $value['middle'] ) ) $parts[] = $value['middle'];
					if ( !empty( $value['last'] ) ) $parts[] = $value['last'];
					if ( !empty( $value['suffix'] ) ) $parts[] = $value['suffix'];
					$value = implode( ' ', $parts );
				}
				if ( $format === 'html' ) {
					$value = esc_html( $value );
				}
				break;

			case 'date':
				$value = DFE_Common::format_date( $value );
				break;

			case 'email':
				if ( $format === 'html' ) {
					$value = sprintf( '<a href="mailto:%s">%s</a>', esc_attr( $value ), esc_html( $value ) );
				}
				break;

			case 'url':
			case 'website':
				if ( $format === 'html' ) {
					$value = sprintf( '<a href="%s" target="_blank">%s</a>', esc_url( $value ), esc_html( $value ) );
				}
				break;

			case 'fileupload':
				$value = self::format_file_upload( $value, $field_settings, $format );
				break;

			case 'list':
				$value = self::format_list_field( $value, $field_settings, $format );
				break;

			case 'checkbox':
				$value = self::format_checkbox_field( $value, $field_settings, $format );
				break;

			case 'radio':
			case 'select':
				$value = self::format_choice_field( $value, $field_settings, $format );
				break;

			case 'textarea':
				if ( $format === 'html' ) {
					$value = nl2br( esc_html( $value ) );
				}
				break;

			default:
				if ( $format === 'html' ) {
					$value = esc_html( $value );
				}
				break;
		}

		return $value;
	}

	/**
	 * Format file upload field
	 *
	 * @param mixed  $value Field value
	 * @param array  $field_settings Field settings
	 * @param string $format Output format
	 *
	 * @return string Formatted value
	 */
	private static function format_file_upload( $value, $field_settings, $format ) {
		if ( empty( $value ) ) {
			return '';
		}

		// Handle multiple files
		if ( is_array( $value ) ) {
			$files = array();
			foreach ( $value as $file ) {
				$files[] = self::format_single_file( $file, $format );
			}
			return implode( ', ', $files );
		}

		return self::format_single_file( $value, $format );
	}

	/**
	 * Format single file
	 *
	 * @param string $file File URL
	 * @param string $format Output format
	 *
	 * @return string Formatted file
	 */
	private static function format_single_file( $file, $format ) {
		if ( empty( $file ) ) {
			return '';
		}

		$filename = basename( $file );

		if ( $format === 'html' ) {
			return sprintf( '<a href="%s" target="_blank">%s</a>', esc_url( $file ), esc_html( $filename ) );
		}

		return $filename;
	}

	/**
	 * Format list field
	 *
	 * @param mixed  $value Field value
	 * @param array  $field_settings Field settings
	 * @param string $format Output format
	 *
	 * @return string Formatted value
	 */
	private static function format_list_field( $value, $field_settings, $format ) {
		if ( empty( $value ) ) {
			return '';
		}

		if ( ! is_array( $value ) ) {
			return esc_html( $value );
		}

		if ( $format === 'html' ) {
			$output = '<ul class="dfe-list-field">';
			foreach ( $value as $row ) {
				if ( is_array( $row ) ) {
					$output .= '<li>' . esc_html( implode( ' - ', $row ) ) . '</li>';
				} else {
					$output .= '<li>' . esc_html( $row ) . '</li>';
				}
			}
			$output .= '</ul>';
			return $output;
		}

		return implode( ', ', array_map( function( $row ) {
			return is_array( $row ) ? implode( ' - ', $row ) : $row;
		}, $value ) );
	}

	/**
	 * Format checkbox field
	 *
	 * @param mixed  $value Field value
	 * @param array  $field_settings Field settings
	 * @param string $format Output format
	 *
	 * @return string Formatted value
	 */
	private static function format_checkbox_field( $value, $field_settings, $format ) {
		if ( empty( $value ) ) {
			return '';
		}

		// Handle multiple checkboxes
		if ( is_array( $value ) ) {
			$values = array();
			foreach ( $value as $val ) {
				$values[] = self::format_choice_value( $val, $field_settings );
			}
			return implode( ', ', $values );
		}

		return self::format_choice_value( $value, $field_settings );
	}

	/**
	 * Format choice field (radio, select)
	 *
	 * @param mixed  $value Field value
	 * @param array  $field_settings Field settings
	 * @param string $format Output format
	 *
	 * @return string Formatted value
	 */
	private static function format_choice_field( $value, $field_settings, $format ) {
		if ( empty( $value ) ) {
			return '';
		}

		return self::format_choice_value( $value, $field_settings );
	}

	/**
	 * Format choice value
	 *
	 * @param mixed $value Field value
	 * @param array $field_settings Field settings
	 *
	 * @return string Formatted value
	 */
	private static function format_choice_value( $value, $field_settings ) {
		if ( empty( $field_settings['choices'] ) || ! is_array( $field_settings['choices'] ) ) {
			return esc_html( $value );
		}

		foreach ( $field_settings['choices'] as $choice ) {
			if ( isset( $choice['value'] ) && $choice['value'] === $value ) {
				return esc_html( isset( $choice['text'] ) ? $choice['text'] : $value );
			}
		}

		return esc_html( $value );
	}

	/**
	 * Generate entry link HTML
	 *
	 * @param array  $entry Entry array
	 * @param string $anchor_text Link text
	 * @param array  $passed_tag_atts Link attributes
	 * @param array  $field_settings Field settings
	 * @param int    $base_id Base ID
	 *
	 * @return string|null Link HTML
	 */
	public static function entry_link_html( $entry = array(), $anchor_text = '', $passed_tag_atts = array(), $field_settings = array(), $base_id = null ) {
		if ( empty( $entry ) || ! is_array( $entry ) || ! isset( $entry['id'] ) ) {
			return null;
		}

		$href = self::entry_link( $entry, $base_id );

		if ( '' === $href ) {
			return null;
		}

		$link = DFE_Common::get_link_html( $href, $anchor_text, $passed_tag_atts );

		/**
		 * Modify the link HTML.
		 *
		 * @param string $link HTML output of the link
		 * @param string $href URL of the link
		 * @param array  $entry The GF entry array
		 * @param array  $field_settings Settings for the particular field
		 */
		$output = apply_filters( 'dfe_field_entry_link', $link, $href, $entry, $field_settings );

		return $output;
	}

	/**
	 * Generate entry link URL
	 *
	 * @param array $entry Entry array
	 * @param int   $base_id Base ID
	 *
	 * @return string Entry link URL
	 */
	public static function entry_link( $entry, $base_id = null ) {
		if ( empty( $entry ) || ! is_array( $entry ) || ! isset( $entry['id'] ) ) {
			return '';
		}

		$entry_id = $entry['id'];
		$form_id = isset( $entry['form_id'] ) ? $entry['form_id'] : '';

		// Create a simple entry URL
		$url = add_query_arg( array(
			'entry_id' => $entry_id,
			'form_id'  => $form_id,
		), get_permalink( $base_id ) );

		return $url;
	}

	/**
	 * Get "No Results" text
	 *
	 * @param bool $wpautop Apply wpautop
	 *
	 * @return string No results text
	 */
	public static function no_results( $wpautop = true ) {
		$output = __( 'No entries match your request.', 'displayformentries' );

		if ( $wpautop ) {
			$output = wpautop( $output );
		}

		return $output;
	}
} 