<?php
/**
 * Common functions for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Ristretto Apps
 * @link      https://ristrettoapps.com
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_Common {
	/**
	 * Contains micro cached Forms.
	 *
	 * @var array
	 */
	private static array $forms = [];

	/**
	 * Clears the internal microcache.
	 */
	public static function clear_cache(): void {
		self::$forms = [];
	}

	/**
	 * Returns the form object for a given Form ID.
	 *
	 * @param int|string $form_id The Form ID.
	 *
	 * @return array|false Array: Form object returned from Gravity Forms; False: no form ID specified or Gravity Forms isn't active.
	 */
	public static function get_form( $form_id ) {
		if ( empty( $form_id ) ) {
			return false;
		}

		if ( ! class_exists( 'GFAPI' ) ) {
			return false;
		}

		if ( isset( self::$forms[ $form_id ] ) ) {
			return self::$forms[ $form_id ];
		}

		$form = \GFAPI::get_form( $form_id );

		self::$forms[ $form_id ] = $form;

		return $form;
	}

	/**
	 * Alias of GFAPI::get_forms()
	 *
	 * @param bool|string  $active Status of forms. Use `any` to get array of forms with any status. Default: `true`
	 * @param bool         $trash Include forms in trash? Default: `false`
	 * @param string|array $order_by Optional. Either the field name to order by or an array of multiple orderby fields as $orderby => $order.
	 * @param string       $order Optional. Either 'ASC' or 'DESC'. Only used if $orderby is a string.
	 *
	 * @return array Empty array if GFAPI class isn't available or no forms. Otherwise, the array of Forms
	 */
	public static function get_forms( $active = true, $trash = false, $order_by = 'date_created', $order = 'ASC' ) {
		if ( ! class_exists( 'GFAPI' ) ) {
			return array();
		}

		$forms = \GFAPI::get_forms( $active, $trash );

		if ( ! empty( $forms ) && ! empty( $order_by ) ) {
			$forms = wp_list_sort( $forms, $order_by, $order );
		}

		return $forms;
	}

	/**
	 * Get form fields with optional properties
	 *
	 * @param array|int $form Form array or form ID
	 * @param bool      $add_default_properties Whether to add default properties to fields
	 * @param bool      $include_parent_field Whether to include parent fields
	 *
	 * @return array Array of form fields
	 */
	public static function get_form_fields( $form = '', $add_default_properties = false, $include_parent_field = true ) {
		if ( empty( $form ) ) {
			return array();
		}

		// If form ID is passed, get the form
		if ( is_numeric( $form ) ) {
			$form = self::get_form( $form );
		}

		if ( empty( $form ) || ! is_array( $form ) ) {
			return array();
		}

		$fields = isset( $form['fields'] ) ? $form['fields'] : array();

		if ( ! $include_parent_field ) {
			$fields = self::remove_parent_fields( $fields );
		}

		if ( $add_default_properties ) {
			$fields = self::add_default_properties( $fields );
		}

		return $fields;
	}

	/**
	 * Remove parent fields from field array
	 *
	 * @param array $fields Array of fields
	 *
	 * @return array Filtered array of fields
	 */
	private static function remove_parent_fields( $fields ) {
		$filtered_fields = array();

		foreach ( $fields as $field ) {
			if ( ! isset( $field['parentId'] ) || empty( $field['parentId'] ) ) {
				$filtered_fields[] = $field;
			}
		}

		return $filtered_fields;
	}

	/**
	 * Add default properties to fields
	 *
	 * @param array $fields Array of fields
	 *
	 * @return array Fields with default properties
	 */
	private static function add_default_properties( $fields ) {
		foreach ( $fields as &$field ) {
			$field['default_value'] = isset( $field['default_value'] ) ? $field['default_value'] : '';
			$field['placeholder']   = isset( $field['placeholder'] ) ? $field['placeholder'] : '';
			$field['description']   = isset( $field['description'] ) ? $field['description'] : '';
		}

		return $fields;
	}

	/**
	 * Get entries from Gravity Forms
	 *
	 * @param int|array $form_ids Form ID or array of form IDs
	 * @param array     $passed_criteria Search criteria
	 * @param int       $total Reference to total count
	 *
	 * @return array Array of entries
	 */
	public static function get_entries( $form_ids = null, $passed_criteria = null, &$total = null ) {
		if ( ! class_exists( 'GFAPI' ) ) {
			return array();
		}

		$criteria = self::calculate_get_entries_criteria( $passed_criteria, $form_ids );

		$entries = \GFAPI::get_entries( $form_ids, $criteria, null, null, $total );

		return is_array( $entries ) ? $entries : array();
	}

	/**
	 * Calculate search criteria for get_entries
	 *
	 * @param array $passed_criteria Passed criteria
	 * @param array $form_ids Form IDs
	 *
	 * @return array Calculated criteria
	 */
	private static function calculate_get_entries_criteria( $passed_criteria = array(), $form_ids = array() ) {
		$criteria = array();

		if ( ! empty( $passed_criteria ) ) {
			$criteria = $passed_criteria;
		}

		// Add default sorting if not specified
		if ( empty( $criteria['sorting'] ) ) {
			$criteria['sorting'] = array(
				'key'       => 'date_created',
				'direction' => 'DESC',
			);
		}

		return $criteria;
	}

	/**
	 * Get a single entry
	 *
	 * @param int|string $entry_slug Entry ID or slug
	 * @param bool       $force_allow_ids Whether to force allow IDs
	 *
	 * @return array|false Entry array or false if not found
	 */
	public static function get_entry( $entry_slug, $force_allow_ids = false ) {
		if ( empty( $entry_slug ) ) {
			return false;
		}

		if ( ! class_exists( 'GFAPI' ) ) {
			return false;
		}

		$entry_id = self::get_entry_id( $entry_slug, $force_allow_ids );

		if ( empty( $entry_id ) ) {
			return false;
		}

		$entry = \GFAPI::get_entry( $entry_id );

		return $entry;
	}

	/**
	 * Get entry ID from slug or ID
	 *
	 * @param int|string $entry_id_or_slug Entry ID or slug
	 * @param bool       $force_allow_ids Whether to force allow IDs
	 *
	 * @return int|false Entry ID or false if not found
	 */
	public static function get_entry_id( $entry_id_or_slug = '', $force_allow_ids = false ) {
		if ( empty( $entry_id_or_slug ) ) {
			return false;
		}

		// If it's already a numeric ID and we allow IDs
		if ( is_numeric( $entry_id_or_slug ) && $force_allow_ids ) {
			return intval( $entry_id_or_slug );
		}

		// Try to get entry by slug
		$entry = \GFAPI::get_entry( $entry_id_or_slug );

		if ( $entry && ! is_wp_error( $entry ) ) {
			return $entry['id'];
		}

		return false;
	}

	/**
	 * Get form from entry ID
	 *
	 * @param int|string $entry_slug Entry slug
	 *
	 * @return array|false Form array or false if not found
	 */
	public static function get_form_from_entry_id( $entry_slug ) {
		$entry = self::get_entry( $entry_slug, true, false );

		if ( ! $entry ) {
			return false;
		}

		return self::get_form( $entry['form_id'] );
	}

	/**
	 * Get field from form
	 *
	 * @param array $form Form array
	 * @param mixed $field_id Field ID
	 *
	 * @return array|false Field array or false if not found
	 */
	public static function get_field( $form, $field_id ) {
		if ( empty( $form ) || empty( $field_id ) ) {
			return false;
		}

		$fields = self::get_form_fields( $form );

		foreach ( $fields as $field ) {
			if ( $field['id'] == $field_id ) {
				return $field;
			}
		}

		return false;
	}

	/**
	 * Get field type
	 *
	 * @param array $form Form array
	 * @param mixed $field_id Field ID
	 *
	 * @return string Field type
	 */
	public static function get_field_type( $form = null, $field_id = '' ) {
		$field = self::get_field( $form, $field_id );

		if ( ! $field ) {
			return '';
		}

		return isset( $field['type'] ) ? $field['type'] : '';
	}

	/**
	 * Get field label
	 *
	 * @param array  $form Form array
	 * @param mixed  $field_id Field ID
	 * @param string $field_value Field value
	 *
	 * @return string Field label
	 */
	public static function get_field_label( $form = array(), $field_id = '', $field_value = '' ) {
		$field = self::get_field( $form, $field_id );

		if ( ! $field ) {
			return '';
		}

		$label = isset( $field['label'] ) ? $field['label'] : '';

		// Handle choice-based fields
		if ( ! empty( $field_value ) && isset( $field['choices'] ) && is_array( $field['choices'] ) ) {
			foreach ( $field['choices'] as $choice ) {
				if ( isset( $choice['value'] ) && $choice['value'] === $field_value ) {
					$label = isset( $choice['text'] ) ? $choice['text'] : $label;
					break;
				}
			}
		}

		return $label;
	}

	/**
	 * Check if value is empty
	 *
	 * @param mixed   $value Value to check
	 * @param boolean $zero_is_empty Should zero be treated as empty
	 * @param boolean $allow_string_booleans Whether to check string booleans
	 *
	 * @return boolean True if empty, false if not
	 */
	public static function is_empty( $value, $zero_is_empty = true, $allow_string_booleans = true ) {
		if ( is_array( $value ) ) {
			foreach ( $value as $v ) {
				if ( ! self::is_empty( $v, $zero_is_empty, $allow_string_booleans ) ) {
					return false;
				}
			}
			return true;
		}

		if (
			! isset( $value ) ||
			false === $value ||
			null === $value ||
			'' === $value ||
			array() === $value ||
			( is_object( $value ) && ! get_object_vars( $value ) )
		) {
			return true;
		}

		if ( is_string( $value ) && $allow_string_booleans ) {
			$value = trim( $value );
			$value = strtolower( $value );

			if ( in_array( $value, array( 'yes', 'true' ), true ) ) {
				$value = true;
			} elseif ( in_array( $value, array( 'no', 'false' ), true ) ) {
				$value = false;
			}
		}

		if ( ! $zero_is_empty && is_numeric( $value ) && empty( $value ) ) {
			return false;
		}

		return empty( $value );
	}

	/**
	 * Check if value is not empty
	 *
	 * @param mixed   $value Value to check
	 * @param boolean $zero_is_empty Should zero be treated as empty
	 * @param boolean $allow_string_booleans Whether to check string booleans
	 *
	 * @return boolean True if not empty, false if empty
	 */
	public static function is_not_empty( $value, $zero_is_empty = false, $allow_string_booleans = false ) {
		return ! self::is_empty( $value, $zero_is_empty, $allow_string_booleans );
	}

	/**
	 * Format date
	 *
	 * @param string $date_string Date string
	 * @param array  $args Format arguments
	 *
	 * @return string Formatted date
	 */
	public static function format_date( $date_string = '', $args = array() ) {
		if ( empty( $date_string ) ) {
			return '';
		}

		$defaults = array(
			'format' => get_option( 'date_format' ),
			'timezone' => get_option( 'timezone_string' ),
		);

		$args = wp_parse_args( $args, $defaults );

		$timestamp = strtotime( $date_string );

		if ( false === $timestamp ) {
			return $date_string;
		}

		if ( ! empty( $args['timezone'] ) ) {
			$date = new DateTime( '@' . $timestamp );
			$date->setTimezone( new DateTimeZone( $args['timezone'] ) );
			$timestamp = $date->getTimestamp();
		}

		return date_i18n( $args['format'], $timestamp );
	}

	/**
	 * Generate HTML link
	 *
	 * @param string $href Link URL
	 * @param string $anchor_text Link text
	 * @param array  $atts Link attributes
	 *
	 * @return string HTML link
	 */
	public static function get_link_html( $href = '', $anchor_text = '', $atts = array() ) {
		if ( empty( $href ) ) {
			return '';
		}

		$defaults = array(
			'target' => '',
			'rel'    => '',
			'class'  => '',
		);

		$atts = wp_parse_args( $atts, $defaults );

		$attributes = '';
		foreach ( $atts as $key => $value ) {
			if ( ! empty( $value ) ) {
				$attributes .= ' ' . esc_attr( $key ) . '="' . esc_attr( $value ) . '"';
			}
		}

		return sprintf(
			'<a href="%s"%s>%s</a>',
			esc_url( $href ),
			$attributes,
			esc_html( $anchor_text )
		);
	}
} 