<?php
/**
 * Helper functions for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Ristretto Apps
 * @link      https://ristrettoapps.com
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_Helper {

	/**
	 * Get the URL for a CSS file
	 *
	 * @param string $css_file Filename of the CSS file
	 * @param string $dir_path Absolute path to the directory where the CSS file is stored
	 *
	 * @return string URL path to the file
	 */
	public static function css_url( $css_file = '', $dir_path = '' ) {
		// If there's an overriding CSS file in the current template folder, use it
		$template_css_path = trailingslashit( get_stylesheet_directory() ) . 'displayformentries/css/' . $css_file;

		if ( file_exists( $template_css_path ) ) {
			$path = trailingslashit( get_stylesheet_directory_uri() ) . 'displayformentries/css/' . $css_file;
		} else {
			// Default: use plugin CSS file
			if ( '' === $dir_path ) {
				$dir_path = DFE_DIR . 'assets/css/';
			}

			$path = plugins_url( $css_file, trailingslashit( $dir_path ) . 'stripped-by-plugin_basename.php' );
		}

		return $path;
	}

	/**
	 * Check whether a variable is not an empty string
	 *
	 * @param mixed $mixed Variable to check
	 *
	 * @return bool true: $mixed is *not* an empty string; false: $mixed *is* an empty string
	 */
	public static function is_not_empty_string( $mixed = '' ) {
		return ( '' !== $mixed );
	}

	/**
	 * Get permalink query args
	 *
	 * @param int|WP_Post $id Optional. Post ID or post object. Default current post.
	 *
	 * @return array URL args, if exists. Empty array if not.
	 */
	public static function get_permalink_query_args( $id = 0 ) {
		$parsed_permalink = parse_url( get_permalink( $id ) );

		$permalink_args = isset( $parsed_permalink['query'] ) ? $parsed_permalink['query'] : false;

		if ( empty( $permalink_args ) ) {
			return array();
		}

		parse_str( $permalink_args, $args );

		return $args;
	}

	/**
	 * Similar to WordPress selected(), checked(), and disabled() functions, but allows arrays
	 *
	 * @param string $value One of the values to compare
	 * @param mixed  $current The other value to compare
	 * @param bool   $echo Whether to echo or just return the string
	 * @param string $type The type of checked|selected|disabled we are doing
	 *
	 * @return string html attribute or empty string
	 */
	public static function selected( $value, $current, $echo = true, $type = 'selected' ) {
		$output = '';
		
		if ( is_array( $current ) ) {
			if ( in_array( $value, $current ) ) {
				$output = __checked_selected_helper( true, true, false, $type );
			}
		} else {
			$output = __checked_selected_helper( $value, $current, false, $type );
		}

		if ( $echo ) {
			echo $output;
		}

		return $output;
	}

	/**
	 * Sanitize HTML class names
	 *
	 * @param string|array $classes Text or array of classes to sanitize
	 *
	 * @return string Sanitized CSS string
	 */
	public static function sanitize_html_class( $classes ) {
		if ( is_string( $classes ) ) {
			$classes = explode( ' ', $classes );
		}

		// If someone passes something not string or array, we get outta here
		if ( ! is_array( $classes ) ) {
			return $classes;
		}

		$classes = array_map( 'sanitize_html_class', $classes );

		return implode( ' ', $classes );
	}

	/**
	 * Strip whitespace from string
	 *
	 * @param string $string String to strip whitespace from
	 *
	 * @return string String with whitespace stripped
	 */
	public static function strip_whitespace( $string ) {
		if ( ! is_string( $string ) ) {
			return $string;
		}

		return preg_replace( '/\s+/', ' ', trim( $string ) );
	}

	/**
	 * Get the contents of a file using include() and ob_start()
	 *
	 * @param string $file_path Full path to a file
	 * @param mixed  $object Pass pseudo-global to the included file
	 *
	 * @return string Included file contents
	 */
	public static function ob_include( $file_path, $object = null ) {
		if ( ! file_exists( $file_path ) ) {
			return '';
		}

		ob_start();
		include $file_path;
		return ob_get_clean();
	}

	/**
	 * Format number
	 *
	 * @param mixed  $number Number to format
	 * @param string $decimals Number of decimal places
	 * @param bool   $separator Whether to use thousand separator
	 *
	 * @return string Formatted number
	 */
	public static function number_format( $number, $decimals = '', $separator = true ) {
		if ( ! is_numeric( $number ) ) {
			return $number;
		}

		$decimals = ( '' === $decimals ) ? 0 : intval( $decimals );

		if ( $separator ) {
			return number_format( $number, $decimals );
		}

		return number_format( $number, $decimals, '.', '' );
	}

	/**
	 * Format link
	 *
	 * @param string $value Link value
	 *
	 * @return string Formatted link
	 */
	public static function format_link( $value = null ) {
		if ( empty( $value ) ) {
			return '';
		}

		// If it's already a link, return as is
		if ( preg_match( '/^https?:\/\//', $value ) ) {
			return $value;
		}

		// Add http:// if no protocol
		return 'http://' . $value;
	}

	/**
	 * Check if datetime is valid
	 *
	 * @param string $datetime Datetime string
	 * @param string $expected_format Expected format
	 *
	 * @return bool True if valid, false if not
	 */
	public static function is_valid_datetime( $datetime, $expected_format = 'Y-m-d' ) {
		if ( empty( $datetime ) ) {
			return false;
		}

		$date = DateTime::createFromFormat( $expected_format, $datetime );

		return $date && $date->format( $expected_format ) === $datetime;
	}

	/**
	 * Get input ID from field ID
	 *
	 * @param string $field_id Field ID
	 *
	 * @return string Input ID
	 */
	public static function get_input_id_from_id( $field_id = '' ) {
		if ( empty( $field_id ) ) {
			return '';
		}

		// Handle input IDs (e.g., 1.3 -> 1_3)
		if ( strpos( $field_id, '.' ) !== false ) {
			return str_replace( '.', '_', $field_id );
		}

		return $field_id;
	}

	/**
	 * Get terms choices for select fields
	 *
	 * @param array $args Arguments for get_terms
	 *
	 * @return array Array of term choices
	 */
	public static function get_terms_choices( $args = array() ) {
		$defaults = array(
			'taxonomy'   => 'category',
			'hide_empty' => false,
		);

		$args = wp_parse_args( $args, $defaults );

		$terms = get_terms( $args );

		if ( is_wp_error( $terms ) || empty( $terms ) ) {
			return array();
		}

		$choices = array();
		foreach ( $terms as $term ) {
			$choices[] = array(
				'text'  => $term->name,
				'value' => $term->term_id,
			);
		}

		return $choices;
	}

	/**
	 * Maybe convert date string to timestamp
	 *
	 * @param string $value Date string
	 *
	 * @return int|string Timestamp or original value
	 */
	public static function maybe_convert_date_string_to_timestamp( $value = '' ) {
		if ( empty( $value ) ) {
			return $value;
		}

		// If it's already a timestamp, return as is
		if ( is_numeric( $value ) ) {
			return $value;
		}

		$timestamp = strtotime( $value );

		return $timestamp ? $timestamp : $value;
	}

	/**
	 * Maybe JSON decode
	 *
	 * @param mixed $value Value to decode
	 * @param bool  $assoc Whether to return associative array
	 * @param int   $depth Maximum depth
	 * @param int   $options JSON decode options
	 *
	 * @return mixed Decoded value or original value
	 */
	public static function maybe_json_decode( $value, $assoc = false, $depth = 512, $options = 0 ) {
		if ( empty( $value ) ) {
			return $value;
		}

		if ( is_array( $value ) || is_object( $value ) ) {
			return $value;
		}

		if ( ! is_string( $value ) ) {
			return $value;
		}

		$decoded = json_decode( $value, $assoc, $depth, $options );

		return ( json_last_error() === JSON_ERROR_NONE ) ? $decoded : $value;
	}

	/**
	 * Map deep
	 *
	 * @param mixed    $value Value to map
	 * @param callable $callback Callback function
	 *
	 * @return mixed Mapped value
	 */
	public static function map_deep( $value, $callback ) {
		if ( is_array( $value ) ) {
			foreach ( $value as $index => $item ) {
				$value[ $index ] = self::map_deep( $item, $callback );
			}
		} elseif ( is_object( $value ) ) {
			$object_vars = get_object_vars( $value );
			foreach ( $object_vars as $property_name => $property_value ) {
				$value->$property_name = self::map_deep( $property_value, $callback );
			}
		} else {
			$value = call_user_func( $callback, $value );
		}

		return $value;
	}

	/**
	 * Get query arguments from GET request
	 *
	 * @return array Query arguments
	 */
	public static function get_query_args() {
		$passed_get = isset( $_GET ) ? $_GET : array();

		$passed_get = stripslashes_deep( $passed_get );
		$passed_get = urldecode_deep( $passed_get );

		if ( empty( $passed_get ) ) {
			return array();
		}

		$query_args = $passed_get;

		$reserved_args = array(
			'entry',
			'gvid',
			'status',
			'action',
			'view_id',
			'entry_id',
			'pagenum',
		);

		/**
		 * Modify the URL arguments that should not be used because they are internal.
		 *
		 * @param array $reserved_args Array of URL query keys that should not be used except internally.
		 */
		$reserved_args = apply_filters( 'dfe/reserved_query_args', $reserved_args );

		foreach ( $reserved_args as $reserved_arg ) {
			unset( $query_args[ $reserved_arg ] );
		}

		return $query_args;
	}

	/**
	 * Get current view ID
	 *
	 * @return int Current view ID
	 */
	public static function get_current_view_id() {
		global $post;

		if ( ! $post ) {
			return 0;
		}

		// Check if current post has DFE shortcode
		if ( has_shortcode( $post->post_content, 'dfe_entries' ) ) {
			return $post->ID;
		}

		return 0;
	}

	/**
	 * Get current entry ID from URL
	 *
	 * @return int|false Entry ID or false if not found
	 */
	public static function get_current_entry_id() {
		$entry_id = isset( $_GET['entry_id'] ) ? intval( $_GET['entry_id'] ) : 0;

		return $entry_id > 0 ? $entry_id : false;
	}

	/**
	 * Get current form ID from URL
	 *
	 * @return int|false Form ID or false if not found
	 */
	public static function get_current_form_id() {
		$form_id = isset( $_GET['form_id'] ) ? intval( $_GET['form_id'] ) : 0;

		return $form_id > 0 ? $form_id : false;
	}

	/**
	 * Check if current page is a single entry view
	 *
	 * @return bool True if single entry view, false if not
	 */
	public static function is_single_entry() {
		return self::get_current_entry_id() !== false;
	}

	/**
	 * Get back link
	 *
	 * @return string Back link URL
	 */
	public static function get_back_link() {
		$referer = wp_get_referer();

		if ( $referer ) {
			return $referer;
		}

		// Fallback to current page without entry parameters
		$current_url = remove_query_arg( array( 'entry_id', 'form_id' ) );

		return $current_url;
	}
} 