<?php

namespace WeDevs\Wpuf\Pro\Fields;

/**
 * LinkedIn URL Field Class
 *
 * @since 4.2.0
 */
class Field_Linkedin extends Field_Social {


    public function __construct() {
        $this->name                 = __( 'LinkedIn', 'wpuf-pro' );
        $this->input_type           = 'linkedin_url';
        $this->icon                 = 'linkedin';
        $this->base_url             = 'https://linkedin.com/in/';
        $this->username_pattern     = '^[a-zA-Z0-9\-]{3,100}$';
        $this->url_pattern          = '/^(?:https?:\/\/)?(?:www\.)?(?:linkedin\.com)\/(?:in|company)\/([A-Za-z0-9\-]{3,100})\/?$/i';
        $this->max_username_length  = 100;
        $this->example_username     = 'john-doe';
    }

    /**
     * Get the field props
     *
     * @return array
     */
    public function get_field_props() {
        $defaults = $this->default_attributes();

        $props = [
            'input_type'       => 'linkedin_url',
            'required'         => 'no',
            'name'             => 'wpuf_social_linkedin',
            'meta_key'         => 'wpuf_social_linkedin',
            'help'             => __( 'Enter your LinkedIn username or full URL', 'wpuf-pro' ),
            'css'              => '',
            'placeholder'      => __( 'username', 'wpuf-pro' ),
            'default'          => '',
            'size'             => 40,
            'width'            => 'large',
            'id'               => 0,
            'is_new'           => true,
            'show_icon'        => 'no',
            'readonly'         => 'no',
            'open_in_new_window' => 'yes',
            'nofollow'         => 'yes',
            'username_validation' => 'strict',
            'wpuf_cond'        => '',
        ];

        return array_merge( $defaults, $props );
    }

    /**
     * Render the social media field
     *
     * @since 4.1.7
     *
     * @param array $field_settings
     * @param int $form_id
     * @param string $type
     * @param int $post_id
     *
     * @return void
     */
    public function render( $field_settings, $form_id, $type = 'post', $post_id = null ) {
        // Enqueue LinkedIn validation script
        wp_enqueue_script( 'wpuf-frontend-linkedin-validation' );

        if ( isset( $post_id ) && $post_id != 0 ) {
            $value = $this->get_user_data( $post_id, $field_settings['name'] );
        } else {
            $value = $field_settings['default'];
        }

        $show_icon = isset( $field_settings['show_icon'] ) ? $field_settings['show_icon'] : 'no';
        $readonly = isset( $field_settings['readonly'] ) ? $field_settings['readonly'] : 'no';
        $css_class = isset( $field_settings['css'] ) ? $field_settings['css'] : '';
        $width_class = isset( $field_settings['width'] ) ? 'wpuf-' . $field_settings['width'] : 'wpuf-medium';
        ?>

        <li <?php $this->print_list_attributes( $field_settings ); ?>>
            <div class="wpuf-label-with-icon">
                <?php
                $this->print_label( $field_settings, $form_id );
                // Show icon next to label if enabled
                if ( $show_icon === 'yes' && ! empty( $this->icon_svg ) ) : ?>
                    <span class="wpuf-social-label-icon">
                        <?php
                        echo wp_kses( $this->icon_svg, $this->get_allowed_svg_attributes() );
                        ?>
                    </span>
                <?php endif; ?>
            </div>

            <div class="wpuf-fields">
                <div class="wpuf-social-input-wrapper <?php echo esc_attr( $width_class ); ?>">
                    <?php
                    $input_type = 'text';
                    ?>
                    <input
                        class="textfield <?php echo esc_attr( $input_type ); ?> <?php echo 'wpuf_' . $field_settings['name'] . '_' . $form_id; ?> <?php echo esc_attr( $css_class ); ?>"
                        id="<?php echo $field_settings['name'] . '_' . $form_id; ?>"
                        type="<?php echo esc_attr( $input_type ); ?>"
                        data-required="<?php echo $field_settings['required'] ?>"
                        data-type="<?php echo esc_attr( $input_type ); ?>"
                        data-username-validation="<?php echo isset( $field_settings['username_validation'] ) ? esc_attr( $field_settings['username_validation'] ) : 'strict'; ?>"
                        name="<?php echo esc_attr( $field_settings['name'] ); ?>"
                        placeholder="<?php echo esc_attr( $field_settings['placeholder'] ); ?>"
                        value="<?php echo esc_attr( $value ); ?>"
                        size="<?php echo esc_attr( isset( $field_settings['size'] ) ? $field_settings['size'] : 40 ); ?>"
                        <?php if ( isset( $this->username_pattern ) && ! empty( $this->username_pattern ) ): ?>
                        pattern="<?php echo esc_attr( $this->username_pattern ); ?>"
                        <?php endif; ?>
                        title="<?php echo esc_attr( sprintf( __( 'Enter a valid %s username or URL', 'wpuf-pro' ), $this->name ) ); ?>"
                        <?php echo $readonly === 'yes' ? 'readonly' : ''; ?>
                    />
                </div>

                <?php $this->help_text( $field_settings ); ?>
            </div>

            <?php if ( isset( $field_settings['word_restriction'] ) && $field_settings['word_restriction'] ) : ?>
                <div class="wpuf-wordlimit-message wpuf-help"></div>
            <?php endif; ?>
        </li>

        <?php $this->conditional_logic( $field_settings, $form_id ); ?>

        <?php
    }

    /**
     * Get options settings for the form builder
     *
     * @return array
     */
    public function get_options_settings() {
        // Get default options but exclude fields that would duplicate parent class settings
        $default_options = $this->get_default_option_settings( false, array('dynamic') );

        $settings = [
            [
                'name'      => 'meta_key',
                'title'     => __( 'Meta Key', 'wpuf-pro' ),
                'type'      => 'text',
                'default'   => 'wpuf_social_linkedin',
                'section'   => 'basic',
                'priority'  => 15,
                'readonly'  => true,
                'help_text' => __( 'The meta key used to store this field data in the database', 'wpuf-pro' ),
            ],
            [
                'name'      => 'placeholder',
                'title'     => __( 'Placeholder', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'basic',
                'priority'  => 15.5,
                'help_text' => __( 'Placeholder text for the input field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'default',
                'title'     => __( 'Default value', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 15.7,
                'help_text' => __( 'Enter a default value for this field', 'wpuf-pro' ),
            ],
            [
                'name'      => 'username_validation',
                'title'     => __( 'Username Validation', 'wpuf-pro' ),
                'type'      => 'radio',
                'options'   => [
                    'strict' => __( 'Strict (LinkedIn username format)', 'wpuf-pro' ),
                    'allow_full_url' => __( 'Allow full URLs', 'wpuf-pro' ),
                ],
                'default'   => 'strict',
                'section'   => 'basic',
                'priority'  => 16,
                'inline'    => true,
                'help_text' => __( 'Choose validation type: strict username format or allow full URLs', 'wpuf-pro' ),
            ],
            [
                'name'      => 'readonly',
                'title'     => __( 'Read Only', 'wpuf-pro' ),
                'type'      => 'checkbox',
                'is_single_opt' => true,
                'options'   => [
                    'yes' => __( 'Yes', 'wpuf-pro' ),
                ],
                'section'   => 'basic',
                'priority'  => 18,
                'help_text' => __( 'Make this field read-only', 'wpuf-pro' ),
            ],
            [
                'name'      => 'size',
                'title'     => __( 'Size', 'wpuf-pro' ),
                'type'      => 'text',
                'section'   => 'advanced',
                'priority'  => 20,
                'help_text' => __( 'Size of the input field', 'wpuf-pro' ),
            ],
        ];

        return array_merge( $default_options, $settings );
    }
}