<?php

/**
 *  MailChimp Class
 *
 *  Handle API request and connect with mailchimp
 */
class MailChimp {

    private $api_key;
    private $api_endpoint = 'https://<dc>.api.mailchimp.com/3.0';
    private $verify_ssl   = false;

    /**
     * Create a new instance
     * @param string $api_key Your MailChimp API key
     */
    public function __construct( $api_key ) {
        $this->api_key = $api_key;
        list(, $datacentre) = explode( '-', $this->api_key );
        $this->api_endpoint = str_replace( '<dc>', $datacentre, $this->api_endpoint );
    }

    /**
     * Call an API method. Every request needs the API key, so that is added automatically -- you don't need to pass it in.
     * @param  string $method The API method to call, e.g. 'lists/list'
     * @param  array  $args   An array of arguments to pass to the method. Will be json-encoded for you.
     * @return array          Associative array of json decoded API response.
     */
    public function call( $method, $args = array() ) {
        return $this->makeRequest( $method, $args );
    }

    /**
     * Performs the underlying HTTP request. Not very exciting
     * @param  string $method The API method to be called
     * @param  array  $args   Assoc array of parameters to be passed
     * @return array          Assoc array of decoded result
     */
    private function makeRequest( $method, $args = array() ) {
        $json_data = json_encode( $args, true );

        if ( count( $args ) && 'lists' === $method ) {
            $url = $this->api_endpoint . '/' . $method . '?' . http_build_query( $args );
        } else {
            $url = $this->api_endpoint . '/' . $method;
        }

        if ( 'lists' !== $method ) {
            $member_id = md5( strtolower( $args['email_address'] ) );
            $url .= '/members/';
        }

        if ( function_exists( 'curl_init' ) && function_exists( 'curl_setopt' ) ) {
            $ch = curl_init();
            curl_setopt( $ch, CURLOPT_CUSTOMREQUEST, 'lists' !== $method ? 'POST' : 'GET' );
            curl_setopt( $ch, CURLOPT_URL, $url );
            curl_setopt( $ch, CURLOPT_USERPWD, 'user:' . $this->api_key );
            curl_setopt( $ch, CURLOPT_HTTPHEADER, array( 'Content-Type: application/json' ) );
            curl_setopt( $ch, CURLOPT_USERAGENT, 'PHP-MCAPI/3.0' );
            curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
            curl_setopt( $ch, CURLOPT_TIMEOUT, 10 );
            curl_setopt( $ch, CURLOPT_POST, true );
            curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, $this->verify_ssl );
            curl_setopt( $ch, CURLOPT_POSTFIELDS, $json_data );
            $result = curl_exec( $ch );
            curl_close( $ch );
            //            WP_User_Frontend::log(print_r($result,true));
        } else {
            $result = file_get_contents(
                $url, null, stream_context_create(
                    array(
                        'http' => array(
                            'protocol_version' => 1.1,
                            'user_agent'       => 'PHP-MCAPI/3.0',
                            'method'           => 'POST',
                            'header'           => "Content-type: application/json\r\n" .
                                                  "Connection: close\r\n" .
                                                  'Content-length: ' . strlen( $json_data ) . "\r\n",
                            'content'          => $json_data,
                        ),
                    )
                )
            );
        }

        return $result ? json_decode( $result, true ) : false;
    }
}
