import { __ } from '@wordpress/i18n';

const UserComments = ({ user, attributes, context }) => {
    const {
        numberOfComments = 5,
        commentStatus = ['approve'],
        excludePosts = [],
        order = 'DESC',
        showPostTitles = true,
        showCommentDate = true,
        commentLength = 100,
        showAvatar = true,
        avatarSize = 32,
        linkToComments = true,
        dateFormat = 'relative',
        customDateFormat = 'F j, Y',
        emptyMessage = '',
        showLabel = true,
        customLabel = 'Recent Comments',
        layoutStyle = 'list'
    } = attributes || {};

    // Mock comments data for display
    const getMockComments = () => {
        if (!user) {
            return [
                {
                    id: 1,
                    content: __('This is a sample comment that shows how user comments will be displayed.', 'wpuf-pro'),
                    post_title: __('Sample Post', 'wpuf-pro'),
                    date: '2024-01-12',
                    avatar: 'https://secure.gravatar.com/avatar/example?s=32',
                    link: '#'
                },
                {
                    id: 2,
                    content: __('Another sample comment to demonstrate the layout and functionality.', 'wpuf-pro'),
                    post_title: __('Another Post', 'wpuf-pro'),
                    date: '2024-01-10',
                    avatar: 'https://secure.gravatar.com/avatar/example?s=32',
                    link: '#'
                }
            ];
        }

        // In production, this would fetch real comments from the API
        return [
            {
                id: 1,
                content: __('This is a user comment that shows how comments will be displayed.', 'wpuf-pro'),
                post_title: __('User Post', 'wpuf-pro'),
                date: '2024-01-12',
                avatar: 'https://secure.gravatar.com/avatar/example?s=32',
                link: '#'
            }
        ];
    };

    const comments = getMockComments();

    // Format date
    const formatDate = (dateString) => {
        if (dateFormat === 'relative') {
            const date = new Date(dateString);
            const now = new Date();
            const diffTime = Math.abs(now - date);
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
            
            if (diffDays === 1) return __('1 day ago', 'wpuf-pro');
            if (diffDays < 7) return `${diffDays} ${__('days ago', 'wpuf-pro')}`;
            if (diffDays < 30) return `${Math.floor(diffDays / 7)} ${__('weeks ago', 'wpuf-pro')}`;
            return date.toLocaleDateString('en-US', { 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric' 
            });
        }
        
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'long', 
            day: 'numeric' 
        });
    };

    // Render comment item
    const renderCommentItem = (comment) => {
        const displayContent = comment.content.length > commentLength 
            ? comment.content.substring(0, commentLength) + '...' 
            : comment.content;

        return (
            <div key={comment.id} className="wpuf-comment-item">
                {showAvatar && (
                    <div className="wpuf-comment-avatar">
                        <img 
                            src={comment.avatar} 
                            alt=""
                            width={avatarSize}
                            height={avatarSize}
                        />
                    </div>
                )}
                <div className="wpuf-comment-content">
                    <p className="wpuf-comment-text">{displayContent}</p>
                    <div className="wpuf-comment-meta">
                        {showPostTitles && (
                            <span className="wpuf-comment-post">
                                {__('on', 'wpuf-pro')} 
                                {linkToComments ? (
                                    <a href={comment.link}>{comment.post_title}</a>
                                ) : (
                                    comment.post_title
                                )}
                            </span>
                        )}
                        {showCommentDate && (
                            <span className="wpuf-comment-date">
                                {formatDate(comment.date)}
                            </span>
                        )}
                    </div>
                </div>
            </div>
        );
    };

    return (
        <div className={`wpuf-user-comments wpuf-layout-${layoutStyle}`}>
            {showLabel && customLabel && (
                <h3 className="wpuf-comments-label">{customLabel}</h3>
            )}

            <div className="wpuf-comments-container">
                {comments.length === 0 ? (
                    <div className="wpuf-comments-empty">
                        <p>{emptyMessage || __('No comments found.', 'wpuf-pro')}</p>
                    </div>
                ) : (
                    comments.slice(0, numberOfComments).map(comment => renderCommentItem(comment))
                )}
            </div>
        </div>
    );
};

export default UserComments; 