import { __ } from '@wordpress/i18n';

const UserContactInfo = ({ user, attributes, context }) => {
    const {
        showFields = ['display_name', 'user_email', 'user_url'],
        emailFormat = 'mailto_link',
        phoneFormat = 'tel_link',
        showWebsite = true,
        websiteFormat = 'link',
        addressFormat = 'block',
        customContactFields = [],
        showFieldLabels = true,
        visibilityRule = 'public',
        layoutStyle = 'list',
        showIcon = true
    } = attributes || {};

    // Icon components for contact fields
    const ContactIcons = {
        display_name: (
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth="1.5"
                 stroke="currentColor" className="wpuf-size-6">
                <path strokeLinecap="round" strokeLinejoin="round"
                      d="M15.75 6a3.75 3.75 0 1 1-7.5 0 3.75 3.75 0 0 1 7.5 0ZM4.501 20.118a7.5 7.5 0 0 1 14.998 0A17.933 17.933 0 0 1 12 21.75c-2.676 0-5.216-.584-7.499-1.632Z"/>
            </svg>
        ),
        first_name: (
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth="1.5"
                 stroke="currentColor" className="wpuf-size-6">
                <path strokeLinecap="round" strokeLinejoin="round"
                      d="M15.75 6a3.75 3.75 0 1 1-7.5 0 3.75 3.75 0 0 1 7.5 0ZM4.501 20.118a7.5 7.5 0 0 1 14.998 0A17.933 17.933 0 0 1 12 21.75c-2.676 0-5.216-.584-7.499-1.632Z" />
            </svg>
        ),
        last_name: (
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor" className="wpuf-size-6">
                <path strokeLinecap="round" strokeLinejoin="round" d="M15.75 6a3.75 3.75 0 1 1-7.5 0 3.75 3.75 0 0 1 7.5 0ZM4.501 20.118a7.5 7.5 0 0 1 14.998 0A17.933 17.933 0 0 1 12 21.75c-2.676 0-5.216-.584-7.499-1.632Z" />
            </svg>
        ),
        username: (
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor" className="wpuf-size-6">
                <path strokeLinecap="round" strokeLinejoin="round" d="M15.75 6a3.75 3.75 0 1 1-7.5 0 3.75 3.75 0 0 1 7.5 0ZM4.501 20.118a7.5 7.5 0 0 1 14.998 0A17.933 17.933 0 0 1 12 21.75c-2.676 0-5.216-.584-7.499-1.632Z" />
            </svg>
        ),
        nickname: (
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor" className="wpuf-size-6">
                <path strokeLinecap="round" strokeLinejoin="round" d="M15.75 6a3.75 3.75 0 1 1-7.5 0 3.75 3.75 0 0 1 7.5 0ZM4.501 20.118a7.5 7.5 0 0 1 14.998 0A17.933 17.933 0 0 1 12 21.75c-2.676 0-5.216-.584-7.499-1.632Z" />
            </svg>
        ),
        user_email: (
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor" className="wpuf-size-6">
                <path strokeLinecap="round" strokeLinejoin="round" d="M21.75 6.75v10.5a2.25 2.25 0 0 1-2.25 2.25h-15a2.25 2.25 0 0 1-2.25-2.25V6.75m19.5 0A2.25 2.25 0 0 0 19.5 4.5h-15a2.25 2.25 0 0 0-2.25 2.25m19.5 0v.243a2.25 2.25 0 0 1-1.07 1.916l-7.5 4.615a2.25 2.25 0 0 1-2.36 0L3.32 8.91a2.25 2.25 0 0 1-1.07-1.916V6.75" />
            </svg>
        ),
        user_url: (
            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor" className="wpuf-size-6">
                <path strokeLinecap="round" strokeLinejoin="round" d="M12 21a9.004 9.004 0 0 0 8.716-6.747M12 21a9.004 9.004 0 0 1-8.716-6.747M12 21c2.485 0 4.5-4.03 4.5-9S14.485 3 12 3m0 18c-2.485 0-4.5-4.03-4.5-9S9.515 3 12 3m0 0a8.997 8.997 0 0 1 7.843 4.582M12 3a8.997 8.997 0 0 0-7.843 4.582m15.686 0A11.953 11.953 0 0 1 12 10.5c-2.998 0-5.74-1.1-7.843-2.918m15.686 0A8.959 8.959 0 0 1 21 12c0 .778-.099 1.533-.284 2.253m0 0A17.919 17.919 0 0 1 12 16.5c-3.162 0-6.133-.815-8.716-2.247m0 0A9.015 9.015 0 0 1 3 12c0-1.605.42-3.113 1.157-4.418" />
            </svg>
        )
    };

    // Available contact field options
    const CONTACT_FIELD_OPTIONS = {
        display_name: {
            label: __('Display Name', 'wpuf-pro'),
            key: 'display_name',
            icon: ContactIcons.display_name,
            default: __('Display Name', 'wpuf-pro')
        },
        first_name: {
            label: __('First Name', 'wpuf-pro'),
            key: 'first_name',
            icon: ContactIcons.first_name,
            default: __('First Name', 'wpuf-pro')
        },
        last_name: {
            label: __('Last Name', 'wpuf-pro'),
            key: 'last_name',
            icon: ContactIcons.last_name,
            default: __('Last Name', 'wpuf-pro')
        },
        username: {
            label: __('Username', 'wpuf-pro'),
            key: 'username',
            icon: ContactIcons.username,
            default: __('Username', 'wpuf-pro')
        },
        nickname: {
            label: __('Nickname', 'wpuf-pro'),
            key: 'nickname',
            icon: ContactIcons.nickname,
            default: __('Nickname', 'wpuf-pro')
        },
        user_email: {
            label: __('Email', 'wpuf-pro'),
            key: 'email',
            icon: ContactIcons.user_email,
            default: __('Email', 'wpuf-pro')
        },
        user_url: {
            label: __('Website', 'wpuf-pro'),
            key: 'website',
            icon: ContactIcons.user_url,
            default: __('Website', 'wpuf-pro')
        }
    };

    // Mock contact data for display
    const getContactData = () => {
        if (!user) {
            return {
                display_name: 'John Doe',
                first_name: 'John',
                last_name: 'Doe',
                username: 'johndoe',
                nickname: 'Johnny',
                user_email: 'user@example.com',
                user_url: 'https://example.com',
                roles: ['subscriber']
            };
        }

        // Get data from both direct user properties and meta
        const userMeta = user.meta || {};
        
        return {
            display_name: user.display_name || user.name || userMeta.display_name || '',
            first_name: user.first_name || userMeta.first_name || '',
            last_name: user.last_name || userMeta.last_name || '',
            username: user.user_login || user.username || userMeta.user_login || '',
            nickname: user.nickname || userMeta.nickname || '',
            user_email: user.user_email || userMeta.user_email || '',
            user_url: user.user_url || userMeta.user_url || '',
            roles: user.roles || []
        };
    };

    const contactData = getContactData();

    // Format contact field display
    const formatContactField = (type, value) => {
        if (!value) return null;

        switch (type) {
            case 'user_email':
                if (emailFormat === 'mailto_link') {
                    return <a href={`mailto:${value}`}>{value}</a>;
                }
                return value;

            case 'user_url':
                if (websiteFormat === 'link') {
                    return <a href={value} target="_blank" rel="noopener noreferrer">{value}</a>;
                }
                return value;

            default:
                return value;
        }
    };

    // Get field label
    const getFieldLabel = (fieldKey) => {
        return CONTACT_FIELD_OPTIONS[fieldKey]?.label || fieldKey;
    };

    // Get field icon
    const getFieldIcon = (fieldKey) => {
        return ContactIcons[fieldKey] || null;
    };

    // Render contact field
    const renderContactField = (fieldKey) => {
        const value = contactData[fieldKey];
        if (!value) return null;

        const formattedValue = formatContactField(fieldKey, value);
        if (!formattedValue) return null;

        return (
            <div className={`wpuf-contact-field wpuf-flex wpuf-contact-${fieldKey}`} key={fieldKey}>
                {(showIcon && getFieldIcon(fieldKey)) && (
                    <span className="wpuf-contact-icon wpuf-mr-1">
                        {getFieldIcon(fieldKey)}
                    </span>
                )}
                {showFieldLabels && (
                    <span className="wpuf-contact-label">{getFieldLabel(fieldKey)}:</span>
                )}
                <span className="wpuf-contact-value">
                    {formattedValue}
                </span>
            </div>
        );
    };

    return (
        <div className={`wpuf-user-contact-info wpuf-contact-layout-${layoutStyle}`}>
            <div className="wpuf-contact-fields">
                {showFields.length === 0 ? (
                    <div className="wpuf-contact-placeholder">
                        <p>{__('Select contact fields to display in the sidebar.', 'wpuf-pro')}</p>
                    </div>
                ) : (
                    showFields.map(fieldKey => renderContactField(fieldKey))
                )}
            </div>
        </div>
    );
};

export default UserContactInfo; 