import React, { useState, useEffect } from 'react';
import { __ } from '@wordpress/i18n';
import DraggableTabsEditor from '../DraggableTabsEditor';
import AboutTabsSetup from '../AboutTabsSetup';

const ProfileContent = ({ formData, setFormData, isEditing = false }) => {
    // Initialize tab options with saved custom labels if available
    const [tabOptions, setTabOptions] = useState(() => {
        const defaultOptions = wpuf_ud.profile_tabs || {};
        const savedLabels = formData.profile_tabs_labels || {};
        
        // Merge saved custom labels into the default options
        const mergedOptions = { ...defaultOptions };
        Object.keys(savedLabels).forEach(key => {
            if (mergedOptions[key]) {
                if (typeof mergedOptions[key] === 'object') {
                    mergedOptions[key].customLabel = savedLabels[key];
                } else {
                    mergedOptions[key] = {
                        label: mergedOptions[key],
                        customLabel: savedLabels[key]
                    };
                }
            }
        });
        
        return mergedOptions;
    });

    // Handle changes from the DraggableTabsEditor
    const handleTabsChange = (tabs, updatedOptions, allTabsOrder) => {
        // Update the form data with the enabled tabs array
        setFormData(prev => ({ ...prev, profile_tabs: tabs }));
        
        // If custom labels were provided, save them separately
        if (updatedOptions) {
            setTabOptions(updatedOptions);
            const customLabels = Object.keys(updatedOptions).reduce((acc, key) => {
                if (updatedOptions[key]?.customLabel) {
                    acc[key] = updatedOptions[key].customLabel;
                }
                return acc;
            }, {});
            
            // Save both enabled tabs, custom labels, and complete order
            setFormData(prev => ({ 
                ...prev, 
                profile_tabs: tabs,
                profile_tabs_labels: customLabels,
                profile_tabs_order: allTabsOrder || tabs // Complete order of all tabs
            }));
        }
    };

    return (
        <div className="wpuf-w-full wpuf-bg-white wpuf-max-w-3xl wpuf-mx-auto">
            <h2 className="wpuf-text-center"
                style={{
                    
                    fontWeight: 400,
                    fontSize: '20px',
                    lineHeight: '36px',
                    letterSpacing: '0%',
                    textAlign: 'center',
                    color: '#000000'
                }}
            >
                {__('Profile Content', 'wpuf-pro')}
            </h2>
            <p className="wpuf-text-center wpuf-mt-2 wpuf-mb-2"
                style={{
                    
                    fontWeight: 400,
                    fontSize: '14px',
                    lineHeight: '20px',
                    letterSpacing: '0%',
                    textAlign: 'center',
                    color: '#64748B'
                }}
            >{__('Customize user tab layout and content for an organized profile experience', 'wpuf-pro')}</p>
            <h1 className="wpuf-mt-8"
                style={{
                    
                    fontWeight: 400,
                    fontSize: '18px',
                    lineHeight: '28px',
                    letterSpacing: '0%',
                    color: '#000000'
                }}
            >
                {__('Customize Profile Tabs', 'wpuf-pro')}
            </h1>
            <p className="wpuf-mt-2"
                style={{
                    
                    fontWeight: 400,
                    fontSize: '14px',
                    lineHeight: '20px',
                    letterSpacing: '0%',
                    color: '#D1D5DB!important'
                }}
            >{__('Easily manage your profile tabs. Drag to reorder, toggle visibility on the front end, and customize tab names using the edit icon', 'wpuf-pro')}</p>
            <div className="wpuf-mt-4">
                <DraggableTabsEditor
                    options={tabOptions}
                    value={formData.profile_tabs || []}
                    onChange={handleTabsChange}
                    savedOrder={formData.profile_tabs_order || []}
                />
            </div>

            {/* About Tabs Setup Section */}
            <AboutTabsSetup 
                formData={formData}
                setFormData={setFormData}
                isEditing={isEditing}
            />
        </div>
    );
};

export default ProfileContent;