<?php
/**
 * User Directory Block Template - Layout 1
 *
 * Classic table-style layout with organized rows and block editor styling support.
 *
 * @since 4.2.0
 */

// Ensure we have the required data
if (!isset($users) || !is_array($users) || empty($users)) {
    echo '<div class="wpuf-user-directory-no-users">' . esc_html__('No users found.', 'wpuf-pro') . '</div>';
    return;
}

// Get block styling
$styling = wpuf_get_block_styling($attributes, 'user-directory');
$wrapper_attrs = wpuf_get_block_wrapper_attributes($attributes, 'user-directory');

// Build classes
$classes = array_merge(
    $styling['classes'],
    ['wpuf-layout-1', 'wpuf-user-directory-layout']
);

// Add responsive classes
$responsive_classes = wpuf_get_responsive_classes($attributes);
$classes = array_merge($classes, $responsive_classes);

// Add animation classes
$animation_classes = wpuf_get_animation_classes($attributes);
$classes = array_merge($classes, $animation_classes);

// Build inline styles
$inline_styles = wpuf_build_inline_styles($styling['styles']);

// Start the template output
?>
<div <?php echo $wrapper_attrs; ?>>
    <div class="<?php echo esc_attr(implode(' ', $classes)); ?>"
         <?php echo $inline_styles ? 'style="' . esc_attr($inline_styles) . '"' : ''; ?>>

        <?php if (!empty($attributes['enable_search'])): ?>
            <div class="wpuf-search-field wpuf-mb-6">
                <?php
                // Use the proper search field rendering method that includes the search by dropdown
                $blocks_instance = new \WPUF\UserDirectory\Blocks();
                $search_data = [
                    'block_id' => $attributes['anchor'] ?? 'wpuf-directory-' . uniqid(),
                    'page_id' => get_the_ID(),
                    'search_placeholder' => $attributes['search_placeholder'] ?? __('Search Users', 'wpuf-pro'),
                    'searchable_fields' => $attributes['searchable_fields'] ?? ['user_login', 'user_email', 'display_name'],
                ];
                echo $blocks_instance->render_search_field($search_data, 'block');
                ?>
            </div>
        <?php endif; ?>

        <div class="wpuf-user-directory-content">
            <?php foreach ($users as $user): ?>
                <div class="wpuf-user-directory-item" data-user-id="<?php echo esc_attr($user->ID); ?>">
                    <div class="wpuf-user-item-content">

                        <!-- User Avatar -->
                        <div class="wpuf-user-avatar-section">
                            <?php
                            $avatar_options = [
                                'size' => 'medium',
                                'shape' => 'circle',
                                'show_link' => true,
                                'link_target' => '_self',
                            ];

                            // Load avatar component
                            $avatar_template_data = [
                                'user' => $user,
                                'options' => $avatar_options,
                            ];

                            echo wpuf_load_block_template('blocks/shared/components/avatar.php', $avatar_template_data);
                            ?>
                        </div>

                        <!-- User Information -->
                        <div class="wpuf-user-info-section">
                            <?php
                            $user_info_options = [
                                'show_name' => true,
                                'name_format' => 'display_name',
                                'name_heading' => 'h3',
                                'show_bio' => true,
                                'bio_limit' => 150,
                                'show_role' => !empty($attributes['showUserRole']),
                                'show_registration_date' => !empty($attributes['showRegistrationDate']),
                                'show_contact' => !empty($attributes['showContactInfo']),
                                'contact_options' => [
                                    'show_email' => !empty($attributes['showEmail']),
                                    'show_website' => !empty($attributes['showWebsite']),
                                    'show_phone' => !empty($attributes['showPhone']),
                                ],
                            ];

                            // Load user info component
                            $user_info_template_data = [
                                'user' => $user,
                                'options' => $user_info_options,
                            ];

                            echo wpuf_load_block_template('blocks/shared/components/user-info.php', $user_info_template_data);
                            ?>
                        </div>

                        <!-- User Actions -->
                        <?php if (!empty($attributes['showUserActions'])): ?>
                            <div class="wpuf-user-actions-section">
                                <a href="<?php echo esc_url(get_author_posts_url($user->ID)); ?>"
                                   class="wpuf-user-action-link">
                                    <?php echo esc_html__('View Profile', 'wpuf-pro'); ?>
                                </a>

                                <?php if (!empty($attributes['showContactButton'])): ?>
                                    <a href="mailto:<?php echo esc_attr($user->user_email); ?>"
                                       class="wpuf-user-action-link wpuf-contact-button">
                                        <?php echo esc_html__('Contact', 'wpuf-pro'); ?>
                                    </a>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>

                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <?php if (!empty($pagination) && !empty($pagination['total_pages']) && $pagination['total_pages'] > 1): ?>
            <div class="wpuf-user-directory-pagination">
                <?php
                $pagination_template_data = [
                    'pagination' => $pagination,
                    'current_page' => $pagination['current_page'] ?? 1,
                    'total_pages' => $pagination['total_pages'],
                    'base_url' => $pagination['base_url'] ?? '',
                ];

                echo wpuf_load_block_template('blocks/user-directory/components/pagination.php', $pagination_template_data);
                ?>
            </div>
        <?php endif; ?>

    </div>
</div>

<style>
.wpuf-layout-1 {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.wpuf-user-directory-item {
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    padding: 1.5rem;
    background: #ffffff;
    transition: all 0.2s ease-in-out;
}

.wpuf-user-directory-item:hover {
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    transform: translateY(-2px);
}

.wpuf-user-item-content {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
}

.wpuf-user-avatar-section {
    flex-shrink: 0;
}

.wpuf-user-info-section {
    flex: 1;
    min-width: 0;
}

.wpuf-user-actions-section {
    flex-shrink: 0;
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.wpuf-user-action-link {
    display: inline-block;
    padding: 0.5rem 1rem;
    background: #3b82f6;
    color: #ffffff;
    text-decoration: none;
    border-radius: 0.25rem;
    font-size: 0.875rem;
    transition: background-color 0.2s ease-in-out;
}

.wpuf-user-action-link:hover {
    background: #2563eb;
    color: #ffffff;
}

.wpuf-contact-button {
    background: #10b981;
}

.wpuf-contact-button:hover {
    background: #059669;
}



.wpuf-user-directory-pagination {
    margin-top: 2rem;
    display: flex;
    justify-content: center;
}

/* Responsive Design */
@media (max-width: 768px) {
    .wpuf-user-item-content {
        flex-direction: column;
        text-align: center;
    }

    .wpuf-user-actions-section {
        flex-direction: row;
        justify-content: center;
        margin-top: 1rem;
    }
}
</style>
