<?php
/**
 * User Posts Block Template
 *
 * @since 4.2.0
 *
 * @var WP_User $user User object
 * @var array $attributes Block attributes
 * @var string $content Block content
 * @var WP_Block $block Block instance
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! $user ) {
    return;
}

$posts_per_page = ! empty( $attributes['postsPerPage'] ) ? intval( $attributes['postsPerPage'] ) : 10;
$post_type      = ! empty( $attributes['postType'] ) ? sanitize_text_field( $attributes['postType'] ) : 'post';
$show_pagination = ! empty( $attributes['showPagination'] ) ? $attributes['showPagination'] : true;

// Get current page from URL parameters
$current_page = isset( $_GET['posts_page'] ) ? max( 1, intval( $_GET['posts_page'] ) ) : 1;


// Use WP_Query for pagination support
$posts_query = new \WP_Query( [
    'author'         => $user->ID,
    'post_type'      => $post_type,
    'post_status'    => 'publish',
    'posts_per_page' => $posts_per_page,
    'paged'          => $current_page,
    'orderby'        => 'date',
    'order'          => 'DESC',
] );

$user_posts = $posts_query->posts;
$total_posts = $posts_query->found_posts;
$total_pages = $posts_query->max_num_pages;


// Don't show anything if no posts
if ( empty( $user_posts ) ) {
    return;
}

// Get block wrapper attributes (includes styling from block editor)
$wrapper_attributes = wpuf_get_block_wrapper_attributes( $attributes, 'posts', [
    'class' => 'wpuf-w-full',
    'data-user-id' => $user->ID,
    'data-posts-per-page' => $posts_per_page,
    'data-show-pagination' => $show_pagination ? 'true' : 'false',
    'data-current-page' => $current_page,
    'data-total-pages' => $total_pages,
] );

?>
<div <?php echo $wrapper_attributes; ?>>
    <div class="wpuf-overflow-hidden wpuf-shadow wpuf-outline wpuf-outline-1 wpuf-outline-black/5 wpuf-rounded-lg wpuf-dark:shadow-none wpuf-dark:-outline-offset-1 wpuf-dark:outline-white/10">
        <table class="wpuf-relative wpuf-min-w-full wpuf-divide-y wpuf-divide-gray-300 wpuf-dark:divide-white/15">
            <thead class="wpuf-bg-gray-50 wpuf-dark:bg-gray-800/75">
                <tr>
                    <th scope="col" class="wpuf-py-3.5 wpuf-pr-3 wpuf-pl-4 wpuf-text-left wpuf-text-sm wpuf-font-semibold wpuf-text-gray-900 wpuf-sm:pl-6 wpuf-dark:text-gray-200">
                        <?php echo esc_html( apply_filters( 'wpuf_user_posts_title_label', $post_type_label, $post_type ) ); ?>
                    </th>
                    <th scope="col" class="wpuf-px-3 wpuf-py-3.5 wpuf-text-left wpuf-text-sm wpuf-font-semibold wpuf-text-gray-900 wpuf-dark:text-gray-200">
                        <?php echo esc_html( apply_filters( 'wpuf_user_posts_publish_date_label', __( 'Publish Date', 'wpuf-pro' ), $post_type ) ); ?>
                    </th>
                    <th scope="col" class="wpuf-py-3.5 wpuf-pr-4 wpuf-pl-3 wpuf-sm:pr-6">
                        <span class="wpuf-sr-only"><?php echo esc_html__( 'View', 'wpuf-pro' ); ?></span>
                    </th>
                </tr>
            </thead>
            <tbody class="wpuf-divide-y wpuf-divide-gray-200 wpuf-bg-white wpuf-dark:divide-white/10 wpuf-dark:bg-gray-800/50">
                <?php foreach ( $user_posts as $post ) : ?>
                <tr>
                    <td class="wpuf-py-4 wpuf-pr-3 wpuf-pl-4 wpuf-text-sm wpuf-font-medium wpuf-whitespace-nowrap wpuf-text-gray-900 wpuf-sm:pl-6 wpuf-dark:text-white">
                        <?php echo esc_html( get_the_title( $post->ID ) ); ?>
                    </td>
                    <td class="wpuf-px-3 wpuf-py-4 wpuf-text-sm wpuf-whitespace-nowrap wpuf-text-gray-500 wpuf-dark:text-gray-400">
                        <?php echo esc_html( get_the_date( 'F j, Y', $post->ID ) ); ?>
                    </td>
                    <td class="wpuf-py-4 wpuf-pr-4 wpuf-pl-3 wpuf-text-right wpuf-text-sm wpuf-font-medium wpuf-whitespace-nowrap wpuf-sm:pr-6">
                        <a
                            href="<?php echo esc_url( get_permalink( $post->ID ) ); ?>"
                            class="wpuf-text-indigo-600 wpuf-hover:text-indigo-900 wpuf-dark:text-indigo-400 wpuf-dark:hover:text-indigo-300"
                        >
                            <?php echo esc_html( __( 'Link ->', 'wpuf-pro' ) ); ?>
                            <span class="wpuf-sr-only">, <?php echo esc_html( get_the_title( $post->ID ) ); ?></span>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <?php 
    // Note: Pagination is handled by the parent block renderer (Blocks.php)
    // to avoid duplication when used in tabs or standalone contexts
    ?>
</div>

