jQuery(document).ready(function($) {
    'use strict';
    
    class DynamicMultiStepCRED {
        constructor(container) {
            this.container = container;
            this.currentStep = 1;
            this.totalSteps = parseInt(container.data('total-steps'));
            this.formId = container.data('form-id');
            this.postType = container.data('post-type');
            this.form = container.find('.multistep-cred-form');
            this.init();
        }
        
        init() {
            this.bindEvents();
            this.updateUI();
            this.setupFieldMapping();
        }
        
        bindEvents() {
            const self = this;
            
            // Next button
            $('.next-btn').on('click', function(e) {
                e.preventDefault();
                if (self.validateStep(self.currentStep)) {
                    self.goToStep(self.currentStep + 1);
                }
            });
            
            // Previous button
            $('.prev-btn').on('click', function(e) {
                e.preventDefault();
                self.goToStep(self.currentStep - 1);
            });
            
            // Submit button
            $('.submit-btn').on('click', function(e) {
                e.preventDefault();
                if (self.validateAllSteps()) {
                    self.submitForm();
                }
            });
            
            // Field changes - update progress
            $('.cred-field').on('change input', function() {
                self.updateStepCompletion(self.currentStep);
            });
        }
        
        goToStep(step) {
            if (step < 1 || step > this.totalSteps) return;
            
            // Hide current step
            this.form.find('.form-step[data-step="' + this.currentStep + '"]').removeClass('active').addClass('hidden');
            
            // Show new step
            this.form.find('.form-step[data-step="' + step + '"]').removeClass('hidden').addClass('active');
            
            this.currentStep = step;
            this.updateUI();
            
            // Trigger custom event
            $(document).trigger('multistep_change', [this.currentStep, this.totalSteps]);
        }
        
        validateStep(step) {
            const stepFields = this.form.find('.form-step[data-step="' + step + '"]').find('.cred-field');
            let isValid = true;
            
            stepFields.each(function() {
                const field = $(this);
                const isRequired = field.prop('required') || field.closest('.form-field').hasClass('cred-required');
                
                if (isRequired && !field.val().trim()) {
                    field.addClass('error');
                    field.closest('.form-field').addClass('has-error');
                    isValid = false;
                    
                    // Add error message
                    if (!field.closest('.form-field').find('.error-message').length) {
                        field.closest('.form-field').append('<span class="error-message">This field is required</span>');
                    }
                } else {
                    field.removeClass('error');
                    field.closest('.form-field').removeClass('has-error');
                    field.closest('.form-field').find('.error-message').remove();
                }
            });
            
            return isValid;
        }
        
        validateAllSteps() {
            let isValid = true;
            
            for (let i = 1; i <= this.totalSteps; i++) {
                if (!this.validateStep(i)) {
                    isValid = false;
                    // Go to first step with error
                    if (isValid === false && this.currentStep !== i) {
                        this.goToStep(i);
                        break;
                    }
                }
            }
            
            if (!isValid) {
                this.showMessage('Please fill all required fields before submitting.', 'error');
            }
            
            return isValid;
        }
        
        updateUI() {
            // Update progress bar
            const progressPercentage = (this.currentStep / this.totalSteps) * 100;
            $('.progress-fill').css('width', progressPercentage + '%');
            
            // Update progress steps
            $('.progress-step').removeClass('active completed');
            $('.progress-step').each(function() {
                const step = parseInt($(this).data('step'));
                if (step < this.currentStep) {
                    $(this).addClass('completed');
                } else if (step === this.currentStep) {
                    $(this).addClass('active');
                }
            });
            
            // Update navigation buttons
            $('.prev-btn').toggle(this.currentStep > 1);
            $('.next-btn').toggle(this.currentStep < this.totalSteps);
            $('.submit-btn').toggle(this.currentStep === this.totalSteps);
            
            // Update step headers
            this.updateStepHeaders();
        }
        
        updateStepHeaders() {
            $('.step-header').hide();
            this.form.find('.form-step[data-step="' + this.currentStep + '"] .step-header').show();
        }
        
        updateStepCompletion(step) {
            const stepContainer = this.form.find('.form-step[data-step="' + step + '"]');
            const filledFields = stepContainer.find('.cred-field').filter(function() {
                return $(this).val().trim() !== '';
            }).length;
            const totalFields = stepContainer.find('.cred-field').length;
            
            if (filledFields > 0) {
                stepContainer.addClass('has-data');
            } else {
                stepContainer.removeClass('has-data');
            }
        }
        
        setupFieldMapping() {
            // Map our custom fields to the hidden CRED form
            $(document).on('change input', '.cred-field', function() {
                const field = $(this);
                const fieldName = field.attr('name');
                const fieldValue = field.val();
                
                // Update corresponding field in hidden CRED form
                $('#multistep-cred-form-' + this.formId).find('[name="' + fieldName + '"]').val(fieldValue);
            });
        }
        
        submitForm() {
            this.showLoading(true);
            
            // Transfer all data to hidden CRED form
            $('.cred-field').each(function() {
                const field = $(this);
                const fieldName = field.attr('name');
                const fieldValue = field.val();
                
                $('#multistep-cred-form-' + this.formId).find('[name="' + fieldName + '"]').val(fieldValue);
            });
            
            // Submit the actual CRED form
            setTimeout(() => {
                const credForm = $('#multistep-cred-form-' + this.formId).find('form');
                if (credForm.length) {
                    credForm.submit();
                } else {
                    this.showMessage('Form submitted successfully!', 'success');
                    this.showLoading(false);
                }
            }, 1000);
        }
        
        showLoading(show) {
            if (show) {
                $('.multistep-loading').show();
                $('.multistep-navigation').hide();
            } else {
                $('.multistep-loading').hide();
                $('.multistep-navigation').show();
            }
        }
        
        showMessage(message, type) {
            const messageClass = type === 'error' ? 'error-message' : 'success-message';
            $('.multistep-message').remove();
            
            const messageEl = $('<div class="multistep-message ' + messageClass + '">' + message + '</div>');
            this.container.prepend(messageEl);
            
            setTimeout(() => {
                messageEl.fadeOut();
            }, 5000);
        }
    }
    
    // Initialize multi-step forms
    $('.dynamic-multistep-cred-container').each(function() {
        new DynamicMultiStepCRED($(this));
    });
});