/**
 * Display Form Entries - JavaScript
 */

(function($) {
    'use strict';

    // Main DFE object
    window.DFE = window.DFE || {};

    // Initialize when document is ready
    $(document).ready(function() {
        DFE.init();
    });

    // Main initialization
    DFE.init = function() {
        DFE.initSearch();
        DFE.initPagination();
        DFE.initSorting();
        DFE.initResponsive();
        DFE.initAccessibility();
    };

    // Search functionality
    DFE.initSearch = function() {
        $('.dfe-search form').on('submit', function(e) {
            var $form = $(this);
            var $input = $form.find('input[type="text"]');
            var searchTerm = $input.val().trim();

            if (searchTerm === '') {
                e.preventDefault();
                return false;
            }

            // Show loading state
            DFE.showLoading($form.closest('.dfe-container'));
        });
    };

    // Pagination functionality
    DFE.initPagination = function() {
        $('.dfe-pagination .page-numbers').on('click', function(e) {
            var $link = $(this);
            var href = $link.attr('href');

            if (href && href !== '#') {
                // Show loading state
                DFE.showLoading($link.closest('.dfe-container'));
            }
        });
    };

    // Sorting functionality
    DFE.initSorting = function() {
        $('.dfe-table th[data-sort]').on('click', function(e) {
            e.preventDefault();
            
            var $th = $(this);
            var sortKey = $th.data('sort');
            var currentDirection = $th.data('direction') || 'asc';
            var newDirection = currentDirection === 'asc' ? 'desc' : 'asc';

            // Update URL with sort parameters
            var url = new URL(window.location);
            url.searchParams.set('sort', sortKey);
            url.searchParams.set('sort_direction', newDirection);

            // Show loading state
            DFE.showLoading($th.closest('.dfe-container'));

            // Navigate to new URL
            window.location.href = url.toString();
        });
    };

    // Responsive functionality
    DFE.initResponsive = function() {
        // Handle table responsiveness
        $('.dfe-table-container').each(function() {
            var $container = $(this);
            var $table = $container.find('.dfe-table');
            
            if ($table.width() > $container.width()) {
                $container.addClass('dfe-table-scrollable');
            }
        });

        // Handle card grid responsiveness
        $(window).on('resize', function() {
            DFE.updateCardGrid();
        });

        DFE.updateCardGrid();
    };

    // Update card grid layout
    DFE.updateCardGrid = function() {
        $('.dfe-cards').each(function() {
            var $cards = $(this);
            var containerWidth = $cards.width();
            var cardWidth = 300; // Minimum card width
            var gap = 20; // Gap between cards

            // Calculate optimal number of columns
            var columns = Math.floor((containerWidth + gap) / (cardWidth + gap));
            columns = Math.max(1, Math.min(columns, 4)); // Between 1 and 4 columns

            $cards.css('grid-template-columns', 'repeat(' + columns + ', 1fr)');
        });
    };

    // Accessibility enhancements
    DFE.initAccessibility = function() {
        // Add keyboard navigation to tables
        $('.dfe-table').each(function() {
            var $table = $(this);
            var $rows = $table.find('tbody tr');

            $rows.on('keydown', function(e) {
                var $currentRow = $(this);
                var currentIndex = $rows.index($currentRow);

                switch(e.keyCode) {
                    case 38: // Up arrow
                        e.preventDefault();
                        if (currentIndex > 0) {
                            $rows.eq(currentIndex - 1).focus();
                        }
                        break;
                    case 40: // Down arrow
                        e.preventDefault();
                        if (currentIndex < $rows.length - 1) {
                            $rows.eq(currentIndex + 1).focus();
                        }
                        break;
                }
            });

            // Make rows focusable
            $rows.attr('tabindex', '0');
        });

        // Add ARIA labels
        $('.dfe-search input[type="text"]').attr('aria-label', 'Search entries');
        $('.dfe-pagination .page-numbers').attr('aria-label', function() {
            var text = $(this).text();
            return 'Go to page ' + text;
        });
    };

    // Show loading state
    DFE.showLoading = function($container) {
        if ($container.length) {
            $container.addClass('dfe-loading');
            
            // Remove loading class after a delay (in case of AJAX)
            setTimeout(function() {
                $container.removeClass('dfe-loading');
            }, 2000);
        }
    };

    // Utility functions
    DFE.utils = {
        // Debounce function
        debounce: function(func, wait, immediate) {
            var timeout;
            return function() {
                var context = this, args = arguments;
                var later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                var callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        },

        // Throttle function
        throttle: function(func, limit) {
            var inThrottle;
            return function() {
                var args = arguments;
                var context = this;
                if (!inThrottle) {
                    func.apply(context, args);
                    inThrottle = true;
                    setTimeout(function() {
                        inThrottle = false;
                    }, limit);
                }
            };
        },

        // Format date
        formatDate: function(dateString) {
            if (!dateString) return '';
            
            var date = new Date(dateString);
            if (isNaN(date.getTime())) return dateString;
            
            return date.toLocaleDateString();
        },

        // Format number
        formatNumber: function(number, decimals) {
            if (isNaN(number)) return number;
            
            return parseFloat(number).toFixed(decimals || 0);
        },

        // Escape HTML
        escapeHtml: function(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }
    };

    // AJAX functionality (if needed)
    DFE.ajax = {
        // Load entries via AJAX
        loadEntries: function(params, callback) {
            $.ajax({
                url: dfe_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'dfe_load_entries',
                    nonce: dfe_ajax.nonce,
                    params: params
                },
                success: function(response) {
                    if (response.success) {
                        callback(response.data);
                    } else {
                        console.error('DFE AJAX Error:', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('DFE AJAX Error:', error);
                }
            });
        },

        // Search entries via AJAX
        searchEntries: function(searchTerm, callback) {
            DFE.ajax.loadEntries({
                search: searchTerm
            }, callback);
        }
    };

    // Event handlers
    DFE.events = {
        // Handle entry link clicks
        onEntryClick: function(e) {
            var $link = $(this);
            var href = $link.attr('href');
            
            if (href && href !== '#') {
                // Add loading state
                DFE.showLoading($link.closest('.dfe-container'));
            }
        },

        // Handle field value copy
        onFieldCopy: function(e) {
            e.preventDefault();
            
            var $field = $(this);
            var text = $field.text();
            
            // Copy to clipboard
            if (navigator.clipboard) {
                navigator.clipboard.writeText(text).then(function() {
                    DFE.showNotification('Copied to clipboard!', 'success');
                });
            } else {
                // Fallback for older browsers
                var textArea = document.createElement('textarea');
                textArea.value = text;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                DFE.showNotification('Copied to clipboard!', 'success');
            }
        }
    };

    // Notification system
    DFE.showNotification = function(message, type) {
        var $notification = $('<div class="dfe-notification dfe-notification-' + type + '">' + message + '</div>');
        
        $('body').append($notification);
        
        // Show notification
        setTimeout(function() {
            $notification.addClass('dfe-notification-show');
        }, 100);
        
        // Hide notification
        setTimeout(function() {
            $notification.removeClass('dfe-notification-show');
            setTimeout(function() {
                $notification.remove();
            }, 300);
        }, 3000);
    };

    // Initialize event handlers
    $(document).on('click', '.dfe-entry-link', DFE.events.onEntryClick);
    $(document).on('click', '.dfe-field-copy', DFE.events.onFieldCopy);

    // Expose DFE to global scope
    window.DFE = DFE;

})(jQuery); 