/**
 * Token MultiSelect Control
 * 
 * A tokenized multiselect control for the block editor
 * Based on: https://github.com/rmorse/wp-token-multiselect-control
 */

(function() {
	'use strict';

	if (typeof wp === 'undefined') {
		return;
	}

	const { Component, createElement: el } = wp.element;
	const { BaseControl } = wp.components;
	const { __ } = wp.i18n;

	class TokenMultiSelectControl extends Component {
		constructor(props) {
			super(props);
			
			this.state = {
				suggestions: [],
				showSuggestions: false,
				activeSuggestion: -1,
				inputValue: '',
				focusedToken: -1
			};

			this.containerRef = wp.element.createRef();
			this.inputRef = wp.element.createRef();
			
			this.handleInputChange = this.handleInputChange.bind(this);
			this.handleInputFocus = this.handleInputFocus.bind(this);
			this.handleInputBlur = this.handleInputBlur.bind(this);
			this.handleInputKeyDown = this.handleInputKeyDown.bind(this);
			this.handleSuggestionClick = this.handleSuggestionClick.bind(this);
			this.handleTokenClick = this.handleTokenClick.bind(this);
			this.handleTokenKeyDown = this.handleTokenKeyDown.bind(this);
			this.handleContainerClick = this.handleContainerClick.bind(this);
		}

		componentDidMount() {
			this.updateSuggestions();
		}

		componentDidUpdate(prevProps) {
			if (prevProps.options !== this.props.options) {
				this.updateSuggestions();
			}
		}

		updateSuggestions() {
			const { options = [], value = [] } = this.props;
			const { inputValue } = this.state;
			
			// Filter out already selected options and match input
			const filteredSuggestions = options.filter(option => {
				const isSelected = value.includes(option.value);
				const matchesInput = !inputValue || 
					option.label.toLowerCase().includes(inputValue.toLowerCase());
				return !isSelected && matchesInput;
			});

			this.setState({
				suggestions: filteredSuggestions,
				activeSuggestion: filteredSuggestions.length > 0 ? 0 : -1
			});
		}

		handleInputChange(event) {
			const inputValue = event.target.value;
			this.setState({ inputValue }, () => {
				this.updateSuggestions();
				this.setState({ showSuggestions: true });
			});
		}

		handleInputFocus() {
			this.setState({ showSuggestions: true });
			this.updateSuggestions();
		}

		handleInputBlur(event) {
			// Delay hiding suggestions to allow for clicks
			setTimeout(() => {
				this.setState({ showSuggestions: false, inputValue: '' });
			}, 200);
		}

		handleInputKeyDown(event) {
			const { suggestions, activeSuggestion, showSuggestions } = this.state;
			const { value = [] } = this.props;

			switch (event.key) {
				case 'ArrowDown':
					event.preventDefault();
					if (showSuggestions && suggestions.length > 0) {
						this.setState({
							activeSuggestion: Math.min(activeSuggestion + 1, suggestions.length - 1)
						});
					} else {
						this.setState({ showSuggestions: true });
						this.updateSuggestions();
					}
					break;

				case 'ArrowUp':
					event.preventDefault();
					if (showSuggestions && suggestions.length > 0) {
						this.setState({
							activeSuggestion: Math.max(activeSuggestion - 1, 0)
						});
					}
					break;

				case 'Enter':
					event.preventDefault();
					if (showSuggestions && activeSuggestion >= 0 && suggestions[activeSuggestion]) {
						this.addToken(suggestions[activeSuggestion].value);
					}
					break;

				case 'Escape':
					this.setState({ showSuggestions: false, inputValue: '' });
					break;

				case 'Backspace':
					if (this.state.inputValue === '' && value.length > 0) {
						this.removeToken(value[value.length - 1]);
					}
					break;

				case 'ArrowLeft':
					if (this.state.inputValue === '' && value.length > 0) {
						this.setState({ focusedToken: value.length - 1 });
					}
					break;

				case 'ArrowRight':
					if (this.state.inputValue === '' && this.state.focusedToken >= 0) {
						this.setState({ focusedToken: -1 });
					}
					break;
			}
		}

		handleSuggestionClick(suggestionValue) {
			this.addToken(suggestionValue);
			// Keep focus on input and show suggestions again
			if (this.inputRef.current) {
				this.inputRef.current.focus();
				setTimeout(() => {
					this.setState({ showSuggestions: true });
					this.updateSuggestions();
				}, 50);
			}
		}

		handleTokenClick(tokenValue) {
			this.removeToken(tokenValue);
		}

		handleTokenKeyDown(event, tokenValue) {
			if (event.key === 'Backspace' || event.key === 'Delete') {
				this.removeToken(tokenValue);
			}
		}

		handleContainerClick() {
			if (this.inputRef.current) {
				this.inputRef.current.focus();
				// Show suggestions when clicking container
				this.setState({ showSuggestions: true });
				this.updateSuggestions();
			}
		}

		addToken(tokenValue) {
			const { value = [], onChange } = this.props;
			if (!value.includes(tokenValue)) {
				const newValue = [...value, tokenValue];
				onChange(newValue);
			}
			this.setState({ 
				inputValue: '', 
				focusedToken: -1
			});
		}

		removeToken(tokenValue) {
			const { value = [], onChange } = this.props;
			const newValue = value.filter(v => v !== tokenValue);
			onChange(newValue);
			this.setState({ focusedToken: -1 });
		}

		getTokenLabel(tokenValue) {
			const { options = [] } = this.props;
			const option = options.find(opt => opt.value === tokenValue);
			return option ? option.label : tokenValue;
		}

		render() {
			const { label, help, className = '' } = this.props;
			const { value = [] } = this.props;
			const { suggestions, showSuggestions, activeSuggestion, inputValue, focusedToken } = this.state;

			return el(BaseControl, {
				label,
				help,
				className: `token-multiselect-control ${className}`
			},
				el('div', {
					className: 'token-multiselect-container',
					onClick: this.handleContainerClick,
					ref: this.containerRef
				},
					// Render selected tokens
					value.map((tokenValue, index) => 
						el('span', {
							key: tokenValue,
							className: `token-multiselect-token ${focusedToken === index ? 'focused' : ''}`,
							onClick: (e) => {
								e.stopPropagation();
								this.handleTokenClick(tokenValue);
							},
							onKeyDown: (e) => this.handleTokenKeyDown(e, tokenValue),
							tabIndex: 0,
							role: 'button',
							'aria-label': __('Remove', 'displayformentries') + ' ' + this.getTokenLabel(tokenValue)
						},
							el('span', { className: 'token-text' }, this.getTokenLabel(tokenValue)),
							el('span', { className: 'token-remove', 'aria-hidden': 'true' }, '×')
						)
					),
					
					// Input field
					el('input', {
						type: 'text',
						className: 'token-multiselect-input',
						value: inputValue,
						onChange: this.handleInputChange,
						onFocus: this.handleInputFocus,
						onBlur: this.handleInputBlur,
						onKeyDown: this.handleInputKeyDown,
						placeholder: value.length === 0 ? __('Select fields...', 'displayformentries') : '',
						ref: this.inputRef
					}),

					// Suggestions dropdown
					showSuggestions && suggestions.length > 0 && el('div', {
						className: 'token-multiselect-suggestions'
					},
						suggestions.map((suggestion, index) =>
							el('div', {
								key: suggestion.value,
								className: `token-multiselect-suggestion ${index === activeSuggestion ? 'active' : ''}`,
								onClick: () => this.handleSuggestionClick(suggestion.value),
								onMouseEnter: () => this.setState({ activeSuggestion: index })
							}, suggestion.label)
						)
					)
				)
			);
		}
	}

	// Export for use in other files
	window.TokenMultiSelectControl = TokenMultiSelectControl;

})(); 