<?php
/**
 * Plugin Name:         Display Form Entries
 * Plugin URI:          https://ristrettoapps.com/product/display-form-entries/
 * Description:         Display Gravity Forms entries on the frontend with customizable layouts and field formatting.
 * Version:             1.0.1
 * Requires PHP:        7.4.0
 * Author:              Ristretto Apps
 * Author URI:          https://ristrettoapps.com
 * Text Domain:         displayformentries
 * License:             GPLv2 or later
 * License URI:         http://www.gnu.org/licenses/gpl-2.0.html
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

/** Constants */
define( 'DFE_PLUGIN_VERSION', '1.0.1' );
define( 'DFE_FILE', __FILE__ );
define( 'DFE_URL', plugin_dir_url( __FILE__ ) );
define( 'DFE_DIR', plugin_dir_path( __FILE__ ) );

/**
 * Main plugin class
 */
class DisplayFormEntries {
	
	/**
	 * Plugin instance
	 */
	private static $instance = null;
	
	/**
	 * Get plugin instance
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
	
	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init();
	}
	
	/**
	 * Initialize plugin
	 */
	private function init() {
		// Load required files first
		$this->load_files();
		
		// Initialize hooks - check for Gravity Forms later
		$this->init_hooks();
	}
	
	/**
	 * Load required files
	 */
	private function load_files() {
		$files = array(
			'includes/class-dfe-common.php',
			'includes/class-dfe-api.php',
			'includes/class-dfe-helper.php',
			'includes/class-dfe-connector.php',
			'includes/class-dfe-field-renderer.php',
			'includes/class-dfe-template.php',
			'includes/class-dfe-shortcodes.php',
			'includes/class-dfe-ajax.php',
			'includes/class-dfe-blocks.php',
		);
		
		foreach ( $files as $file ) {
			$file_path = DFE_DIR . $file;
			if ( file_exists( $file_path ) ) {
				require_once $file_path;
			}
		}
	}
	
	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		// Check for Gravity Forms on plugins_loaded hook (after all plugins are loaded)
		add_action( 'plugins_loaded', array( $this, 'check_gravity_forms' ), 20 );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}
	
	/**
	 * Check if Gravity Forms is available and initialize plugin
	 */
	public function check_gravity_forms() {
		// Check multiple ways Gravity Forms might be available
		$gravity_forms_available = false;
		
		// Method 1: Check for GFAPI class
		if ( class_exists( 'GFAPI' ) ) {
			$gravity_forms_available = true;
		}
		
		// Method 2: Check for Gravity Forms plugin function
		if ( function_exists( 'gravity_form' ) ) {
			$gravity_forms_available = true;
		}
		
		// Method 3: Check if Gravity Forms plugin is active
		if ( is_plugin_active( 'gravityforms/gravityforms.php' ) ) {
			$gravity_forms_available = true;
		}
		
		// Method 4: Check for Gravity Forms constant
		if ( defined( 'GF_MINIMUM_WP_VERSION' ) ) {
			$gravity_forms_available = true;
		}
		
		if ( $gravity_forms_available ) {
			$this->init_plugin();
		} else {
			add_action( 'admin_notices', array( $this, 'gravity_forms_missing_notice' ) );
		}
	}
	
	/**
	 * Initialize plugin after WordPress is loaded
	 */
	public function init_plugin() {
		// Initialize shortcodes
		DFE_Shortcodes::init();
		
		// Initialize AJAX handlers
		DFE_AJAX::init();
		
		// Initialize blocks
		DFE_Blocks::init();
	}
	
	/**
	 * Enqueue frontend scripts and styles
	 */
	public function enqueue_scripts() {
		wp_enqueue_style( 'dfe-styles', DFE_URL . 'assets/css/dfe-styles.css', array(), DFE_PLUGIN_VERSION );
		wp_enqueue_script( 'dfe-scripts', DFE_URL . 'assets/js/dfe-scripts.js', array( 'jquery' ), DFE_PLUGIN_VERSION, true );
	}
	
	/**
	 * Admin notice for missing Gravity Forms
	 */
	public function gravity_forms_missing_notice() {
		?>
		<div class="notice notice-error">
			<p><?php _e( 'Display Form Entries requires Gravity Forms to be installed and activated.', 'displayformentries' ); ?></p>
		</div>
		<?php
	}
}

// Initialize the plugin
DisplayFormEntries::get_instance();