<?php
/**
 * AJAX handlers for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Your Name
 * @link      https://example.com
 * @copyright Copyright 2024
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_AJAX {

	/**
	 * Initialize AJAX handlers
	 */
	public static function init() {
		add_action( 'wp_ajax_dfe_load_entry_detail', array( __CLASS__, 'load_entry_detail' ) );
		add_action( 'wp_ajax_nopriv_dfe_load_entry_detail', array( __CLASS__, 'load_entry_detail' ) );
		add_action( 'wp_ajax_dfe_get_form_fields', array( __CLASS__, 'get_form_fields' ) );
	}

	/**
	 * Load entry detail via AJAX
	 */
	public static function load_entry_detail() {
		// Verify nonce
		if ( ! wp_verify_nonce( $_POST['nonce'], 'dfe_ajax_nonce' ) ) {
			wp_die( 'Security check failed' );
		}

		$entry_id = intval( $_POST['entry_id'] );
		$form_id = intval( $_POST['form_id'] );

		if ( ! $entry_id || ! $form_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid entry or form ID.', 'displayformentries' ) ) );
		}

		// Get form and entry
		$form = DFE_Connector::get_form( $form_id );
		$entry = DFE_Connector::get_entry( $entry_id );

		if ( ! $form || ! $entry ) {
			wp_send_json_error( array( 'message' => __( 'Entry not found.', 'displayformentries' ) ) );
		}

		// Get all form fields
		$fields = DFE_Connector::get_form_fields( $form );

		// Render entry detail
		$html = DFE_Field_Renderer::render_single_entry( $entry, $fields, $form, array(
			'show_labels' => true,
			'hide_empty'  => true,
			'show_edit'   => true,
		) );

		wp_send_json_success( array( 'html' => $html ) );
	}

	/**
	 * Get form fields via AJAX for the block editor
	 */
	public static function get_form_fields() {
		// Verify nonce
		if ( ! wp_verify_nonce( $_POST['nonce'], 'dfe_blocks_nonce' ) ) {
			wp_die( 'Security check failed' );
		}

		$form_id = intval( $_POST['form_id'] );

		if ( ! $form_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid form ID.', 'displayformentries' ) ) );
		}

		// Get form
		$form = DFE_Connector::get_form( $form_id );
		if ( ! $form ) {
			wp_send_json_error( array( 'message' => __( 'Form not found.', 'displayformentries' ) ) );
		}

		// Get form fields
		$fields = DFE_Connector::get_form_fields( $form );
		
		// Format fields for the token multiselect
		$field_options = array();
		
		// Add form fields first
		foreach ( $fields as $field ) {
			// Skip fields that shouldn't be displayed
			if ( in_array( $field['type'], array( 'page', 'section', 'html' ) ) ) {
				continue;
			}
			
			$label = ! empty( $field['adminLabel'] ) ? $field['adminLabel'] : $field['label'];
			
			// Handle complex fields with inputs
			if ( isset( $field['inputs'] ) && is_array( $field['inputs'] ) ) {
				// Add parent field option first for complex fields
				$field_options[] = array(
					'value' => $field['id'],
					'label' => $label . ' (Complete Field)'
				);
				
				// Add individual subfield options
				foreach ( $field['inputs'] as $input ) {
					if ( ! empty( $input['label'] ) ) {
						$subfield_label = $input['label'];
						// Add note about grouping for name and date fields
						if ( in_array( $field['type'], array( 'name', 'date' ) ) ) {
							$subfield_label .= ' → Combined in "' . $label . '"';
						}
						$field_options[] = array(
							'value' => $input['id'],
							'label' => $subfield_label
						);
					}
				}
			} else {
				$field_options[] = array(
					'value' => $field['id'],
					'label' => $label . ' (ID: ' . $field['id'] . ')'
				);
			}
		}
		
		// Add standard entry meta fields at the end
		$entry_meta_fields = array(
			'id' => 'Entry ID',
			'date_created' => 'Entry Date', 
			'date_updated' => 'Date Updated',
			'is_starred' => 'Is Starred',
			'is_read' => 'Is Read',
			'ip' => 'User IP',
			'source_url' => 'Source URL',
			'user_agent' => 'User Agent',
			'payment_date' => 'Payment Date',
			'payment_amount' => 'Payment Amount', 
			'payment_status' => 'Payment Status',
			'transaction_id' => 'Transaction ID',
			'created_by' => 'Created By (User ID)',
			'post_id' => 'Post ID'
		);
		
		foreach ( $entry_meta_fields as $field_key => $field_label ) {
			$field_options[] = array(
				'value' => $field_key,
				'label' => $field_label . ' (Entry Meta)'
			);
		}

		wp_send_json_success( array( 'fields' => $field_options ) );
	}
} 