<?php
/**
 * Connector functions for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Ristretto Apps
 * @link      https://ristrettoapps.com
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_Connector {

	/**
	 * Returns the form object for a given Form ID.
	 *
	 * @param mixed $form_id Form ID
	 *
	 * @return mixed False: no form ID specified or Gravity Forms isn't active. Array: Form returned from Gravity Forms
	 */
	public static function get_form( $form_id ) {
		return DFE_Common::get_form( $form_id );
	}

	/**
	 * Get the form array for an entry based only on the entry ID
	 *
	 * @param int|string $entry_slug Entry slug
	 *
	 * @return array Gravity Forms form array
	 */
	public static function get_form_from_entry_id( $entry_slug ) {
		return DFE_Common::get_form_from_entry_id( $entry_slug );
	}

	/**
	 * Alias of GFAPI::get_forms()
	 *
	 * @param bool|string  $active Status of forms. Use `any` to get array of forms with any status. Default: `true`
	 * @param bool         $trash Include forms in trash? Default: `false`
	 * @param string|array $order_by Optional. Either the field name to order by or an array of multiple orderby fields as $orderby => $order.
	 * @param string       $order Optional. Either 'ASC' or 'DESC'. Only used if $orderby is a string.
	 *
	 * @return array Empty array if GFAPI class isn't available or no forms. Otherwise, the array of Forms
	 */
	public static function get_forms( $active = true, $trash = false, $order_by = 'date_created', $order = 'ASC' ) {
		return DFE_Common::get_forms( $active, $trash, $order_by, $order );
	}

	/**
	 * Get form fields
	 *
	 * @param array|int $form Form array or form ID
	 * @param bool      $add_default_properties Whether to add default properties to fields
	 * @param bool      $include_parent_field Whether to include parent fields
	 *
	 * @return array Array of form fields
	 */
	public static function get_form_fields( $form = '', $add_default_properties = false, $include_parent_field = true ) {
		return DFE_Common::get_form_fields( $form, $add_default_properties, $include_parent_field );
	}

	/**
	 * Get entry meta
	 *
	 * @param int  $form_id Form ID
	 * @param bool $only_default_column Whether to only get default column
	 *
	 * @return array Entry meta
	 */
	public static function get_entry_meta( $form_id, $only_default_column = true ) {
		if ( ! class_exists( 'GFAPI' ) ) {
			return array();
		}

		$meta = \GFAPI::get_entry_meta( $form_id );

		if ( $only_default_column ) {
			$meta = array_filter( $meta, function( $field ) {
				return isset( $field['isDefaultColumn'] ) && $field['isDefaultColumn'];
			} );
		}

		return $meta;
	}

	/**
	 * Get entry IDs
	 *
	 * @param int   $form_id Form ID
	 * @param array $search_criteria Search criteria
	 *
	 * @return array Array of entry IDs
	 */
	public static function get_entry_ids( $form_id, $search_criteria = array() ) {
		if ( ! class_exists( 'GFAPI' ) ) {
			return array();
		}

		$entries = \GFAPI::get_entries( $form_id, $search_criteria, null, null, $total );

		if ( ! is_array( $entries ) ) {
			return array();
		}

		return wp_list_pluck( $entries, 'id' );
	}

	/**
	 * Get entries
	 *
	 * @param int|array $form_ids Form ID or array of form IDs
	 * @param array     $passed_criteria Search criteria
	 * @param int       $total Reference to total count
	 *
	 * @return array Array of entries
	 */
	public static function get_entries( $form_ids = null, $passed_criteria = null, &$total = null ) {
		return DFE_Common::get_entries( $form_ids, $passed_criteria, $total );
	}

	/**
	 * Get a single entry
	 *
	 * @param int|string $entry_slug Entry ID or slug
	 * @param bool       $force_allow_ids Whether to force allow IDs
	 *
	 * @return array|false Entry array or false if not found
	 */
	public static function get_entry( $entry_slug, $force_allow_ids = false ) {
		return DFE_Common::get_entry( $entry_slug, $force_allow_ids );
	}

	/**
	 * Get field label
	 *
	 * @param array  $form Form array
	 * @param mixed  $field_id Field ID
	 * @param string $field_value Field value
	 *
	 * @return string Field label
	 */
	public static function get_field_label( $form, $field_id, $field_value = '' ) {
		return DFE_Common::get_field_label( $form, $field_id, $field_value );
	}

	/**
	 * Get field from form
	 *
	 * @param array $form Form array
	 * @param mixed $field_id Field ID
	 *
	 * @return array|false Field array or false if not found
	 */
	public static function get_field( $form, $field_id ) {
		return DFE_Common::get_field( $form, $field_id );
	}

	/**
	 * Check if post has shortcode
	 *
	 * @param WP_Post|null $post Post object
	 *
	 * @return bool True if post has shortcode, false if not
	 */
	public static function has_shortcode( $post = null ) {
		if ( ! $post ) {
			global $post;
		}

		if ( ! $post ) {
			return false;
		}

		return self::has_shortcode_r( $post->post_content, 'dfe_entries' );
	}

	/**
	 * Check if content has shortcode recursively
	 *
	 * @param string $content Content to check
	 * @param string $tag Shortcode tag
	 *
	 * @return bool True if content has shortcode, false if not
	 */
	public static function has_shortcode_r( $content, $tag = 'dfe_entries' ) {
		if ( empty( $content ) ) {
			return false;
		}

		// Check for shortcode in content
		if ( has_shortcode( $content, $tag ) ) {
			return true;
		}

		// Check for shortcode in blocks
		if ( function_exists( 'parse_blocks' ) ) {
			$blocks = parse_blocks( $content );
			
			foreach ( $blocks as $block ) {
				if ( isset( $block['innerContent'] ) ) {
					foreach ( $block['innerContent'] as $inner_content ) {
						if ( self::has_shortcode_r( $inner_content, $tag ) ) {
							return true;
						}
					}
				}
			}
		}

		return false;
	}

	/**
	 * Get field type
	 *
	 * @param array $form Form array
	 * @param mixed $field_id Field ID
	 *
	 * @return string Field type
	 */
	public static function get_field_type( $form = null, $field_id = '' ) {
		return DFE_Common::get_field_type( $form, $field_id );
	}

	/**
	 * Check if field is numeric
	 *
	 * @param array $form Form array
	 * @param mixed $field Field array or field ID
	 *
	 * @return bool True if field is numeric, false if not
	 */
	public static function is_field_numeric( $form = null, $field = '' ) {
		if ( empty( $field ) ) {
			return false;
		}

		// If field ID is passed, get the field
		if ( is_numeric( $field ) && $form ) {
			$field = DFE_Common::get_field( $form, $field );
		}

		if ( ! $field || ! is_array( $field ) ) {
			return false;
		}

		$numeric_types = array(
			'number',
			'calculation',
			'quantity',
			'product',
			'shipping',
			'total',
		);

		return in_array( $field['type'], $numeric_types, true );
	}

	/**
	 * Get sortable fields
	 *
	 * @param int    $form_id Form ID
	 * @param string $current Current field
	 *
	 * @return array Array of sortable fields
	 */
	public static function get_sortable_fields( $form_id, $current = '' ) {
		$form = DFE_Common::get_form( $form_id );

		if ( ! $form ) {
			return array();
		}

		$fields = DFE_Common::get_form_fields( $form );
		$sortable_fields = array();

		// Add default sortable fields
		$sortable_fields['id'] = __( 'Entry ID', 'displayformentries' );
		$sortable_fields['date_created'] = __( 'Entry Date', 'displayformentries' );
		$sortable_fields['date_updated'] = __( 'Entry Updated', 'displayformentries' );

		// Add form fields
		foreach ( $fields as $field ) {
			if ( self::is_field_sortable( $field ) ) {
				$sortable_fields[ $field['id'] ] = $field['label'];
			}
		}

		return $sortable_fields;
	}

	/**
	 * Check if field is sortable
	 *
	 * @param array $field Field array
	 *
	 * @return bool True if field is sortable, false if not
	 */
	private static function is_field_sortable( $field ) {
		$non_sortable_types = array(
			'list',
			'textarea',
			'fileupload',
			'html',
			'section',
			'page',
		);

		return ! in_array( $field['type'], $non_sortable_types, true );
	}

	/**
	 * Get forms as options for select fields
	 *
	 * @param bool $active Whether to get active forms only
	 * @param bool $trash Whether to include trashed forms
	 *
	 * @return array Array of form options
	 */
	public static function get_forms_as_options( $active = true, $trash = false ) {
		$forms = self::get_forms( $active, $trash );
		$options = array();

		foreach ( $forms as $form ) {
			$options[ $form['id'] ] = $form['title'];
		}

		return $options;
	}

	/**
	 * Get form fields as options for select fields
	 *
	 * @param int $form_id Form ID
	 *
	 * @return array Array of field options
	 */
	public static function get_form_fields_as_options( $form_id ) {
		$form = DFE_Common::get_form( $form_id );

		if ( ! $form ) {
			return array();
		}

		$fields = DFE_Common::get_form_fields( $form );
		$options = array();

		foreach ( $fields as $field ) {
			$options[ $field['id'] ] = $field['label'];
		}

		return $options;
	}

	/**
	 * Check if form has entries
	 *
	 * @param int $form_id Form ID
	 *
	 * @return bool True if form has entries, false if not
	 */
	public static function form_has_entries( $form_id ) {
		if ( ! class_exists( 'GFAPI' ) ) {
			return false;
		}

		$entries = \GFAPI::get_entries( $form_id, array(), null, null, $total );

		return is_array( $entries ) && ! empty( $entries );
	}

	/**
	 * Get entry count for form
	 *
	 * @param int $form_id Form ID
	 *
	 * @return int Entry count
	 */
	public static function get_entry_count( $form_id ) {
		if ( ! class_exists( 'GFAPI' ) ) {
			return 0;
		}

		$entries = \GFAPI::get_entries( $form_id, array(), null, null, $total );

		return is_array( $entries ) ? count( $entries ) : 0;
	}

	/**
	 * Get recent entries
	 *
	 * @param int $form_id Form ID
	 * @param int $limit Number of entries to get
	 *
	 * @return array Array of recent entries
	 */
	public static function get_recent_entries( $form_id, $limit = 10 ) {
		$criteria = array(
			'sorting' => array(
				'key'       => 'date_created',
				'direction' => 'DESC',
			),
		);

		return self::get_entries( $form_id, $criteria, $total );
	}

	/**
	 * Search entries
	 *
	 * @param int    $form_id Form ID
	 * @param string $search_term Search term
	 * @param array  $field_ids Field IDs to search in
	 *
	 * @return array Array of matching entries
	 */
	public static function search_entries( $form_id, $search_term, $field_ids = array() ) {
		if ( empty( $search_term ) ) {
			return self::get_entries( $form_id );
		}

		$criteria = array();

		if ( ! empty( $field_ids ) ) {
			$criteria['field_filters'] = array(
				'mode' => 'any',
			);

			foreach ( $field_ids as $field_id ) {
				$criteria['field_filters']['filters'][] = array(
					'key'   => $field_id,
					'value' => $search_term,
				);
			}
		} else {
			$criteria['search'] = $search_term;
		}

		return self::get_entries( $form_id, $criteria, $total );
	}
} 