<?php
/**
 * Field renderer for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Ristretto Apps
 * @link      https://ristrettoapps.com
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_Field_Renderer {

	/**
	 * Render field output
	 *
	 * @param array $passed_args Field arguments
	 *
	 * @return string Field output HTML
	 */
	public static function field_output( $passed_args ) {
		$defaults = array(
			'entry'        => null,
			'field'        => null,
			'form'         => null,
			'hide_empty'   => true,
			'show_labels'  => true,
			'label_markup' => '',
			'wpautop'      => false,
		);

		$args = wp_parse_args( $passed_args, $defaults );

		// Required fields
		if ( empty( $args['field'] ) || empty( $args['form'] ) ) {
			return '';
		}

		$field = $args['field'];
		$form = $args['form'];
		$entry = $args['entry'];

		// Get field value
		$value = DFE_API::field_value( $entry, $field, 'html' );

		// Handle empty values
		if ( DFE_Common::is_empty( $value ) && $args['hide_empty'] ) {
			return '';
		}

		// Get field label
		$label = DFE_API::field_label( $field, $entry );

		// Get field class
		$class = DFE_API::field_class( $field, $form, $entry );

		// Get field ID
		$field_id = DFE_API::field_html_attr_id( $field, $form, $entry );

		// Build the output with proper structure
		$output = '<div id="' . esc_attr( $field_id ) . '" class="' . esc_attr( $class ) . '">';
		
		// Add label if requested
		if ( $args['show_labels'] && ! empty( $label ) ) {
			$output .= '<div class="dfe-field-label">' . esc_html( $label ) . '</div>';
		}
		
		// Add value
		$output .= '<div class="dfe-field-value">' . $value . '</div>';
		
		$output .= '</div>';

		// Apply wpautop if requested
		if ( $args['wpautop'] ) {
			$output = wpautop( $output );
		}

		return $output;
	}

	/**
	 * Process markup with placeholders
	 *
	 * @param string $markup Markup template
	 * @param array  $data Data to replace placeholders
	 *
	 * @return string Processed markup
	 */
	private static function process_markup( $markup, $data ) {
		foreach ( $data as $key => $value ) {
			$markup = str_replace( '{{ ' . $key . ' }}', $value, $markup );
		}

		return $markup;
	}

	/**
	 * Render field value
	 *
	 * @param array  $entry Entry array
	 * @param array  $field Field array
	 * @param string $format Output format
	 *
	 * @return string Field value
	 */
	public static function field_value( $entry, $field, $format = 'html' ) {
		return DFE_API::field_value( $entry, $field, $format );
	}

	/**
	 * Render field label
	 *
	 * @param array $field Field array
	 * @param array $entry Entry array
	 *
	 * @return string Field label
	 */
	public static function field_label( $field, $entry = array() ) {
		return DFE_API::field_label( $field, $entry );
	}

	/**
	 * Render field class
	 *
	 * @param array $field Field array
	 * @param array $form Form array
	 * @param array $entry Entry array
	 *
	 * @return string Field class
	 */
	public static function field_class( $field, $form = null, $entry = null ) {
		return DFE_API::field_class( $field, $form, $entry );
	}

	/**
	 * Render container class
	 *
	 * @param string $passed_css_class Additional CSS class
	 * @param bool   $echo Whether to echo the output
	 *
	 * @return string Container class
	 */
	public static function container_class( $passed_css_class = '', $echo = true ) {
		$view_id = DFE_Helper::get_current_view_id();
		$is_single = DFE_Helper::is_single_entry();

		$passed_css_class = trim( $passed_css_class );

		$default_css_class = ! empty( $view_id ) ? sprintf( 'dfe-container dfe-container-%d', $view_id ) : 'dfe-container';

		if ( $is_single ) {
			$default_css_class .= ' dfe-container-single';
		}

		$css_class = $default_css_class;

		if ( ! empty( $passed_css_class ) ) {
			$css_class .= ' ' . DFE_Helper::sanitize_html_class( $passed_css_class );
		}

		if ( $echo ) {
			echo esc_attr( $css_class );
		}

		return $css_class;
	}

	/**
	 * Render entry link
	 *
	 * @param array  $entry Entry array
	 * @param string $anchor_text Link text
	 * @param array  $atts Link attributes
	 *
	 * @return string Entry link HTML
	 */
	public static function entry_link( $entry, $anchor_text = '', $atts = array() ) {
		if ( empty( $entry ) || ! is_array( $entry ) || ! isset( $entry['id'] ) ) {
			return '';
		}

		$href = DFE_API::entry_link( $entry );

		if ( '' === $href ) {
			return '';
		}

		return DFE_Common::get_link_html( $href, $anchor_text, $atts );
	}

	/**
	 * Render no results message
	 *
	 * @param bool $wpautop Whether to apply wpautop
	 *
	 * @return string No results message
	 */
	public static function no_results( $wpautop = true ) {
		return DFE_API::no_results( $wpautop );
	}

	/**
	 * Render back link
	 *
	 * @param string $text Link text
	 * @param array  $atts Link attributes
	 *
	 * @return string Back link HTML
	 */
	public static function back_link( $text = '', $atts = array() ) {
		if ( empty( $text ) ) {
			$text = __( '← Back to Entries', 'displayformentries' );
		}

		$href = DFE_Helper::get_back_link();

		return DFE_Common::get_link_html( $href, $text, $atts );
	}

	/**
	 * Render field files array
	 *
	 * @param mixed  $value Field value
	 * @param string $css_class CSS class
	 *
	 * @return array Array of file data
	 */
	public static function get_files_array( $value, $css_class = '' ) {
		if ( empty( $value ) ) {
			return array();
		}

		// Handle multiple files
		if ( is_array( $value ) ) {
			$files = array();
			foreach ( $value as $file ) {
				$files[] = self::format_file_data( $file, $css_class );
			}
			return $files;
		}

		return array( self::format_file_data( $value, $css_class ) );
	}

	/**
	 * Format file data
	 *
	 * @param string $file File URL
	 * @param string $css_class CSS class
	 *
	 * @return array File data
	 */
	private static function format_file_data( $file, $css_class = '' ) {
		if ( empty( $file ) ) {
			return array();
		}

		$filename = basename( $file );
		$file_extension = pathinfo( $filename, PATHINFO_EXTENSION );

		return array(
			'url'      => $file,
			'filename' => $filename,
			'extension' => $file_extension,
			'class'    => $css_class,
		);
	}

	/**
	 * Render map link
	 *
	 * @param string $address Address
	 * @param array  $atts Link attributes
	 *
	 * @return string Map link HTML
	 */
	public static function get_map_link( $address, $atts = array() ) {
		if ( empty( $address ) ) {
			return '';
		}

		$defaults = array(
			'target' => '_blank',
			'rel'    => 'noopener',
			'class'  => 'dfe-map-link',
		);

		$atts = wp_parse_args( $atts, $defaults );

		$encoded_address = urlencode( $address );
		$map_url = "https://maps.google.com/maps?q={$encoded_address}";

		return DFE_Common::get_link_html( $map_url, $address, $atts );
	}

	/**
	 * Render field with custom template
	 *
	 * @param string $template Template file path
	 * @param array  $data Data to pass to template
	 *
	 * @return string Rendered template
	 */
	public static function render_template( $template, $data = array() ) {
		if ( empty( $template ) || ! file_exists( $template ) ) {
			return '';
		}

		// Extract data to variables
		extract( $data );

		ob_start();
		include $template;
		return ob_get_clean();
	}

	/**
	 * Render field list
	 *
	 * @param array $entries Entries array
	 * @param array $fields Fields array
	 * @param array $form Form array
	 * @param array $args Display arguments
	 *
	 * @return string Rendered list HTML
	 */
	public static function render_list( $entries, $fields, $form, $args = array() ) {
		if ( empty( $entries ) ) {
			return self::no_results();
		}

		$defaults = array(
			'container_class' => 'dfe-entries-list',
			'item_class'      => 'dfe-entry-item',
			'show_labels'     => true,
			'hide_empty'      => true,
			'show_edit'       => true,
			'edit_position'   => 'after', // 'before', 'after', 'separate'
		);

		$args = wp_parse_args( $args, $defaults );

		$output = '<div class="' . esc_attr( $args['container_class'] ) . '">';

		foreach ( $entries as $entry ) {
			$output .= '<div class="' . esc_attr( $args['item_class'] ) . '">';

			// Add edit link before content if requested
			if ( $args['edit_position'] === 'before' ) {
				$output .= self::render_edit_link( $entry, $form, $args );
			}

			foreach ( $fields as $field ) {
				$field_output = self::field_output( array(
					'entry'       => $entry,
					'field'       => $field,
					'form'        => $form,
					'hide_empty'  => $args['hide_empty'],
					'show_labels' => $args['show_labels'],
				) );

				if ( ! empty( $field_output ) ) {
					$output .= $field_output;
				}
			}

			// Add edit link after content if requested
			if ( $args['edit_position'] === 'after' ) {
				$output .= self::render_edit_link( $entry, $form, $args );
			}

			$output .= '</div>';
		}

		$output .= '</div>';

		// Add edit link as separate section if requested
		if ( $args['edit_position'] === 'separate' ) {
			$output .= '<div class="dfe-edit-links-section">';
			foreach ( $entries as $entry ) {
				$output .= self::render_edit_link( $entry, $form, $args );
			}
			$output .= '</div>';
		}

		return $output;
	}

	/**
	 * Render table view
	 *
	 * @param array $entries Entries array
	 * @param array $fields Fields array
	 * @param array $form Form array
	 * @param array $args Display arguments
	 *
	 * @return string Rendered table HTML
	 */
	public static function render_table( $entries, $fields, $form, $args = array() ) {
		if ( empty( $entries ) ) {
			return self::no_results();
		}

		$defaults = array(
			'container_class' => 'dfe-table-container',
			'table_class'     => 'dfe-table',
			'show_labels'     => true,
			'hide_empty'      => true,
			'sortable'        => true,
			'responsive'      => true,
			'striped'         => true,
			'hover'           => true,
			'compact'         => false,
			'entry_links'     => false,
			'pagination'      => false,
			'search'          => false,
			'show_edit'       => true,
			'edit_column'     => true, // Add separate edit column
		);

		$args = wp_parse_args( $args, $defaults );

		// Build container classes
		$container_classes = array( $args['container_class'] );
		if ( $args['responsive'] ) {
			$container_classes[] = 'dfe-responsive';
		}

		// Build table classes
		$table_classes = array( $args['table_class'] );
		if ( $args['striped'] ) {
			$table_classes[] = 'dfe-striped';
		}
		if ( $args['hover'] ) {
			$table_classes[] = 'dfe-hover';
		}
		if ( $args['compact'] ) {
			$table_classes[] = 'dfe-compact';
		}
		if ( $args['sortable'] ) {
			$table_classes[] = 'dfe-sortable';
		}

		$output = '<div class="' . esc_attr( implode( ' ', $container_classes ) ) . '">';

		// Add search if enabled
		if ( $args['search'] ) {
			$output .= self::render_table_search( $form['id'] );
		}

		$output .= '<table class="' . esc_attr( implode( ' ', $table_classes ) ) . '">';

		// Table header
		if ( $args['show_labels'] ) {
			$output .= '<thead><tr>';
			foreach ( $fields as $field ) {
				$label = DFE_API::field_label( $field );
				$sortable = $args['sortable'] && self::is_field_sortable( $field );
				
				if ( $sortable ) {
					$output .= '<th class="dfe-sortable" data-field="' . esc_attr( $field['id'] ) . '">';
					$output .= '<span class="dfe-sort-label">' . esc_html( $label ) . '</span>';
					$output .= '<span class="dfe-sort-icon">↕</span>';
					$output .= '</th>';
				} else {
					$output .= '<th>' . esc_html( $label ) . '</th>';
				}
			}
			
			// Add edit column header if enabled
			if ( $args['show_edit'] && $args['edit_column'] ) {
				$output .= '<th class="dfe-edit-column">' . __( 'Actions', 'displayformentries' ) . '</th>';
			}
			
			$output .= '</tr></thead>';
		}

		// Table body
		$output .= '<tbody>';
		foreach ( $entries as $index => $entry ) {
			$row_classes = array( 'dfe-entry-row' );
			if ( $args['striped'] && $index % 2 === 1 ) {
				$row_classes[] = 'dfe-striped-row';
			}
			if ( $args['entry_links'] ) {
				$row_classes[] = 'dfe-clickable';
			}

			$output .= '<tr class="' . esc_attr( implode( ' ', $row_classes ) ) . '" data-entry-id="' . esc_attr( $entry['id'] ) . '">';
			
			foreach ( $fields as $field ) {
				$cell_classes = array( 'dfe-field-cell' );
				$cell_classes[] = 'dfe-field-' . $field['type'];
				$cell_classes[] = 'dfe-field-' . $field['id'];

				$value = DFE_API::field_value( $entry, $field, 'html' );
				
				if ( DFE_Common::is_empty( $value ) && $args['hide_empty'] ) {
					$value = '<span class="dfe-empty">—</span>';
				}

				$output .= '<td class="' . esc_attr( implode( ' ', $cell_classes ) ) . '">' . $value . '</td>';
			}
			
			// Add edit column if enabled
			if ( $args['show_edit'] && $args['edit_column'] ) {
				$edit_link = self::render_edit_link( $entry, $form, array( 'edit_class' => 'dfe-edit-link-inline' ) );
				$output .= '<td class="dfe-edit-cell">' . $edit_link . '</td>';
			}
			
			$output .= '</tr>';
		}
		$output .= '</tbody>';

		$output .= '</table>';

		// Add pagination if enabled
		if ( $args['pagination'] ) {
			$output .= self::render_table_pagination( $entries, $args );
		}

		$output .= '</div>';

		return $output;
	}

	/**
	 * Render table search
	 *
	 * @param int $form_id Form ID
	 *
	 * @return string Search HTML
	 */
	private static function render_table_search( $form_id ) {
		$output = '<div class="dfe-table-search">';
		$output .= '<input type="text" class="dfe-search-input" placeholder="' . esc_attr__( 'Search entries...', 'displayformentries' ) . '" data-form-id="' . esc_attr( $form_id ) . '">';
		$output .= '<button type="button" class="dfe-search-clear" style="display: none;">' . esc_html__( 'Clear', 'displayformentries' ) . '</button>';
		$output .= '</div>';
		
		return $output;
	}

	/**
	 * Render table pagination
	 *
	 * @param array $entries Entries array
	 * @param array $args Display arguments
	 *
	 * @return string Pagination HTML
	 */
	private static function render_table_pagination( $entries, $args ) {
		// This would be implemented based on your pagination logic
		$output = '<div class="dfe-table-pagination">';
		$output .= '<span class="dfe-pagination-info">' . sprintf( __( 'Showing %d entries', 'displayformentries' ), count( $entries ) ) . '</span>';
		$output .= '</div>';
		
		return $output;
	}

	/**
	 * Check if field is sortable
	 *
	 * @param array $field Field array
	 *
	 * @return bool True if sortable
	 */
	private static function is_field_sortable( $field ) {
		$non_sortable_types = array(
			'list',
			'textarea',
			'fileupload',
			'html',
			'section',
			'page',
			'captcha',
			'consent',
		);

		return ! in_array( $field['type'], $non_sortable_types, true );
	}

	/**
	 * Render single entry
	 *
	 * @param array $entry Entry array
	 * @param array $fields Fields array
	 * @param array $form Form array
	 * @param array $args Display arguments
	 *
	 * @return string Rendered single entry HTML
	 */
	public static function render_single_entry( $entry, $fields, $form, $args = array() ) {
		if ( empty( $entry ) ) {
			return self::no_results();
		}

		$defaults = array(
			'container_class' => 'dfe-single-entry',
			'field_class'     => 'dfe-field',
			'show_labels'     => true,
			'hide_empty'      => true,
			'show_edit'       => true,
			'edit_position'   => 'after', // 'before', 'after', 'separate'
		);

		$args = wp_parse_args( $args, $defaults );

		$output = '<div class="' . esc_attr( $args['container_class'] ) . '">';

		// Add edit link before content if requested
		if ( $args['edit_position'] === 'before' ) {
			$output .= self::render_edit_link( $entry, $form, $args );
		}

		foreach ( $fields as $field ) {
			$field_output = self::field_output( array(
				'entry'       => $entry,
				'field'       => $field,
				'form'        => $form,
				'hide_empty'  => $args['hide_empty'],
				'show_labels' => $args['show_labels'],
			) );

			if ( ! empty( $field_output ) ) {
				$output .= $field_output;
			}
		}

		// Add edit link after content if requested
		if ( $args['edit_position'] === 'after' ) {
			$output .= self::render_edit_link( $entry, $form, $args );
		}

		$output .= '</div>';

		// Add edit link as separate section if requested
		if ( $args['edit_position'] === 'separate' ) {
			$output .= '<div class="dfe-edit-links-section">';
			$output .= self::render_edit_link( $entry, $form, $args );
			$output .= '</div>';
		}

		return $output;
	}

	/**
	 * Render cards view
	 *
	 * @param array $entries Entries array
	 * @param array $fields Fields array
	 * @param array $form Form array
	 * @param array $args Display arguments
	 *
	 * @return string Rendered cards HTML
	 */
	public static function render_cards( $entries, $fields, $form, $args = array() ) {
		if ( empty( $entries ) ) {
			return self::no_results();
		}

		$defaults = array(
			'container_class' => 'dfe-cards',
			'card_class'      => 'dfe-card',
			'show_labels'     => true,
			'hide_empty'      => true,
			'show_edit'       => true,
			'edit_position'   => 'after', // 'before', 'after', 'separate'
			'columns'         => 3, // Number of columns for grid
			'image_field'     => '', // Field ID to use as card image
			'title_field'     => '', // Field ID to use as card title
		);

		$args = wp_parse_args( $args, $defaults );

		$output = '<div class="' . esc_attr( $args['container_class'] ) . '">';

		foreach ( $entries as $entry ) {
			$output .= '<div class="' . esc_attr( $args['card_class'] ) . '">';

			// Add edit link before content if requested
			if ( $args['edit_position'] === 'before' ) {
				$output .= self::render_edit_link( $entry, $form, $args );
			}

			// Add card image if specified
			if ( ! empty( $args['image_field'] ) ) {
				$image_field = DFE_Common::get_field( $form, $args['image_field'] );
				if ( $image_field ) {
					$image_value = DFE_API::field_value( $entry, $image_field, 'html' );
					if ( ! DFE_Common::is_empty( $image_value ) ) {
						$output .= '<div class="dfe-card-image">' . $image_value . '</div>';
					}
				}
			}

			// Add card title if specified
			if ( ! empty( $args['title_field'] ) ) {
				$title_field = DFE_Common::get_field( $form, $args['title_field'] );
				if ( $title_field ) {
					$title_value = DFE_API::field_value( $entry, $title_field, 'html' );
					if ( ! DFE_Common::is_empty( $title_value ) ) {
						$output .= '<h3 class="dfe-card-title">' . $title_value . '</h3>';
					}
				}
			}

			// Add card content
			$output .= '<div class="dfe-card-content">';
			foreach ( $fields as $field ) {
				// Skip image and title fields if they're already displayed
				if ( $field['id'] == $args['image_field'] || $field['id'] == $args['title_field'] ) {
					continue;
				}

				$field_output = self::field_output( array(
					'entry'       => $entry,
					'field'       => $field,
					'form'        => $form,
					'hide_empty'  => $args['hide_empty'],
					'show_labels' => $args['show_labels'],
				) );

				if ( ! empty( $field_output ) ) {
					$output .= $field_output;
				}
			}
			$output .= '</div>';

			// Add edit link after content if requested
			if ( $args['edit_position'] === 'after' ) {
				$output .= self::render_edit_link( $entry, $form, $args );
			}

			$output .= '</div>';
		}

		$output .= '</div>';

		// Add edit links as separate section if requested
		if ( $args['edit_position'] === 'separate' ) {
			$output .= '<div class="dfe-edit-links-section">';
			foreach ( $entries as $entry ) {
				$output .= self::render_edit_link( $entry, $form, $args );
			}
			$output .= '</div>';
		}

		return $output;
	}

	/**
	 * Render edit link for entry
	 *
	 * @param array $entry Entry array
	 * @param array $form Form array
	 * @param array $args Display arguments
	 *
	 * @return string Edit link HTML
	 */
	public static function render_edit_link( $entry, $form, $args = array() ) {
		$defaults = array(
			'show_edit'      => true,
			'edit_text'      => __( 'Edit', 'displayformentries' ),
			'edit_class'     => 'dfe-edit-link',
			'edit_position'  => 'after', // before, after, or only
		);

		$args = wp_parse_args( $args, $defaults );

		// Check if user can edit this entry
		if ( ! self::can_edit_entry( $entry, $form ) ) {
			return '';
		}

		$edit_url = self::get_entry_edit_url( $entry, $form );
		
		return sprintf(
			'<a href="%s" class="%s" title="%s" target="_blank">✏️</a>',
			esc_url( $edit_url ),
			esc_attr( $args['edit_class'] ),
			esc_attr( $args['edit_text'] )
		);
	}

	/**
	 * Check if current user can edit entry
	 *
	 * @param array $entry Entry array
	 * @param array $form Form array
	 *
	 * @return bool True if user can edit
	 */
	private static function can_edit_entry( $entry, $form ) {
		// Admin can edit any entry
		if ( current_user_can( 'gravityforms_edit_entries' ) ) {
			return true;
		}

		// Check if user is logged in
		if ( ! is_user_logged_in() ) {
			return false;
		}

		// Check if entry belongs to current user
		$current_user_id = get_current_user_id();
		
		// Check entry creator
		if ( isset( $entry['created_by'] ) && $entry['created_by'] == $current_user_id ) {
			return true;
		}

		// Check if there's a user ID field in the form
		$user_id_field = self::get_user_id_field( $form );
		if ( $user_id_field && isset( $entry[ $user_id_field ] ) ) {
			$entry_user_id = intval( $entry[ $user_id_field ] );
			if ( $entry_user_id === $current_user_id ) {
				return true;
			}
		}

		// Check for email field match (if user email matches entry email)
		$email_field = self::get_email_field( $form );
		if ( $email_field && isset( $entry[ $email_field ] ) ) {
			$user = wp_get_current_user();
			if ( $user && $user->user_email === $entry[ $email_field ] ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Get entry edit URL
	 *
	 * @param array $entry Entry array
	 * @param array $form Form array
	 *
	 * @return string Edit URL or empty string
	 */
	private static function get_entry_edit_url( $entry, $form ) {
		// Admin edit URL
		if ( current_user_can( 'gravityforms_edit_entries' ) ) {
			return admin_url( 'admin.php?page=gf_entries&view=entry&id=' . $form['id'] . '&lid=' . $entry['id'] );
		}

		// Frontend edit URL (if using Gravity Forms User Registration or similar)
		// This would need to be customized based on your setup
		$frontend_edit_url = apply_filters( 'dfe_entry_edit_url', '', $entry, $form );
		if ( ! empty( $frontend_edit_url ) ) {
			return $frontend_edit_url;
		}

		// Default to admin URL for now
		return admin_url( 'admin.php?page=gf_entries&view=entry&id=' . $form['id'] . '&lid=' . $entry['id'] );
	}

	/**
	 * Get user ID field from form
	 *
	 * @param array $form Form array
	 *
	 * @return int|false Field ID or false if not found
	 */
	private static function get_user_id_field( $form ) {
		if ( empty( $form['fields'] ) ) {
			return false;
		}

		foreach ( $form['fields'] as $field ) {
			if ( $field['type'] === 'hidden' && isset( $field['inputName'] ) && $field['inputName'] === 'user_id' ) {
				return $field['id'];
			}
		}

		return false;
	}

	/**
	 * Get email field from form
	 *
	 * @param array $form Form array
	 *
	 * @return int|false Field ID or false if not found
	 */
	private static function get_email_field( $form ) {
		if ( empty( $form['fields'] ) ) {
			return false;
		}

		foreach ( $form['fields'] as $field ) {
			if ( $field['type'] === 'email' ) {
				return $field['id'];
			}
		}

		return false;
	}

	/**
	 * Render linked list view
	 *
	 * @param array $entries Entries array
	 * @param array $fields Fields array
	 * @param array $form Form array
	 * @param array $args Display arguments
	 *
	 * @return string Rendered linked list HTML
	 */
	public static function render_linked_list( $entries, $fields, $form, $args = array() ) {
		if ( empty( $entries ) ) {
			return self::no_results();
		}

		$defaults = array(
			'container_class' => 'dfe-linked-list-container',
			'table_class'     => 'dfe-linked-list-table',
			'row_class'       => 'dfe-entry-row',
			'cell_class'      => 'dfe-field-cell',
			'show_labels'     => true,
			'hide_empty'      => true,
			'link_field'      => '',
			'show_edit'       => true,
			'edit_text'       => __( 'Edit', 'displayformentries' ),
			'edit_class'      => 'dfe-edit-link',
		);

		$args = wp_parse_args( $args, $defaults );

		$output = '<div class="' . esc_attr( $args['container_class'] ) . '">';
		$output .= '<table class="' . esc_attr( $args['table_class'] ) . '">';

		// Table header
		if ( $args['show_labels'] ) {
			$output .= '<thead><tr>';
			foreach ( $fields as $field ) {
				$field_label = DFE_API::field_label( $field, $form );
				$output .= '<th class="' . esc_attr( $args['cell_class'] ) . ' dfe-field-header dfe-field-' . esc_attr( $field['id'] ) . '">';
				$output .= esc_html( $field_label );
				$output .= '</th>';
			}
			
			// Add edit column header if showing edit links
			if ( $args['show_edit'] ) {
				$output .= '<th class="' . esc_attr( $args['cell_class'] ) . ' dfe-field-header dfe-edit-header">';
				$output .= __( 'Actions', 'displayformentries' );
				$output .= '</th>';
			}
			
			$output .= '</tr></thead>';
		}

		// Table body
		$output .= '<tbody>';
		foreach ( $entries as $entry ) {
			$output .= '<tr class="' . esc_attr( $args['row_class'] ) . ' dfe-entry-' . esc_attr( $entry['id'] ) . '">';
			
			foreach ( $fields as $field ) {
				$field_value = DFE_API::field_value( $entry, $field, $form );
				$field_class = DFE_API::field_class( $field, $form );
				$field_id = DFE_API::field_html_attr_id( $field, $entry );
				
				// Skip empty fields if hide_empty is enabled
				if ( $args['hide_empty'] && empty( $field_value ) ) {
					$output .= '<td class="' . esc_attr( $args['cell_class'] ) . ' ' . esc_attr( $field_class ) . ' dfe-empty-field"></td>';
					continue;
				}
				
				$output .= '<td class="' . esc_attr( $args['cell_class'] ) . ' ' . esc_attr( $field_class ) . '">';
				
				// Check if this field should be linked
				if ( $args['link_field'] == $field['id'] && ! empty( $field_value ) ) {
					$detail_url = add_query_arg( array(
						'dfe_entry' => $entry['id'],
						'dfe_view'  => 'detail'
					), get_permalink() );
					
					$output .= '<a href="' . esc_url( $detail_url ) . '" class="dfe-entry-detail-link">';
					$output .= esc_html( $field_value );
					$output .= '</a>';
				} else {
					$output .= esc_html( $field_value );
				}
				
				$output .= '</td>';
			}
			
			// Add edit link column if showing edit links
			if ( $args['show_edit'] ) {
				$output .= '<td class="' . esc_attr( $args['cell_class'] ) . ' dfe-edit-cell">';
				$output .= self::render_edit_link( $entry, $form, $args );
				$output .= '</td>';
			}
			
			$output .= '</tr>';
		}
		$output .= '</tbody>';
		$output .= '</table>';
		$output .= '</div>';

		return $output;
	}

	/**
	 * Get entry detail URL (frontend URL)
	 *
	 * @param array $entry Entry array
	 * @param array $form Form array
	 *
	 * @return string Entry detail URL
	 */
	private static function get_entry_detail_url( $entry, $form ) {
		// Add entry ID to current URL for frontend detail view
		$current_url = remove_query_arg( array( 'entry_id', 'form_id' ) );
		$detail_url = add_query_arg( array(
			'entry_id' => $entry['id'],
			'form_id'  => $form['id'],
		), $current_url );

		return $detail_url;
	}
} 