<?php
/**
 * Shortcodes for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Ristretto Apps
 * @link      https://ristrettoapps.com
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_Shortcodes {

	/**
	 * Initialize shortcodes
	 */
	public static function init() {
		add_shortcode( 'dfe_entries', array( __CLASS__, 'entries_shortcode' ) );
		add_shortcode( 'dfe_entry', array( __CLASS__, 'entry_shortcode' ) );
		add_shortcode( 'dfe_field', array( __CLASS__, 'field_shortcode' ) );
		add_shortcode( 'dfe_form', array( __CLASS__, 'form_shortcode' ) );
	}

	/**
	 * Render entries block
	 */
	public static function render_entries_block( $attributes ) {
		// Handle fields attribute - convert array to comma-separated string
		$fields = '';
		if ( isset( $attributes['fields'] ) ) {
			if ( is_array( $attributes['fields'] ) ) {
				$fields = implode( ',', $attributes['fields'] );
			} else {
				$fields = $attributes['fields'];
			}
		}
		
		// Handle link_field attribute - convert array to comma-separated string
		$link_field = '';
		if ( isset( $attributes['linkField'] ) ) {
			if ( is_array( $attributes['linkField'] ) ) {
				$link_field = implode( ',', $attributes['linkField'] );
			} else {
				$link_field = $attributes['linkField'];
			}
		}
		// Default to field ID 1 if empty
		if ( empty( $link_field ) ) {
			$link_field = '1';
		}
		
		$atts = array(
			'form_id'         => isset( $attributes['formId'] ) ? $attributes['formId'] : 0,
			'template'        => isset( $attributes['template'] ) ? $attributes['template'] : 'list',
			'limit'           => isset( $attributes['limit'] ) ? $attributes['limit'] : 10,
			'fields'          => $fields,
			'hide_empty'      => isset( $attributes['hideEmpty'] ) ? $attributes['hideEmpty'] : true,
			'show_labels'     => isset( $attributes['showLabels'] ) ? $attributes['showLabels'] : true,
			'link_field'      => $link_field,
			'show_edit'       => isset( $attributes['showEdit'] ) ? $attributes['showEdit'] : true,
			'edit_text'       => isset( $attributes['editText'] ) ? $attributes['editText'] : 'Edit',
		);

		return self::entries_shortcode( $atts );
	}

	/**
	 * Render entry block
	 */
	public static function render_entry_block( $attributes ) {
		// Handle fields attribute - convert array to comma-separated string
		$fields = '';
		if ( isset( $attributes['fields'] ) ) {
			if ( is_array( $attributes['fields'] ) ) {
				$fields = implode( ',', $attributes['fields'] );
			} else {
				$fields = $attributes['fields'];
			}
		}
		
		$atts = array(
			'entry_id'        => isset( $attributes['entryId'] ) ? $attributes['entryId'] : 0,
			'form_id'         => isset( $attributes['formId'] ) ? $attributes['formId'] : 0,
			'fields'          => $fields,
			'hide_empty'      => isset( $attributes['hideEmpty'] ) ? $attributes['hideEmpty'] : true,
			'show_labels'     => isset( $attributes['showLabels'] ) ? $attributes['showLabels'] : true,
		);

		return self::entry_shortcode( $atts );
	}

	/**
	 * Render form block
	 */
	public static function render_form_block( $attributes ) {
		$atts = array(
			'form_id'         => isset( $attributes['formId'] ) ? $attributes['formId'] : 0,
			'title'           => isset( $attributes['title'] ) ? $attributes['title'] : true,
			'description'     => isset( $attributes['description'] ) ? $attributes['description'] : true,
		);

		return self::form_shortcode( $atts );
	}

	/**
	 * Entries shortcode
	 *
	 * @param array $atts Shortcode attributes
	 *
	 * @return string Shortcode output
	 */
	public static function entries_shortcode( $atts ) {
		$defaults = array(
			'form_id'         => 0,
			'template'        => 'list',
			'limit'           => 10,
			'offset'          => 0,
			'search'          => '',
			'sort'            => 'date_created',
			'sort_direction'  => 'DESC',
			'fields'          => '',
			'hide_empty'      => 'true',
			'show_labels'     => 'true',
			'link_field'      => '1', //default to 1 if not set
			'show_edit'       => 'true',
			'edit_text'       => __( 'Edit', 'displayformentries' ),
		);

		$atts = shortcode_atts( $defaults, $atts, 'dfe_entries' );
		
		// Add this line to handle empty link_field
		if ( empty( $atts['link_field'] ) ) {
			$atts['link_field'] = 1;
		}
		


		// Convert string booleans to actual booleans
		$atts['hide_empty'] = filter_var( $atts['hide_empty'], FILTER_VALIDATE_BOOLEAN );
		$atts['show_labels'] = filter_var( $atts['show_labels'], FILTER_VALIDATE_BOOLEAN );
		$atts['show_edit'] = filter_var( $atts['show_edit'], FILTER_VALIDATE_BOOLEAN );

		// Check if we're viewing a single entry
		$entry_id = isset( $_GET['dfe_entry'] ) ? intval( $_GET['dfe_entry'] ) : 0;
		$view_type = isset( $_GET['dfe_view'] ) ? sanitize_text_field( $_GET['dfe_view'] ) : '';
		
		if ( $entry_id && $view_type === 'detail' ) {
			// Show single entry detail view
			return self::render_single_entry_detail( $entry_id, $atts );
		}

		// Get form
		$form = DFE_Connector::get_form( $atts['form_id'] );
		if ( ! $form ) {
			return '<p>' . __( 'Form not found.', 'displayformentries' ) . '</p>';
		}

		// Get fields
		$fields = DFE_Connector::get_form_fields( $form );
		
		if ( ! empty( $atts['fields'] ) ) {
			// Handle both array (from blocks) and string (from shortcodes) formats
			if ( is_array( $atts['fields'] ) ) {
				$field_ids = $atts['fields'];
			} else {
				$field_ids = array_map( 'trim', explode( ',', $atts['fields'] ) );
			}
			
			// Process field IDs to handle subfields (like 1.3, 1.6)
			$processed_fields = self::process_field_selection( $fields, $field_ids );
			$fields = $processed_fields['fields'];
			
			// Store subfield mapping for later use in field value rendering
			$GLOBALS['dfe_selected_subfields'] = $processed_fields['subfield_map'];
		}

		// Get entries
		$entries = DFE_Connector::get_entries( $form, $atts );

		// Render based on template
		$output = '';
		
		switch ( $atts['template'] ) {
			case 'table':
				$output = DFE_Field_Renderer::render_table( $entries, $fields, $form, $atts );
				break;
			case 'cards':
				$output = DFE_Field_Renderer::render_cards( $entries, $fields, $form, $atts );
				break;
			case 'linked-list':
				$output = DFE_Field_Renderer::render_linked_list( $entries, $fields, $form, $atts );
				break;
			case 'list':
			default:
				$output = DFE_Field_Renderer::render_list( $entries, $fields, $form, $atts );
				break;
		}
		
		// Clean up global variable after rendering
		if ( isset( $GLOBALS['dfe_selected_subfields'] ) ) {
			unset( $GLOBALS['dfe_selected_subfields'] );
		}
		
		return $output;
	}

	/**
	 * Entry shortcode
	 *
	 * @param array $atts Shortcode attributes
	 *
	 * @return string Shortcode output
	 */
	public static function entry_shortcode( $atts ) {
		$defaults = array(
			'entry_id'        => 0,
			'form_id'         => 0,
			'fields'          => '',
			'hide_empty'      => 'true',
			'show_labels'     => 'true',
			'container_class' => '',
		);

		$atts = shortcode_atts( $defaults, $atts, 'dfe_entry' );

		// Get entry ID
		$entry_id = intval( $atts['entry_id'] );
		if ( $entry_id <= 0 ) {
			$entry_id = DFE_Helper::get_current_entry_id();
		}

		if ( ! $entry_id ) {
			return '<p class="dfe-error">' . __( 'No entry specified.', 'displayformentries' ) . '</p>';
		}

		// Get entry
		$entry = DFE_Common::get_entry( $entry_id, true );
		if ( ! $entry ) {
			return '<p class="dfe-error">' . __( 'Entry not found.', 'displayformentries' ) . '</p>';
		}

		// Get form
		$form_id = intval( $atts['form_id'] );
		if ( $form_id <= 0 ) {
			$form_id = $entry['form_id'];
		}

		$form = DFE_Common::get_form( $form_id );
		if ( ! $form ) {
			return '<p class="dfe-error">' . __( 'Form not found.', 'displayformentries' ) . '</p>';
		}

		// Get fields to display
		$field_ids = array();
		if ( ! empty( $atts['fields'] ) ) {
			// Handle both array (from blocks) and string (from shortcodes) formats
			if ( is_array( $atts['fields'] ) ) {
				$field_ids = $atts['fields'];
			} else {
				$field_ids = array_map( 'trim', explode( ',', $atts['fields'] ) );
			}
			
			// Process field IDs to handle subfields (like 1.3, 1.6) for entry shortcode too
			$all_fields = DFE_Common::get_form_fields( $form );
			$processed_fields = self::process_field_selection( $all_fields, $field_ids );
			
			// Store subfield mapping for later use in field value rendering
			$GLOBALS['dfe_selected_subfields'] = $processed_fields['subfield_map'];
			
			// Use processed field IDs for filtering
			$field_ids = array();
			foreach ( $processed_fields['fields'] as $field ) {
				$field_ids[] = $field['id'];
			}
		}

		// If no specific fields, get all fields
		if ( empty( $field_ids ) ) {
			$all_fields = DFE_Common::get_form_fields( $form );
			$field_ids = wp_list_pluck( $all_fields, 'id' );
		}

		// Get fields data
		$fields = array();
		foreach ( $field_ids as $field_id ) {
			$field = DFE_Common::get_field( $form, $field_id );
			if ( $field ) {
				$fields[] = $field;
			}
		}

		$output = self::render_single_entry( $entry, $fields, $form, $atts );
		
		// Clean up global variable after rendering
		if ( isset( $GLOBALS['dfe_selected_subfields'] ) ) {
			unset( $GLOBALS['dfe_selected_subfields'] );
		}
		
		return $output;
	}

	/**
	 * Field shortcode
	 *
	 * @param array $atts Shortcode attributes
	 *
	 * @return string Shortcode output
	 */
	public static function field_shortcode( $atts ) {
		$defaults = array(
			'field_id'        => '',
			'entry_id'        => 0,
			'form_id'         => 0,
			'format'          => 'html',
			'label'           => 'false',
			'container_class' => '',
		);

		$atts = shortcode_atts( $defaults, $atts, 'dfe_field' );

		// Get entry ID
		$entry_id = intval( $atts['entry_id'] );
		if ( $entry_id <= 0 ) {
			$entry_id = DFE_Helper::get_current_entry_id();
		}

		if ( ! $entry_id ) {
			return '<p class="dfe-error">' . __( 'No entry specified.', 'displayformentries' ) . '</p>';
		}

		// Get entry
		$entry = DFE_Common::get_entry( $entry_id, true );
		if ( ! $entry ) {
			return '<p class="dfe-error">' . __( 'Entry not found.', 'displayformentries' ) . '</p>';
		}

		// Get form
		$form_id = intval( $atts['form_id'] );
		if ( $form_id <= 0 ) {
			$form_id = $entry['form_id'];
		}

		$form = DFE_Common::get_form( $form_id );
		if ( ! $form ) {
			return '<p class="dfe-error">' . __( 'Form not found.', 'displayformentries' ) . '</p>';
		}

		// Get field
		$field_id = $atts['field_id'];
		if ( empty( $field_id ) ) {
			return '<p class="dfe-error">' . __( 'No field specified.', 'displayformentries' ) . '</p>';
		}

		$field = DFE_Common::get_field( $form, $field_id );
		if ( ! $field ) {
			return '<p class="dfe-error">' . __( 'Field not found.', 'displayformentries' ) . '</p>';
		}

		// Get field value
		$value = DFE_API::field_value( $entry, $field, $atts['format'] );

		// Build output
		$output = '';

		if ( $atts['label'] === 'true' ) {
			$label = DFE_API::field_label( $field, $entry );
			$output .= '<span class="dfe-field-label">' . esc_html( $label ) . ':</span> ';
		}

		$output .= '<span class="dfe-field-value">' . $value . '</span>';

		// Add container class
		if ( ! empty( $atts['container_class'] ) ) {
			$output = '<div class="' . esc_attr( $atts['container_class'] ) . '">' . $output . '</div>';
		}

		return $output;
	}

	/**
	 * Form shortcode
	 *
	 * @param array $atts Shortcode attributes
	 *
	 * @return string Shortcode output
	 */
	public static function form_shortcode( $atts ) {
		$defaults = array(
			'form_id' => 0,
			'title'   => 'true',
		);

		$atts = shortcode_atts( $defaults, $atts, 'dfe_form' );

		// Validate form ID
		$form_id = intval( $atts['form_id'] );
		if ( $form_id <= 0 ) {
			return '<p class="dfe-error">' . __( 'Invalid form ID.', 'displayformentries' ) . '</p>';
		}

		// Get form
		$form = DFE_Common::get_form( $form_id );
		if ( ! $form ) {
			return '<p class="dfe-error">' . __( 'Form not found.', 'displayformentries' ) . '</p>';
		}

		// Build output
		$output = '<div class="dfe-form-info">';

		if ( $atts['title'] === 'true' ) {
			$output .= '<h3 class="dfe-form-title">' . esc_html( $form['title'] ) . '</h3>';
		}

		if ( ! empty( $form['description'] ) ) {
			$output .= '<div class="dfe-form-description">' . wpautop( esc_html( $form['description'] ) ) . '</div>';
		}

		$output .= '</div>';

		return $output;
	}

	/**
	 * Render entries
	 *
	 * @param array $entries Entries array
	 * @param array $fields Fields array
	 * @param array $form Form array
	 * @param array $atts Shortcode attributes
	 *
	 * @return string Rendered entries HTML
	 */
	private static function render_entries( $entries, $fields, $form, $atts ) {
		if ( empty( $entries ) ) {
			return '<p class="dfe-no-entries">' . __( 'No entries found.', 'displayformentries' ) . '</p>';
		}

		$template = $atts['template'];
		$container_class = $atts['container_class'];
		$show_labels = filter_var( $atts['show_labels'], FILTER_VALIDATE_BOOLEAN );
		$hide_empty = filter_var( $atts['hide_empty'], FILTER_VALIDATE_BOOLEAN );

		// Build display arguments
		$display_args = array(
			'show_labels' => $show_labels,
			'hide_empty'  => $hide_empty,
		);

		// Add template-specific arguments
		switch ( $template ) {
			case 'table':
				$display_args = array_merge( $display_args, array(
					'sortable'    => true,
					'responsive'  => true,
					'striped'     => true,
					'hover'       => true,
					'entry_links' => false,
					'pagination'  => false,
					'search'      => false,
					'edit_column' => filter_var( $atts['edit_column'], FILTER_VALIDATE_BOOLEAN ),
				) );
				return DFE_Field_Renderer::render_table( $entries, $fields, $form, $display_args );

			case 'list':
				return DFE_Field_Renderer::render_list( $entries, $fields, $form, $display_args );

			case 'card':
				$display_args = array_merge( $display_args, array(
					'columns'     => 3,
					'image_field' => '',
					'title_field' => '',
				) );
				return DFE_Field_Renderer::render_cards( $entries, $fields, $form, $display_args );

			case 'linked-list':
				$display_args = array_merge( $display_args, array(
					'link_field'  => $atts['link_field'],
					'link_text'   => $atts['link_text'],
					'responsive'  => true,
					'striped'     => true,
					'hover'       => true,
				) );
				return DFE_Field_Renderer::render_linked_list( $entries, $fields, $form, $display_args );

			default:
				return DFE_Field_Renderer::render_list( $entries, $fields, $form, $display_args );
		}
	}

	/**
	 * Render single entry
	 *
	 * @param array $entry Entry array
	 * @param array $fields Fields array
	 * @param array $form Form array
	 * @param array $atts Shortcode attributes
	 *
	 * @return string Rendered single entry HTML
	 */
	private static function render_single_entry( $entry, $fields, $form, $atts ) {
		if ( empty( $entry ) ) {
			return DFE_Field_Renderer::no_results();
		}

		$template_settings = DFE_Template::get_template_settings( 'single' );

		// Merge shortcode attributes with template settings
		$settings = wp_parse_args( $atts, $template_settings );

		// Validate settings
		$settings = DFE_Template::validate_template_settings( $settings );

		// Enqueue template CSS
		DFE_Template::enqueue_template_css( 'single' );

		// Add back link if not in admin
		$output = '';
		if ( ! is_admin() ) {
			$output .= DFE_Field_Renderer::back_link();
		}

		$output .= DFE_Field_Renderer::render_single_entry( $entry, $fields, $form, $settings );

		return $output;
	}

	/**
	 * Render single entry detail view
	 *
	 * @param int $entry_id Entry ID
	 * @param array $atts Shortcode attributes
	 *
	 * @return string Entry detail HTML
	 */
	private static function render_single_entry_detail( $entry_id, $atts ) {
		// Get entry
		$entry = DFE_Connector::get_entry( $entry_id );
		if ( ! $entry ) {
			return '<p>' . __( 'Entry not found.', 'displayformentries' ) . '</p>';
		}

		// Get form
		$form = DFE_Connector::get_form( $entry['form_id'] );
		if ( ! $form ) {
			return '<p>' . __( 'Form not found.', 'displayformentries' ) . '</p>';
		}

		// Get all fields for the form
		$fields = DFE_Connector::get_form_fields( $form );

		// Build back link URL (remove the entry detail parameters)
		$back_url = remove_query_arg( array( 'dfe_entry', 'dfe_view' ), get_permalink() );
		
		$output = '<div class="dfe-entry-detail-container">';
		
		// Back link
		$output .= '<div class="dfe-back-link-container">';
		$output .= '<a href="' . esc_url( $back_url ) . '" class="dfe-back-link">';
		$output .= '← ' . __( 'Back to List', 'displayformentries' );
		$output .= '</a>';
		$output .= '</div>';
		
		// Entry title
		$output .= '<h2 class="dfe-entry-title">';
		$output .= sprintf( __( 'Entry #%d', 'displayformentries' ), $entry['id'] );
		$output .= '</h2>';
		
		// Entry details
		$output .= '<div class="dfe-entry-detail-content">';
		foreach ( $fields as $field ) {
			$field_output = DFE_Field_Renderer::field_output( array(
				'entry'       => $entry,
				'field'       => $field,
				'form'        => $form,
				'hide_empty'  => $atts['hide_empty'],
				'show_labels' => $atts['show_labels'],
			) );
			
			if ( ! empty( $field_output ) ) {
				$output .= $field_output;
			}
		}
		$output .= '</div>';
		
		// Edit link if user can edit
		if ( $atts['show_edit'] ) {
			$edit_link = DFE_Field_Renderer::render_edit_link( $entry, $form, $atts );
			if ( ! empty( $edit_link ) ) {
				$output .= '<div class="dfe-entry-detail-actions">';
				$output .= $edit_link;
				$output .= '</div>';
			}
		}
		
		$output .= '</div>';

		return $output;
	}

	/**
	 * Get shortcode help
	 *
	 * @return string Help text
	 */
	public static function get_shortcode_help() {
		$help = '<h3>' . __( 'Display Form Entries Shortcodes', 'displayformentries' ) . '</h3>';

		$help .= '<h4>' . __( '[dfe_entries]', 'displayformentries' ) . '</h4>';
		$help .= '<p>' . __( 'Display multiple entries from a Gravity Form.', 'displayformentries' ) . '</p>';
		$help .= '<ul>';
		$help .= '<li><strong>form_id</strong> - ' . __( 'The ID of the Gravity Form', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>template</strong> - ' . __( 'Template to use (list, table, card)', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>limit</strong> - ' . __( 'Number of entries to display', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>fields</strong> - ' . __( 'Comma-separated list of field IDs to display', 'displayformentries' ) . '</li>';
		$help .= '</ul>';

		$help .= '<h4>' . __( '[dfe_entry]', 'displayformentries' ) . '</h4>';
		$help .= '<p>' . __( 'Display a single entry.', 'displayformentries' ) . '</p>';
		$help .= '<ul>';
		$help .= '<li><strong>entry_id</strong> - ' . __( 'The ID of the entry to display', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>form_id</strong> - ' . __( 'The ID of the Gravity Form', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>fields</strong> - ' . __( 'Comma-separated list of field IDs to display', 'displayformentries' ) . '</li>';
		$help .= '</ul>';

		$help .= '<h4>' . __( '[dfe_field]', 'displayformentries' ) . '</h4>';
		$help .= '<p>' . __( 'Display a single field value.', 'displayformentries' ) . '</p>';
		$help .= '<ul>';
		$help .= '<li><strong>field_id</strong> - ' . __( 'The ID of the field to display', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>entry_id</strong> - ' . __( 'The ID of the entry', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>format</strong> - ' . __( 'Output format (html, text)', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>label</strong> - ' . __( 'Show field label (true/false)', 'displayformentries' ) . '</li>';
		$help .= '</ul>';

		$help .= '<h4>' . __( '[dfe_form]', 'displayformentries' ) . '</h4>';
		$help .= '<p>' . __( 'Display form information.', 'displayformentries' ) . '</p>';
		$help .= '<ul>';
		$help .= '<li><strong>form_id</strong> - ' . __( 'The ID of the Gravity Form', 'displayformentries' ) . '</li>';
		$help .= '<li><strong>title</strong> - ' . __( 'Show form title (true/false)', 'displayformentries' ) . '</li>';
		$help .= '</ul>';

		return $help;
	}

	/**
	 * Process field selection to handle subfields
	 *
	 * @param array $all_fields All form fields
	 * @param array $selected_field_ids Selected field IDs (may include subfield IDs)
	 *
	 * @return array Array with 'fields' and 'subfield_map'
	 */
	private static function process_field_selection( $all_fields, $selected_field_ids ) {
		$included_fields = array();
		$subfield_map = array();
		
		// Define entry meta fields
		$entry_meta_fields = array( 'id', 'date_created', 'date_updated', 'is_starred', 'is_read', 'ip', 'source_url', 'user_agent', 'payment_date', 'payment_amount', 'payment_status', 'transaction_id', 'created_by', 'post_id' );
		
		foreach ( $selected_field_ids as $field_id ) {
			// Check if this is an entry meta field
			if ( in_array( $field_id, $entry_meta_fields ) ) {
				// Create a virtual field for entry meta
				$meta_field_labels = array(
					'id' => 'Entry ID',
					'date_created' => 'Entry Date', 
					'date_updated' => 'Date Updated',
					'is_starred' => 'Is Starred',
					'is_read' => 'Is Read',
					'ip' => 'User IP',
					'source_url' => 'Source URL',
					'user_agent' => 'User Agent',
					'payment_date' => 'Payment Date',
					'payment_amount' => 'Payment Amount', 
					'payment_status' => 'Payment Status',
					'transaction_id' => 'Transaction ID',
					'created_by' => 'Created By (User ID)',
					'post_id' => 'Post ID'
				);
				
				$virtual_field = array(
					'id' => $field_id,
					'label' => $meta_field_labels[ $field_id ],
					'type' => 'entry_meta'
				);
				
				$field_key = 'meta_' . $field_id;
				$included_fields[ $field_key ] = $virtual_field;
			}
			// Check if this is a subfield (contains a dot)
			elseif ( strpos( $field_id, '.' ) !== false ) {
				// Extract parent field ID (everything before the first dot)
				$parent_field_id = explode( '.', $field_id )[0];
				
				// Find the parent field
				$parent_field = null;
				foreach ( $all_fields as $field ) {
					if ( $field['id'] == $parent_field_id ) {
						$parent_field = $field;
						break;
					}
				}
				
				if ( $parent_field ) {
					// Add parent field to included fields (if not already added)
					$field_key = 'field_' . $parent_field_id;
					if ( ! isset( $included_fields[ $field_key ] ) ) {
						$included_fields[ $field_key ] = $parent_field;
					}
					
					// Track which subfields are requested for this parent
					if ( ! isset( $subfield_map[ $parent_field_id ] ) ) {
						$subfield_map[ $parent_field_id ] = array();
					}
					$subfield_map[ $parent_field_id ][] = $field_id;
				}
			} else {
				// Regular field ID - find and include the field
				foreach ( $all_fields as $field ) {
					if ( $field['id'] == $field_id ) {
						$field_key = 'field_' . $field_id;
						$included_fields[ $field_key ] = $field;
						break;
					}
				}
			}
		}
		
		return array(
			'fields' => array_values( $included_fields ),
			'subfield_map' => $subfield_map
		);
	}
} 