<?php
/**
 * Template management for Display Form Entries plugin
 *
 * @package   DisplayFormEntries
 * @license   GPL2+
 * @author    Ristretto Apps
 * @link      https://ristrettoapps.com
 *
 * @since 1.0.0
 */

/** If this file is called directly, abort. */
if ( ! defined( 'ABSPATH' ) ) {
	die;
}

class DFE_Template {

	/**
	 * Default templates
	 *
	 * @var array
	 */
	private static $default_templates = array();

	/**
	 * Initialize templates
	 */
	public static function init() {
		self::register_default_templates();
	}

	/**
	 * Register default templates
	 */
	private static function register_default_templates() {
		self::$default_templates = array(
			'list' => array(
				'name'        => __( 'List View', 'displayformentries' ),
				'description' => __( 'Display entries in a list format', 'displayformentries' ),
				'type'        => 'list',
				'file'        => 'list.php',
			),
			'table' => array(
				'name'        => __( 'Table View', 'displayformentries' ),
				'description' => __( 'Display entries in a table format', 'displayformentries' ),
				'type'        => 'table',
				'file'        => 'table.php',
			),
			'card' => array(
				'name'        => __( 'Card View', 'displayformentries' ),
				'description' => __( 'Display entries in a card format', 'displayformentries' ),
				'type'        => 'card',
				'file'        => 'card.php',
			),
			'single' => array(
				'name'        => __( 'Single Entry', 'displayformentries' ),
				'description' => __( 'Display a single entry in detail', 'displayformentries' ),
				'type'        => 'single',
				'file'        => 'single.php',
			),
		);
	}

	/**
	 * Get all templates
	 *
	 * @return array Array of templates
	 */
	public static function get_templates() {
		return self::$default_templates;
	}

	/**
	 * Get template by ID
	 *
	 * @param string $template_id Template ID
	 *
	 * @return array|false Template array or false if not found
	 */
	public static function get_template( $template_id ) {
		if ( isset( self::$default_templates[ $template_id ] ) ) {
			return self::$default_templates[ $template_id ];
		}

		return false;
	}

	/**
	 * Get template file path
	 *
	 * @param string $template_id Template ID
	 *
	 * @return string|false Template file path or false if not found
	 */
	public static function get_template_file( $template_id ) {
		$template = self::get_template( $template_id );

		if ( ! $template ) {
			return false;
		}

		$file = $template['file'];

		// Check for custom template in theme
		$theme_template = get_stylesheet_directory() . '/displayformentries/templates/' . $file;

		if ( file_exists( $theme_template ) ) {
			return $theme_template;
		}

		// Check for custom template in parent theme
		$parent_template = get_template_directory() . '/displayformentries/templates/' . $file;

		if ( file_exists( $parent_template ) ) {
			return $parent_template;
		}

		// Use default template
		$default_template = DFE_DIR . 'templates/' . $file;

		if ( file_exists( $default_template ) ) {
			return $default_template;
		}

		return false;
	}

	/**
	 * Render template
	 *
	 * @param string $template_id Template ID
	 * @param array  $data Data to pass to template
	 *
	 * @return string Rendered template
	 */
	public static function render( $template_id, $data = array() ) {
		$template_file = self::get_template_file( $template_id );

		if ( ! $template_file ) {
			return '';
		}

		// Extract data to variables
		extract( $data );

		ob_start();
		include $template_file;
		return ob_get_clean();
	}

	/**
	 * Get template options for select fields
	 *
	 * @return array Array of template options
	 */
	public static function get_template_options() {
		$templates = self::get_templates();
		$options = array();

		foreach ( $templates as $id => $template ) {
			$options[ $id ] = $template['name'];
		}

		return $options;
	}

	/**
	 * Get template CSS file
	 *
	 * @param string $template_id Template ID
	 *
	 * @return string|false CSS file path or false if not found
	 */
	public static function get_template_css( $template_id ) {
		$template = self::get_template( $template_id );

		if ( ! $template ) {
			return false;
		}

		$css_file = str_replace( '.php', '.css', $template['file'] );

		// Check for custom CSS in theme
		$theme_css = get_stylesheet_directory() . '/displayformentries/css/' . $css_file;

		if ( file_exists( $theme_css ) ) {
			return get_stylesheet_directory_uri() . '/displayformentries/css/' . $css_file;
		}

		// Check for custom CSS in parent theme
		$parent_css = get_template_directory() . '/displayformentries/css/' . $css_file;

		if ( file_exists( $parent_css ) ) {
			return get_template_directory_uri() . '/displayformentries/css/' . $css_file;
		}

		// Use default CSS
		$default_css = DFE_DIR . 'assets/css/' . $css_file;

		if ( file_exists( $default_css ) ) {
			return DFE_URL . 'assets/css/' . $css_file;
		}

		return false;
	}

	/**
	 * Enqueue template CSS
	 *
	 * @param string $template_id Template ID
	 */
	public static function enqueue_template_css( $template_id ) {
		$css_url = self::get_template_css( $template_id );

		if ( $css_url ) {
			wp_enqueue_style( 'dfe-template-' . $template_id, $css_url, array(), DFE_PLUGIN_VERSION );
		}
	}

	/**
	 * Get template settings
	 *
	 * @param string $template_id Template ID
	 *
	 * @return array Template settings
	 */
	public static function get_template_settings( $template_id ) {
		$template = self::get_template( $template_id );

		if ( ! $template ) {
			return array();
		}

		$default_settings = array(
			'container_class' => 'dfe-container',
			'item_class'      => 'dfe-item',
			'show_labels'     => true,
			'hide_empty'      => true,
			'limit'           => 10,
			'pagination'      => true,
			'search'          => false,
			'sort'            => 'date_created',
			'sort_direction'  => 'DESC',
		);

		// Template-specific settings
		switch ( $template_id ) {
			case 'list':
				$default_settings['container_class'] = 'dfe-list';
				$default_settings['item_class'] = 'dfe-list-item';
				break;

			case 'table':
				$default_settings['container_class'] = 'dfe-table-container';
				$default_settings['table_class'] = 'dfe-table';
				break;

			case 'card':
				$default_settings['container_class'] = 'dfe-cards';
				$default_settings['item_class'] = 'dfe-card';
				break;

			case 'single':
				$default_settings['container_class'] = 'dfe-single';
				$default_settings['item_class'] = 'dfe-single-item';
				break;
		}

		return $default_settings;
	}

	/**
	 * Validate template settings
	 *
	 * @param array $settings Template settings
	 *
	 * @return array Validated settings
	 */
	public static function validate_template_settings( $settings ) {
		$valid_settings = array();

		// Validate container class
		if ( isset( $settings['container_class'] ) ) {
			$valid_settings['container_class'] = sanitize_html_class( $settings['container_class'] );
		}

		// Validate item class
		if ( isset( $settings['item_class'] ) ) {
			$valid_settings['item_class'] = sanitize_html_class( $settings['item_class'] );
		}

		// Validate boolean settings
		$boolean_settings = array( 'show_labels', 'hide_empty', 'pagination', 'search' );
		foreach ( $boolean_settings as $setting ) {
			if ( isset( $settings[ $setting ] ) ) {
				$valid_settings[ $setting ] = (bool) $settings[ $setting ];
			}
		}

		// Validate limit
		if ( isset( $settings['limit'] ) ) {
			$valid_settings['limit'] = absint( $settings['limit'] );
			if ( $valid_settings['limit'] < 1 ) {
				$valid_settings['limit'] = 10;
			}
		}

		// Validate sort
		if ( isset( $settings['sort'] ) ) {
			$valid_sort_fields = array( 'id', 'date_created', 'date_updated' );
			if ( in_array( $settings['sort'], $valid_sort_fields, true ) ) {
				$valid_settings['sort'] = $settings['sort'];
			}
		}

		// Validate sort direction
		if ( isset( $settings['sort_direction'] ) ) {
			$valid_directions = array( 'ASC', 'DESC' );
			if ( in_array( strtoupper( $settings['sort_direction'] ), $valid_directions, true ) ) {
				$valid_settings['sort_direction'] = strtoupper( $settings['sort_direction'] );
			}
		}

		return $valid_settings;
	}

	/**
	 * Get template preview HTML
	 *
	 * @param string $template_id Template ID
	 *
	 * @return string Preview HTML
	 */
	public static function get_template_preview( $template_id ) {
		$template = self::get_template( $template_id );

		if ( ! $template ) {
			return '';
		}

		$preview_html = '';

		switch ( $template_id ) {
			case 'list':
				$preview_html = '<div class="dfe-preview-list">
					<div class="dfe-preview-item">
						<strong>Name:</strong> John Doe<br>
						<strong>Email:</strong> john@example.com<br>
						<strong>Date:</strong> January 1, 2024
					</div>
					<div class="dfe-preview-item">
						<strong>Name:</strong> Jane Smith<br>
						<strong>Email:</strong> jane@example.com<br>
						<strong>Date:</strong> January 2, 2024
					</div>
				</div>';
				break;

			case 'table':
				$preview_html = '<div class="dfe-preview-table">
					<table class="dfe-table">
						<thead>
							<tr>
								<th>Name</th>
								<th>Email</th>
								<th>Date</th>
							</tr>
						</thead>
						<tbody>
							<tr>
								<td>John Doe</td>
								<td>john@example.com</td>
								<td>January 1, 2024</td>
							</tr>
							<tr>
								<td>Jane Smith</td>
								<td>jane@example.com</td>
								<td>January 2, 2024</td>
							</tr>
						</tbody>
					</table>
				</div>';
				break;

			case 'card':
				$preview_html = '<div class="dfe-preview-cards">
					<div class="dfe-card">
						<h3>John Doe</h3>
						<p><strong>Email:</strong> john@example.com</p>
						<p><strong>Date:</strong> January 1, 2024</p>
					</div>
					<div class="dfe-card">
						<h3>Jane Smith</h3>
						<p><strong>Email:</strong> jane@example.com</p>
						<p><strong>Date:</strong> January 2, 2024</p>
					</div>
				</div>';
				break;

			case 'single':
				$preview_html = '<div class="dfe-preview-single">
					<h2>Entry Details</h2>
					<div class="dfe-field">
						<label>Name:</label>
						<span>John Doe</span>
					</div>
					<div class="dfe-field">
						<label>Email:</label>
						<span>john@example.com</span>
					</div>
					<div class="dfe-field">
						<label>Date:</label>
						<span>January 1, 2024</span>
					</div>
				</div>';
				break;
		}

		return $preview_html;
	}

	/**
	 * Create template directory structure
	 */
	public static function create_template_directories() {
		$directories = array(
			get_stylesheet_directory() . '/displayformentries',
			get_stylesheet_directory() . '/displayformentries/templates',
			get_stylesheet_directory() . '/displayformentries/css',
		);

		foreach ( $directories as $directory ) {
			if ( ! file_exists( $directory ) ) {
				wp_mkdir_p( $directory );
			}
		}
	}

	/**
	 * Copy default templates to theme
	 */
	public static function copy_default_templates() {
		self::create_template_directories();

		$source_dir = DFE_DIR . 'templates/';
		$destination_dir = get_stylesheet_directory() . '/displayformentries/templates/';

		if ( ! is_dir( $source_dir ) ) {
			return;
		}

		$files = glob( $source_dir . '*.php' );

		foreach ( $files as $file ) {
			$filename = basename( $file );
			$destination = $destination_dir . $filename;

			if ( ! file_exists( $destination ) ) {
				copy( $file, $destination );
			}
		}
	}
} 