<?php
/** Don't load directly */
defined( 'ABSPATH' ) || exit;

class RB_ADMIN_CORE {

	static $panel_slug = 'admin/templates/template';
	static $header = 'header';
	static $icon = 'dashicons-awards';
	static $recommended_plugins = [];
	protected static $instance = null;
	private static $theme_id = null;
	private static $core_plugin_id = null;
	private static $sub_pages;
	private static $dashboard_slug = 'foxiz-admin';
	private static $nonce = 'foxiz-admin';
	private static $keys = null;
	private static $token = null;
	private static $license = null;
	public $panel_name = 'dashboard';
	public $panel_template = 'admin_template';
	private $params = [];
	private $purchase_info = FOXIZ_LICENSE_ID;
	private $import_info = FOXIZ_IMPORT_ID;

	/**
	 * Class constructor.
	 *
	 * Initializes the singleton instance and hooks the initialization method
	 * to the 'plugins_loaded' action.
	 */
	public function __construct() {
		self::$instance = $this;

		add_action( 'plugins_loaded', [ $this, 'init' ], 0 );
	}

	/**
	 * Retrieves the singleton instance of the class.
	 *
	 * If the instance does not exist, it creates a new one.
	 *
	 * @return self The singleton instance of the class.
	 */
	static function get_instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Prevents cloning of the instance.
	 *
	 * This ensures that the singleton pattern is maintained by
	 * preventing the object from being cloned.
	 *
	 * @return void
	 */
	public function __clone() {

		_doing_it_wrong( __FUNCTION__, esc_html__( 'Not allowed!', 'foxiz-core' ), FOXIZ_CORE_VERSION );
	}

	/**
	 * Prevents unserializing of the instance.
	 *
	 * This protects the singleton instance from being unserialized,
	 * which could lead to multiple instances.
	 *
	 * @return void
	 */
	public function __wakeup() {

		_doing_it_wrong( __FUNCTION__, esc_html__( 'Not allowed!', 'foxiz-core' ), FOXIZ_CORE_VERSION );
	}

	/**
	 * Initializes the Foxiz Admin functionality.
	 *
	 * This function checks if the current user has the necessary permissions
	 * before proceeding with the initialization.
	 *
	 * @return void
	 */
	function init() {

		$this->get_configs();

		// Admin Only
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$this->subpage_files();

		add_action( 'init', [ $this, 'get_params' ], 0 );
		add_action( 'admin_menu', [ $this, 'register_dashboard' ], 0 );
		add_action( 'admin_menu', [ $this, 'register_subpages' ], 50 );
		add_action( 'admin_menu', [ $this, 'register_system_info' ], PHP_INT_MAX );

		add_action( 'wp_ajax_rb_register_theme', [ $this, 'register_theme' ] );
		add_action( 'wp_ajax_rb_deregister_theme', [ $this, 'deregister_theme' ] );
		add_action( 'wp_ajax_rb_recommended_plugin', [ $this, 'recommended_plugin' ] );
		add_action( 'wp_ajax_rb_renew_token', [ $this, 'renew_token' ] );
		add_action( 'ruby_refresh_token', [ $this, 'handle_token_refresh' ] );

		add_action( 'redux/' . FOXIZ_TOS_ID . '/panel/before', [ $this, 'header_template' ] );
		add_action( 'admin_init', [ 'Foxiz_Admin_Information', 'get_instance' ], 25 );
		add_action( 'admin_init', [ 'Ruby_Importer', 'get_instance' ], 10 );
		add_action( 'admin_enqueue_scripts', [ $this, 'register_assets' ], 15 );
		add_filter( 'http_request_args', [ $this, 'set_headers' ], PHP_INT_MAX, 2 );
		add_action( 'admin_init', [ $this, 'scheduled' ], PHP_INT_MAX );
		add_filter( 'foxiz_elementor_registered_widgets', [ $this, 'register_widgets' ], 10, 2 );

		add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'update_core_plugin' ], 5, 1 );
		add_filter( 'pre_set_transient_update_plugins', [ $this, 'update_core_plugin' ], 5, 1 );
		add_filter( 'pre_set_site_transient_update_themes', [ $this, 'update_theme' ], 1, 1 );
		add_filter( 'pre_set_transient_update_themes', [ $this, 'update_theme' ], 1, 1 );

		if ( ! $this->is_local() ) {
			add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'update_core_plugin' ], 5, 1 );
			add_filter( 'pre_set_transient_update_plugins', [ $this, 'update_core_plugin' ], 5, 1 );
			add_filter( 'pre_set_site_transient_update_themes', [ $this, 'update_theme' ], 1, 1 );
			add_filter( 'pre_set_transient_update_themes', [ $this, 'update_theme' ], 1, 1 );
		}
	}

	/**
	 * Retrieves and sets theme configuration values.
	 *
	 * This function assigns various configuration values related to the theme,
	 * ensuring they are properly defined before use.
	 *
	 * @return void
	 */
	private function get_configs() {

		self::$theme_id       = RUBY_THEME_NAME;
		self::$core_plugin_id = RUBY_THEME_NAME . '-core/' . RUBY_THEME_NAME . '-core.php';
		self::$keys           = get_option( RUBY_API_KEYS, [] );
		self::$license        = $this->license();
		self::$token          = $this->get_token();
		self::$sub_pages      = $this->set_sub_pages();
	}

	/**
	 * Retrieves the license
	 *
	 * @return mixed The purchase data if available, or false if not found.
	 */
	public function license() {

		return get_option( $this->purchase_info );
	}

	/**
	 * Retrieves the authentication token.
	 * if available.
	 *
	 * @return string/bool The authentication token.
	 */
	public function get_token() {
		return ! empty( self::$keys['token'] ) ? get_option( self::$keys['token'] )['token'] ?? false : false;
	}

	/**
	 * Sets the sub-pages for the admin panel based on purchase code availability.
	 *
	 * @return array The list of sub-pages with their respective classes and paths.
	 */
	public function set_sub_pages() {

		$args = [];

		if ( $this->is_registered() ) {
			$args[] = [
				'class' => 'Import',
				'path'  => 'admin/import/import.php',
			];
			$args[] = [
				'class' => 'Translation',
				'path'  => 'admin/translation/translation.php',
			];
			$args[] = [
				'class' => 'AdobeFonts',
				'path'  => 'admin/fonts/fonts.php',
			];
			$args[] = [
				'class' => 'GTM',
				'path'  => 'admin/gtm/gtm.php',
			];
		}

		return $args;
	}

	/**
	 * Checks if the license is registered.
	 *
	 * @return bool True if the license is registered, false otherwise.
	 */
	public function is_registered() {
		return (bool) $this->get_purchase_code();
	}

	/**
	 * Retrieves the purchase code.
	 *
	 * @return string|null The purchase code if available
	 */
	public function get_purchase_code() {

		return isset( self::$license['purchase_code'] ) ? self::$license['purchase_code'] : false;
	}

	/**
	 * Loads required subpage files.
	 *
	 * Iterates through the list of subpages and includes the corresponding files.
	 *
	 * @return void
	 */
	public function subpage_files() {

		$pages = self::$sub_pages;

		array_push( $pages, [
			'class' => 'SystemInfo',
			'path'  => 'admin/system-info/system-info.php',
		] );

		foreach ( $pages as $sub_page ) {
			if ( ! empty( $sub_page['path'] ) ) {
				require_once FOXIZ_CORE_PATH . $sub_page['path'];
			}
		}
	}

	/**
	 * Checks if the site is running on a local development environment.
	 *
	 * This function detects localhost environments by checking the site URL
	 * against common local addresses, including IPv4 (127.0.0.1) and IPv6 (::1).
	 *
	 * @return bool True if the site is running locally, false otherwise.
	 */
	private function is_local() {

		$site_url = parse_url( get_bloginfo( 'name' ), PHP_URL_HOST );

		if ( empty( $site_url ) ) {
			return false;
		}

		$local_hosts = [ 'localhost', '127.0.0.1' ];

		return in_array( strtolower( $site_url ), $local_hosts, true );
	}

	/**
	 * Retrieve and parse core parameters.
	 *
	 * This method initializes the `$params` property by merging the default values
	 * with the existing license data.
	 */
	public function get_params() {

		$this->params = wp_parse_args( self::$license, [
			'title'               => esc_html__( 'Foxiz', 'foxiz-core' ),
			'purchase_code'       => '',
			'status'              => $this->get_status(),
			'system_info'         => $this->get_system_info(),
			'menu'                => $this->get_dashboard_menu(),
			'expiry'              => $this->get_supported_expiration(),
			'step'                => get_option( 'foxiz_setup_current_step', 1 ),
			'recommended_plugins' => $this->get_recommended_plugins(),
			'can_install_plugins' => current_user_can( 'install_plugins' ),
		] );

		if ( ! $this->is_registered() ) {
			$this->panel_name = 'register';
		}
	}

	/**
	 * Retrieves the activation status
	 *
	 * @return mixed Returns activation details, or false if not available.
	 */
	public function get_status() {
		return ! empty( self::$keys['activation'] ) ? get_option( self::$keys['activation'] ) : false;
	}

	/**
	 * Retrieves system environment information including PHP version, memory limit, and server details.
	 *
	 * This function provides an array of system information used for debugging and compatibility checks.
	 *
	 * @return array An associative array containing system details.
	 */
	public function get_system_info() {

		return [
			'php_version'     => [
				'title'   => esc_html__( 'PHP Version', 'foxiz-core' ),
				'value'   => phpversion(),
				'min'     => '5.6',
				'passed'  => version_compare( phpversion(), '7.0.0' ) >= 0,
				'warning' => esc_html__( 'WordPress recommended PHP version 7.0 or greater to get better performance for your site.', 'foxiz-core' ),
			],
			'memory_limit'    => [
				'title'   => esc_html__( 'Memory Limit', 'foxiz-core' ),
				'value'   => size_format( wp_convert_hr_to_bytes( @ini_get( 'memory_limit' ) ) ),
				'min'     => '64M',
				'passed'  => wp_convert_hr_to_bytes( ini_get( 'memory_limit' ) ) >= 67108864,
				'warning' => esc_html__( 'The memory_limit value is set low. The theme recommended this value to be at least 64MB for the theme in order to work.', 'foxiz-core' ),
			],
			'max_input_vars'  => [
				'title'   => esc_html__( 'Max Input Vars', 'foxiz-core' ),
				'value'   => ini_get( 'max_input_vars' ),
				'min'     => '3000',
				'passed'  => (int) ini_get( 'max_input_vars' ) >= 2000,
				'warning' => esc_html__( 'The max_input_vars value is set low. The theme recommended this value to be at least 3000.', 'foxiz-core' ),
			],
			'post_max_size'   => [
				'title'   => esc_html__( 'Post Max Size', 'foxiz-core' ),
				'value'   => ini_get( 'post_max_size' ),
				'min'     => '32',
				'passed'  => (int) ini_get( 'post_max_size' ) >= 32,
				'warning' => esc_html__( 'The post_max_size value is set low. We recommended this value to be at least 32M.', 'foxiz-core' ),
			],
			'max_upload_size' => [
				'title'   => esc_html__( 'Max Upload Size', 'foxiz-core' ),
				'value'   => size_format( wp_max_upload_size() ),
				'min'     => '32',
				'passed'  => (int) wp_max_upload_size() >= 33554432,
				'warning' => esc_html__( 'The post_max_size value is set low. We recommended this value to be at least 32M.', 'foxiz-core' ),
			],
			'zip_archive'     => [
				'title'   => esc_html__( 'ZipArchive Support', 'foxiz-core' ),
				'value'   => class_exists( '\ZipArchive' ) ? 'Yes' : 'No',
				'passed'  => class_exists( '\ZipArchive' ),
				'warning' => esc_html__( 'ZipArchive can be used to autonomously update the theme.', 'foxiz-core' ),
			],
		];
	}

	/**
	 * Retrieves the dashboard menu structure.
	 *
	 * This function returns an array of menu items for the dashboard,
	 * including system info and other sections based on validation.
	 *
	 * @return array The structured dashboard menu items.
	 */
	public function get_dashboard_menu() {

		if ( ! $this->is_registered() ) {
			return [
				'registration' => [
					'title' => esc_html__( 'Registration', 'foxiz-core' ),
					'icon'  => 'rbi-dash-license',
					'url'   => $this->get_admin_menu_url( self::$dashboard_slug ),
				],
				'system'       => [
					'title' => esc_html__( 'System Info', 'foxiz-core' ),
					'icon'  => 'rbi-dash-info',
					'url'   => $this->get_admin_menu_url( 'ruby-system-info' ),
				],
			];
		}

		$menus              = [];
		$is_imported        = get_option( '_rb_flag_imported', false );
		$import_menu        = [
			'title' => esc_html__( 'Demo Importer', 'foxiz-core' ),
			'icon'  => 'rbi-dash-layer',
			'url'   => $this->get_admin_menu_url( 'rb-demo-importer' ),
		];
		$menus['dashboard'] = [
			'title' => esc_html__( 'Dashboard', 'foxiz-core' ),
			'icon'  => 'rbi-dash-dashboard',
			'url'   => $this->get_admin_menu_url( self::$dashboard_slug ),
		];
		if ( ! $is_imported ) {
			$menus['import'] = $import_menu;
		}
		$menus['options'] = [
			'title' => esc_html__( 'Theme Options', 'foxiz-core' ),
			'icon'  => 'rbi-dash-option',
			'url'   => $this->get_admin_menu_url( 'ruby-options' ),
		];
		$menus['more']    = [
			'title'     => esc_html__( 'Extra Features', 'foxiz-core' ),
			'icon'      => 'rbi-dash-more',
			'url'       => '#',
			'sub_items' => [],
		];
		if ( $is_imported ) {
			$menus['more']['sub_items']['import'] = $import_menu;
		}
		$menus['more']['sub_items']['translation'] = [
			'title' => esc_html__( 'Quick Translation', 'foxiz-core' ),
			'icon'  => 'rbi-dash-translate',
			'url'   => $this->get_admin_menu_url( 'ruby-translation' ),
		];
		$menus['more']['sub_items']['adobe']       = [
			'title' => esc_html__( 'Adobe Fonts', 'foxiz-core' ),
			'icon'  => 'rbi-dash-adobe',
			'url'   => $this->get_admin_menu_url( 'ruby-adobe-fonts' ),
		];
		$menus['more']['sub_items']['gtm']         = [
			'title' => esc_html__( 'Analytics 4', 'foxiz-core' ),
			'icon'  => 'rbi-dash-gtm',
			'url'   => $this->get_admin_menu_url( 'ruby-gmt-integration' ),
		];
		$menus['more']['sub_items']['openai']      = [
			'title' => esc_html__( 'OpenAI Assistant', 'foxiz-core' ),
			'icon'  => 'rbi-dash-openai',
			'url'   => $this->get_admin_menu_url( 'ruby-openai' ),
		];

		if ( foxiz_is_plugin_active( 'bbpress/bbpress.php' ) ) {
			$menus['more']['sub_items']['bbpress'] = [
				'title' => esc_html__( 'bbPress Forums', 'foxiz-core' ),
				'icon'  => 'rbi-dash-chat',
				'url'   => $this->get_admin_menu_url( 'ruby-bbp-supported' ),
			];
		}

		$menus['system'] = [
			'title' => esc_html__( 'System Info', 'foxiz-core' ),
			'icon'  => 'rbi-dash-info',
			'url'   => $this->get_admin_menu_url( 'ruby-system-info' ),
		];

		return $menus;
	}

	/**
	 * Generates the full URL for a given admin menu page.
	 *
	 * This function constructs an admin panel URL based on the provided menu slug.
	 *
	 * @param string $menu_slug The slug of the admin menu page.
	 *
	 * @return string The full admin menu URL.
	 */
	private function get_admin_menu_url( $menu_slug ) {
		return admin_url( add_query_arg( 'page', $menu_slug, 'admin.php' ) );
	}

	/**
	 * Retrieves the expiration date of the support.
	 *
	 * @return string|null The expiration date if available.
	 */
	public function get_supported_expiration() {
		return ! empty( self::$keys['expiration'] ) ? get_option( self::$keys['expiration'] ) : false;
	}

	/**
	 * Retrieves the list of recommended plugins for the theme.
	 *
	 * This function returns an array of recommended plugins, including their
	 * names, slugs, and installation sources (WordPress repository or external).
	 *
	 * @return array The list of recommended plugins.
	 */
	private function get_recommended_plugins() {

		return [
			'elementor'                          => [
				'title'       => esc_html__( 'Elementor', 'foxiz-core' ),
				'description' => esc_html__( 'Leading website builder platform for professionals and business owners.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'elementor',
				'file'        => 'elementor/elementor.php',
				'status'      => $this->plugin_status( 'elementor/elementor.php' ),
				'icon'        => 'https://ps.w.org/elementor/assets/icon-128x128.gif',
			],
			'easy-post-submission'               => [
				'title'       => esc_html__( 'Easy Post Submission', 'foxiz-core' ),
				'description' => esc_html__( 'Enable users to submit posts and manage profiles from the front-end. Ideal for news, magazines, and creative platforms.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'easy-post-submission',
				'file'        => 'easy-post-submission/easy-post-submission.php',
				'status'      => $this->plugin_status( 'easy-post-submission/easy-post-submission.php' ),
				'icon'        => 'https://ps.w.org/easy-post-submission/assets/icon-128x128.gif',
			],
			'local-fonts-uploader'               => [
				'title'       => esc_html__( 'Local Fonts Uploader', 'foxiz-core' ),
				'description' => esc_html__( 'Easily upload and host fonts locally. Avoid external requests to enhance security, privacy, speed, and GDPR compliance.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'local-fonts-uploader',
				'file'        => 'local-fonts-uploader/local-fonts-uploader.php',
				'status'      => $this->plugin_status( 'local-fonts-uploader/local-fonts-uploader.php' ),
				'icon'        => 'https://ps.w.org/local-fonts-uploader/assets/icon-128x128.gif?rev=3260505',
			],
			'leadin'                             => [
				'title'       => esc_html__( 'HubSpot', 'foxiz-core' ),
				'description' => esc_html__( 'All the tools and integrations you need for marketing, sales, and customer service.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'leadin',
				'file'        => 'leadin/leadin.php',
				'status'      => $this->plugin_status( 'leadin/leadin.php' ),
				'icon'        => 'https://ps.w.org/leadin/assets/icon-128x128.png',
			],
			'post-views-counter'                 => [
				'title'       => esc_html__( 'Post Views Counter', 'foxiz-core' ),
				'description' => esc_html__( 'Display how many times a post, page or custom post type.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'post-views-counter',
				'file'        => 'post-views-counter/post-views-counter.php',
				'status'      => $this->plugin_status( 'post-views-counter/post-views-counter.php' ),
				'icon'        => 'https://ps.w.org/post-views-counter/assets/icon-128x128.png',
			],
			'breadcrumb-navxt'                   => [
				'title'       => esc_html__( 'Breadcrumb NavXT', 'foxiz-core' ),
				'description' => esc_html__( 'Generates locational breadcrumb trails for your WordPress powered blog or website.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'breadcrumb-navxt',
				'file'        => 'breadcrumb-navxt/breadcrumb-navxt.php',
				'status'      => $this->plugin_status( 'breadcrumb-navxt/breadcrumb-navxt.php' ),
				'icon'        => 'https://ps.w.org/breadcrumb-navxt/assets/icon.svg',
			],
			'contact-form-7'                     => [
				'title'       => esc_html__( 'Contact Form 7', 'foxiz-core' ),
				'description' => esc_html__( 'Just another contact form plugin. Simple but flexible.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'contact-form-7',
				'file'        => 'contact-form-7/wp-contact-form-7.php',
				'status'      => $this->plugin_status( 'contact-form-7/wp-contact-form-7.php' ),
				'icon'        => 'https://ps.w.org/contact-form-7/assets/icon.svg',
			],
			'mailchimp-for-wp'                   => [
				'title'       => esc_html__( 'Mailchimp for WordPress', 'foxiz-core' ),
				'description' => esc_html__( 'Allows you to add a multitude of newsletter sign-up methods to your site.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'mailchimp-for-wp',
				'file'        => 'mailchimp-for-wp/mailchimp-for-wp.php',
				'status'      => $this->plugin_status( 'mailchimp-for-wp/mailchimp-for-wp.php' ),
				'icon'        => 'https://ps.w.org/mailchimp-for-wp/assets/icon-256x256.png',
			],
			'simple-membership'                  => [
				'title'       => esc_html__( 'Simple Membership', 'foxiz-core' ),
				'description' => esc_html__( 'A flexible, well-supported, and easy-to-use WordPress membership plugin for offering free and premium content from your WordPress site.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'simple-membership',
				'file'        => 'simple-membership/simple-wp-membership.php',
				'status'      => $this->plugin_status( 'simple-membership/simple-wp-membership.php' ),
				'icon'        => 'https://ps.w.org/simple-membership/assets/icon-128x128.png',
			],
			'woocommerce'                        => [
				'title'       => esc_html__( 'WooCommerce', 'foxiz-core' ),
				'description' => esc_html__( 'All the tools and integrations you need for marketing, sales, and customer service.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'woocommerce',
				'file'        => 'woocommerce/woocommerce.php',
				'status'      => $this->plugin_status( 'woocommerce/woocommerce.php' ),
				'icon'        => 'https://ps.w.org/woocommerce/assets/icon-128x128.png',
			],
			'publishpress-authors'               => [
				'title'       => esc_html__( 'Multiple Authors', 'foxiz-core' ),
				'description' => esc_html__( 'Create, manage and display authors for all your WordPress content.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'publishpress-authors',
				'file'        => 'publishpress-authors/publishpress-authors.php',
				'status'      => $this->plugin_status( 'publishpress-authors/publishpress-authors.php' ),
				'icon'        => 'https://ps.w.org/publishpress-authors/assets/icon-256x256.png',
			],
			'custom-post-type-ui'                => [
				'title'       => esc_html__( 'Custom Post Type UI', 'foxiz-core' ),
				'description' => esc_html__( 'Admin UI for creating custom content types like post types and taxonomies.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'custom-post-type-ui',
				'file'        => 'custom-post-type-ui/custom-post-type-ui.php',
				'status'      => $this->plugin_status( 'custom-post-type-ui/custom-post-type-ui.php' ),
				'icon'        => 'https://ps.w.org/custom-post-type-ui/assets/icon-256x256.png',
			],
			'bbpress'                            => [
				'title'       => esc_html__( 'bbPress', 'foxiz-core' ),
				'description' => esc_html__( 'Forum software from the creators of WordPress. Quickly setup a place for asyncronous discussion, subscriptions, and more!', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'bbpress',
				'file'        => 'bbpress/bbpress.php',
				'status'      => $this->plugin_status( 'bbpress/bbpress.php' ),
				'icon'        => 'https://ps.w.org/bbpress/assets/icon.svg',
			],
			'wp-recipe-maker'                    => [
				'title'       => esc_html__( 'WP Recipe Maker', 'foxiz-core' ),
				'description' => esc_html__( 'Easy recipe plugin that everyone can use with automatic JSON-LD metadata for your recipes.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'wp-recipe-maker',
				'file'        => 'wp-recipe-maker/wp-recipe-maker.php',
				'status'      => $this->plugin_status( 'wp-recipe-maker/wp-recipe-maker.php' ),
				'icon'        => 'https://ps.w.org/wp-recipe-maker/assets/icon-128x128.png',
			],
			'web-stories'                        => [
				'title'       => esc_html__( 'Web Stories', 'foxiz-core' ),
				'description' => esc_html__( 'Visual storytelling format for the web, enabling you to easily create visual narratives.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'web-stories',
				'file'        => 'web-stories/web-stories.php',
				'status'      => $this->plugin_status( 'web-stories/web-stories.php' ),
				'icon'        => 'https://ps.w.org/web-stories/assets/icon.svg',
			],
			'cryptocurrency-price-ticker-widget' => [
				'title'       => esc_html__( 'Cryptocurrency Widgets', 'foxiz-core' ),
				'description' => esc_html__( 'Easily display a crypto ticker widget, coins price lists, tables, multi-currency tabs & price labels anywhere.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'cryptocurrency-price-ticker-widget',
				'file'        => 'cryptocurrency-price-ticker-widget/cryptocurrency-price-ticker-widget.php',
				'status'      => $this->plugin_status( 'cryptocurrency-price-ticker-widget/cryptocurrency-price-ticker-widget.php' ),
				'icon'        => 'https://ps.w.org/cryptocurrency-price-ticker-widget/assets/icon-256x256.png',
			],
			'wp-super-cache'                     => [
				'title'       => esc_html__( 'WP Super Cache', 'foxiz-core' ),
				'description' => esc_html__( 'A very fast caching engine for WordPress that produces static html files.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'wp-super-cache',
				'file'        => 'wp-super-cache/wp-cache.php',
				'status'      => $this->plugin_status( 'wp-super-cache/wp-cache.php' ),
				'icon'        => 'https://ps.w.org/wp-super-cache/assets/icon-256x256.png',
			],
			'autoptimize'                        => [
				'title'       => esc_html__( 'Autoptimize', 'foxiz-core' ),
				'description' => esc_html__( 'Autoptimize speeds up your website by optimizing JS, CSS, images.', 'foxiz-core' ),
				'package'     => false,
				'slug'        => 'autoptimize',
				'file'        => 'autoptimize/autoptimize.php',
				'status'      => $this->plugin_status( 'autoptimize/autoptimize.php' ),
				'icon'        => 'https://ps.w.org/autoptimize/assets/icon-256X256.png',
			],
		];
	}

	/**
	 * Retrieves the status of a given plugin.
	 *
	 * This function checks whether a plugin is installed, active, or active for the entire network.
	 *
	 * @param string $plugin Plugin slug or path (e.g., 'plugin-folder/plugin-file.php').
	 *
	 * @return string|false Plugin status: 'active'. Returns false if the user lacks permission.
	 */
	public function plugin_status( $plugin ) {

		if ( ! current_user_can( 'install_plugins' ) ) {
			return false;
		}

		if ( ! function_exists( 'is_plugin_active_for_network' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugin = $this->get_plugin_file_path( $plugin );
		if ( ! file_exists( WP_PLUGIN_DIR . '/' . $plugin ) ) {
			return 'not_found';
		}

		if ( in_array( $plugin, (array) get_option( 'active_plugins', [] ), true ) || is_plugin_active_for_network( $plugin ) ) {
			return 'active';
		}

		return 'inactive';
	}

	/**
	 * Retrieves the plugin file path based on the provided slug.
	 *
	 * This function determines the main plugin file for a given plugin slug.
	 *
	 * @param string $slug Plugin slug (e.g., 'woocommerce' or 'woocommerce/woocommerce.php').
	 *
	 * @return string|WP_Error Plugin file path if found, or WP_Error if the plugin does not exist.
	 */
	public function get_plugin_file_path( $slug ) {

		if ( strpos( $slug, '/' ) ) {
			return $slug;
		}

		$recommended_plugins = $this->get_recommended_plugins();

		if ( isset( $recommended_plugins[ $slug ] ) ) {
			$plugin_information = $recommended_plugins[ $slug ];
			if ( ! empty( $plugin_information['file'] ) ) {
				return $slug . '/' . $plugin_information['file'] . '.php';
			}
		}

		return $slug . '/' . $slug . '.php';
	}

	/**
	 * Registers the enqueue action for loading assets in the WordPress admin panel.
	 *
	 * This function hooks into 'admin_enqueue_scripts' to enqueue necessary assets
	 * for the plugin or theme. The priority is set to 25 to ensure assets are loaded
	 * at the appropriate time.
	 *
	 * @return void
	 */
	public function load_assets() {

		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ], 25 );
	}

	/**
	 * Registers and enqueues JavaScript and CSS assets for the admin dashboard.
	 *
	 * This function determines the appropriate CSS file based on the site's text direction
	 * (LTR or RTL) and registers the necessary assets.
	 *
	 * @return void
	 */
	public function register_assets() {

		$css_path = ! is_rtl() ? 'dashboard' : 'dashboard-rtl';
		wp_enqueue_style( 'rb-admin-style', plugins_url( FOXIZ_REL_PATH . '/admin/assets/' . $css_path . '.css' ), [], FOXIZ_CORE_VERSION, 'all' );
		wp_register_script( 'rb-searcher', plugins_url( FOXIZ_REL_PATH . '/admin/assets/searcher.js' ), [ 'jquery' ], FOXIZ_CORE_VERSION, true );
		wp_register_script( 'rb-admin-script', plugins_url( FOXIZ_REL_PATH . '/admin/assets/panel.js' ), [
			'jquery',
			'wp-util',
			'rb-searcher',
		], FOXIZ_CORE_VERSION, true );
		wp_localize_script( 'rb-admin-script', 'foxizAdminCore', $this->localize_params() );
	}

	/**
	 * Filters and returns localized parameters for the admin interface.
	 *
	 * This function allows modification of admin-related parameters through the
	 * 'rb_admin_' filter hook before returning them.
	 *
	 * @return mixed The filtered parameters for admin use.
	 */
	public function localize_params() {

		return apply_filters( 'rb_admin_localize_data', [
			'ajaxUrl'                => admin_url( 'admin-ajax.php' ),
			'_rbNonce'               => wp_create_nonce( self::$nonce ),
			'updating'               => esc_html__( 'Updating...', 'foxiz-core' ),
			'reload'                 => esc_html__( 'Reload...', 'foxiz-core' ),
			'error'                  => esc_html__( 'Error!', 'foxiz-core' ),
			'confirmDeleteImporter'  => esc_html__( 'Are you sure you want to delete the imported content? This action cannot be undone.', 'foxiz-core' ),
			'confirmUpdateExp'       => esc_html__( 'Revalidate the support period and update demo data?', 'foxiz-core' ),
			'confirmDeleteAdobeFont' => esc_html__( 'Are you sure to delete this font project?', 'foxiz-core' ),
			'confirmDeactivate'      => esc_html__( 'Are you sure you want to deactivate the current license? This action could lead to site errors or disruptions.', 'foxiz-core' ),
			'confirmDeleteGA'        => esc_html__( 'Are you sure to delete the Google Tag?', 'foxiz-core' ),
		] );
	}

	/**
	 * Enqueues admin JavaScript assets.
	 *
	 * This function enqueues the 'rb-admin-script' JavaScript file for use in the
	 * WordPress admin panel.
	 *
	 * @return void
	 */
	public function enqueue_assets() {

		wp_enqueue_script( 'rb-admin-script' );
	}

	/**
	 * Renders the admin panel template.
	 *
	 * This function outputs the header template and the main admin template part
	 * based on the specified panel slug and panel name.
	 *
	 * @return void
	 */
	public function admin_template() {

		$this->header_template();
		echo rb_admin_get_template_part( self::$panel_slug, $this->panel_name, $this->params );
	}

	/**
	 * Renders the admin panel header template.
	 *
	 * This function outputs the header section of the admin panel.
	 *
	 * @return void
	 */
	public function header_template() {

		echo rb_admin_get_template_part( self::$panel_slug, self::$header, $this->params );
	}

	/**
	 * Registers the dashboard menu in the WordPress admin panel.
	 *
	 * This function adds a custom dashboard page if the theme is active.
	 *
	 * @return void
	 */
	public function register_dashboard() {

		if ( ! defined( 'FOXIZ_THEME_VERSION' ) ) {
			return;
		}

		$panel_title = esc_html__( 'Foxiz', 'foxiz-core' );

		$panel_hook_suffix = add_menu_page( $panel_title, $panel_title, 'manage_options', self::$dashboard_slug,
			[ $this, $this->panel_template ], self::$icon, 3 );

		add_action( 'load-' . $panel_hook_suffix, [ $this, 'load_assets' ] );
	}

	/**
	 * Registers subpages in the WordPress admin menu.
	 *
	 * This function checks if the theme is defined and if subpages exist before registering them.
	 *
	 * @return void
	 */
	public function register_subpages() {

		if ( ! defined( 'FOXIZ_THEME_VERSION' ) || empty( self::$sub_pages ) ) {
			return;
		}

		global $submenu;
		foreach ( self::$sub_pages as $sub_page ) {

			if ( empty( $sub_page['class'] ) || empty( $sub_page['path'] ) ) {
				continue;
			}

			$class_name = 'rbSubPage' . $sub_page['class'];
			$sub_page   = new $class_name();
			if ( ! empty( $sub_page->menu_slug ) ) {
				$page_hook_suffix = add_submenu_page( self::$dashboard_slug, $sub_page->page_title, $sub_page->menu_title, $sub_page->capability, $sub_page->menu_slug,
					[ $sub_page, 'render' ], $sub_page->position );
				add_action( 'load-' . $page_hook_suffix, [ $this, 'load_assets' ] );
			}
		}

		if ( isset( $submenu[ self::$dashboard_slug ][0][0] ) ) {
			$submenu[ self::$dashboard_slug ][0][0] = $this->get_dashboard_label();
		}
	}

	/**
	 * Retrieves the dashboard label based on the activation.
	 *
	 * @return string The localized dashboard label.
	 */
	public function get_dashboard_label() {

		return $this->is_registered() ? esc_html__( 'Home', 'foxiz-core' ) : esc_html__( 'Registration', 'foxiz-core' );

	}

	/**
	 * Registers the System Info subpage in the WordPress admin panel.
	 *
	 * This function adds a "System Info" subpage under the main dashboard menu.
	 *
	 * @return void
	 */
	public function register_system_info() {

		$sub_page = [
			'class' => 'SystemInfo',
			'path'  => 'admin/system-info/system-info.php',
		];

		$class_name = 'rbSubPage' . $sub_page['class'];
		$sub_page   = new $class_name();

		if ( ! empty( $sub_page->menu_slug ) ) {
			$page_hook_suffix = add_submenu_page( self::$dashboard_slug, $sub_page->page_title, $sub_page->menu_title, $sub_page->capability, $sub_page->menu_slug,
				[ $sub_page, 'render' ] );
			add_action( 'load-' . $page_hook_suffix, [ $this, 'load_assets' ] );
		}
	}

	/**
	 * Filters and returns the registered widgets.
	 *
	 * This function checks if widgets are registered and returns them;
	 * otherwise, it returns an empty array.
	 *
	 * @param array $widgets The array of widgets to be registered.
	 *
	 * @return array The filtered array of widgets.
	 */
	public function register_widgets( $widgets ) {
		return $this->is_registered() ? $widgets : [];
	}

	/**
	 * Retrieves stored import data.
	 *
	 * This function fetches the import data from the
	 *
	 * @return array|bool The retrieved import data, or an empty array if no data is found.
	 */
	public function get_imports() {

		$data = get_option( $this->import_info, [] );
		if ( ! empty( $data['listing'] ) && is_array( $data['listing'] ) ) {
			foreach ( $data['listing'] as $index => $values ) {
				$data['listing'][ $index ]['content']       = $this->generate_import_url( $index, 'content' );
				$data['listing'][ $index ]['pages']         = $this->generate_import_url( $index, 'pages' );
				$data['listing'][ $index ]['theme_options'] = $this->generate_import_url( $index, 'theme-options' );
				$data['listing'][ $index ]['widgets']       = $this->generate_import_url( $index, 'widgets' );
				$data['listing'][ $index ]['taxonomies']    = $this->generate_import_url( $index, 'taxonomies' );
				$data['listing'][ $index ]['post_types']    = $this->generate_import_url( $index, 'post-types' );
			}

			return $data['listing'];
		}

		return false;
	}

	/**
	 * Generates a URL for importing demo content.
	 *
	 * This function constructs an import URL by appending the specified demo index,
	 * data key, and a purchase code as query parameters.
	 *
	 * @param string|int $index The identifier of the demo to import.
	 * @param string $key The specific data key related to the import process.
	 *
	 * @return string The generated import URL.
	 */
	public function generate_import_url( $index, $key ) {

		return "importer/?demo={$index}&data={$key}";
	}

	/**
	 * Retrieves the email with activation.
	 *
	 * @return string The email
	 */
	public function get_registered_email() {

		return isset( self::$license['email'] ) ? self::$license['email'] : ( is_multisite() ? get_site_option( 'admin_email' ) : get_option( 'admin_email' ) );
	}

	/**
	 * Retrieves the identifier key of token
	 *
	 * @return string|false The key  if available.
	 */
	public function get_identifier() {
		return ! empty( self::$keys['token'] ) ? get_option( self::$keys['token'] )['key'] ?? false : false;
	}

	/**
	 *
	 * This function checks if the request is targeting the WordPress theme update API
	 * and processes the request data accordingly.
	 *
	 * @param array $request The original request arguments.
	 * @param string $url The request URL being checked.
	 *
	 * @return mixed The modified request arguments or processed data.
	 */
	public function set_headers( $request, $url ) {

		if ( false !== strpos( $url, '//api.wordpress.org/themes/update-check/1.1/' ) ) {
			$data = json_decode( $request['body']['themes'] );
			unset( $data->themes->{self::$theme_id} );
			$request['body']['themes'] = wp_json_encode( $data );
		} elseif ( false !== strpos( $url, '//api.wordpress.org/plugins/update-check/1.1/' ) ) {

			$data = json_decode( $request['body']['plugins'] );
			unset( $data->plugins->{self::$core_plugin_id} );

			$request['body']['plugins'] = wp_json_encode( $data );
		} elseif ( false !== strpos( $url, RUBY_API_HOST ) ) {
			$request['headers'] = $this->get_headers();
		}

		return $request;
	}

	/**
	 * Generates HTTP headers with an authorization token.
	 *
	 * @return array An associative array of HTTP headers including 'User-Agent' and 'Authorization'.
	 */
	private function get_headers() {
		return [
			'User-Agent'    => $this->get_user_agent(),
			'Authorization' => $this->get_authorization(),
		];
	}

	/**
	 * Retrieves the user agent string for HTTP requests.
	 *
	 * The user agent includes the WordPress version and the site URL,
	 * which can be useful for API requests and debugging.
	 *
	 * @return string The formatted user agent string.
	 */
	private function get_user_agent() {
		return 'WordPress/' . get_bloginfo( 'version' ) . '; ' . get_bloginfo( 'url' );
	}

	/**
	 * Generates the authorization header for API requests.
	 *
	 * This function retrieves the authentication token and formats it as a
	 * Bearer token for use in API authorization headers.
	 *
	 * @return string The formatted authorization header.
	 */
	private function get_authorization() {
		return 'Bearer ' . $this->get_token();
	}

	/**
	 * Schedules a single event to refresh the token.
	 *
	 * If the event is not already, it schedules another execution.
	 *
	 * @return void
	 */
	public function scheduled() {
		$this->is_schedule_blocked() || $this->handle_token_refresh();
	}

	/**
	 * Checks if the schedule is blocked by a transient.
	 *
	 * @return bool True if the schedule is blocked, false otherwise.
	 */
	public function is_schedule_blocked() {
		return (bool) get_transient( '_' . FOXIZ_LICENSE_ID );
	}

	/**
	 * Checks if a token refresh is needed and performs the refresh if required.
	 *
	 * This method is similar to force_refresh_token(), but may be used in contexts
	 * where the token refresh is part of a larger progress-based workflow or sequence.
	 *
	 * @return void
	 */
	public function handle_token_refresh() {

		if ( ! $this->is_registered() ) {
			return;
		}

		empty( self::$token ) ? $this->try_recover_license() : ( $this->need_refresh_token() && $this->refresh_token() );
	}

	/**
	 * Recovers the license by sending a request to the API.
	 *
	 * This function attempts to recover the license using the stored purchase code.
	 *
	 * @return bool
	 */
	public function try_recover_license() {

		$response = $this->fetch_post_data(
			$this->get_endpoint_url( '/ruby-api/v1/recover' ), [
			'purchase_code' => $this->get_purchase_code(),
			'theme'         => self::$theme_id,
		], true );

		isset( $response['code'] ) && 200 === (int) $response['code']
			? $this->license_activate( $response )
			: null;

		$this->set_schedule_expiration();

		return true;
	}

	/**
	 * Sends a POST request to the specified API URL and retrieves the response.
	 *
	 * This function sends data to a remote API and handles errors. If `$silent` is `true`,
	 * it suppresses fatal errors and returns `false` instead of terminating execution.
	 *
	 * @param string $api_url The API endpoint URL.
	 * @param array $data Optional. The data to send in the request body. Default is an empty array.
	 * @param bool $silent Optional. Whether to suppress errors. If `true`, returns `false` instead of terminating. Default is `false`.
	 *
	 * @return array|false Decoded JSON response as an associative array, or `false` on failure if `$silent` is `true`.
	 */
	public function fetch_post_data( $api_url, $data = [], $silent = false ) {

		$response = wp_remote_post( $api_url, [
			'timeout' => 30,
			'headers' => $this->get_headers(),
			'body'    => ! empty( $data ) ? $data : [],
		] );

		if ( is_wp_error( $response ) ) {
			if ( ! $silent ) {
				wp_send_json_error( esc_html__( 'There is an issue with remote requests. Please check if your hosting allows opening URLs or contact the support team for assistance.', 'foxiz-core' ), 400 );
				wp_die();
			}

			return []; // Return empty array in silent mode instead of dying
		}

		return json_decode( wp_remote_retrieve_body( $response ), true );
	}

	/**
	 * Retrieves the base API URL.
	 *
	 * Constructs and returns the API URL, optionally including the WordPress REST API endpoint.
	 *
	 * @param string $request Optional. Additional request path to append to the base URL.
	 * @param bool $json_endpoint Optional. Whether to include the WordPress REST API prefix ('wp-json/'). Default false.
	 *
	 * @return string The constructed API URL.
	 */
	public function get_endpoint_url( $request = '', $json_endpoint = true ) {
		$path = $json_endpoint ? '/wp-json' : '';
		$path .= rtrim( $request, '/' ); // Prevent double slashes

		return RUBY_API_HOST . $path;
	}

	/**
	 * Activates the license by sanitizing and storing relevant data.
	 *
	 * @param array $data The license data to be processed.
	 */
	private function license_activate( $data = [] ) {

		$excluded_keys = [ 'code', 'success', 'error', 'message', 'retry_after', 'status' ];
		$data          = array_diff_key( (array) $data, array_flip( $excluded_keys ) );

		foreach ( $this->sanitize_data( $data ) as $key => $value ) {
			$this->set_data( $key, $value );
		}
	}

	/**
	 * Recursively sanitizes an array of data.
	 *
	 * @param mixed $data The input data (can be an array or scalar).
	 *
	 * @return mixed The sanitized data.
	 */
	public function sanitize_data( $data ) {
		if ( is_array( $data ) ) {
			return array_map( [ $this, 'sanitize_data' ], $data ); // Recursively apply sanitization
		}

		return sanitize_text_field( $data ); // Sanitize scalar values
	}

	/**
	 * Saves sanitized data.
	 *
	 * @param string $key The key.
	 * @param mixed $value The value to store.
	 */
	private function set_data( $key, $value ) {
		update_option( $key, $value );
	}

	/**
	 * Sets the expiration time for the refresh token.
	 *
	 * This function is intended to define and store the expiration time
	 * for the token refresh process, ensuring that it is managed properly.
	 *
	 * @return void This function does not return anything.
	 */
	public function set_schedule_expiration() {
		set_transient( '_' . FOXIZ_LICENSE_ID, true, 3 * DAY_IN_SECONDS );
	}

	/**
	 * Checks whether a token refresh is needed.
	 *
	 * This function verifies if the transient associated token
	 * If the transient is missing or expired,
	 * it indicates that a token refresh is required.
	 *
	 * @return bool Returns true if the transient is empty or not set,
	 *              meaning a token refresh is needed. Otherwise, returns false.
	 */
	public function need_refresh_token() {

		if ( empty( $this->get_token() ) ) {
			return false;
		}

		return time() + 1 * DAY_IN_SECONDS > $this->get_token_expiry();
	}

	/**
	 * Retrieves the authentication token.
	 * if available.
	 *
	 * @return string/bool The authentication token.
	 */
	public function get_token_expiry() {
		return ! empty( self::$keys['token'] ) ? get_option( self::$keys['token'] )['expiry'] ?? false : false;
	}

	/**
	 * Schedules a single event to refresh the token.
	 *
	 * If the event is not already, it schedules another execution.
	 *
	 * @return bool
	 */
	public function refresh_token( $force_renew = false ) {

		$response = $this->fetch_post_data(
			$this->get_endpoint_url( '/ruby-api/v1/refresh-token' ), [
			'theme'  => self::$theme_id,
			'action' => $force_renew ? 'renew' : 'refresh',
		], true );

		isset( $response['code'] ) && (int) $response['code'] === 200
			? $this->license_activate( $response )
			: null;

		$this->set_schedule_expiration();

		return true;
	}

	/**
	 * Handles theme registration.
	 *
	 * This function verifies the nonce for security and processes the theme
	 * registration request.
	 *
	 * @return void
	 */
	public function register_theme() {

		$nonce = ( isset( $_POST['_nonce'] ) ) ? sanitize_key( $_POST['_nonce'] ) : '';

		if ( empty( $nonce ) || false === wp_verify_nonce( $nonce, self::$nonce ) ) {
			wp_send_json_error( esc_html__( 'Nonce validation failed. Please try again.', 'foxiz-core' ), 400 );

			wp_die();
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( esc_html__( 'Sorry, you are not allowed to perform this action.', 'foxiz-core' ), 403 );
			wp_die();
		}

		$code  = isset( $_POST['purchase_code'] ) ? sanitize_text_field( wp_unslash( $_POST['purchase_code'] ) ) : false;
		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : false;

		// Validate required fields
		if ( ! $code || ! $email ) {
			wp_send_json_error( esc_html__( 'The submission form is invalid. Please take a moment to review the input form.', 'foxiz-core' ), 400 );
			wp_die();
		}

		$this->handle_register_theme( $code, $email );
	}

	/**
	 * Handles theme registration by sending purchase details to the remote API.
	 *
	 * @param string $code The purchase code provided by the user.
	 * @param string $email The email address associated with the purchase.
	 *
	 * @return mixed The API response, or false on failure.
	 */
	public function handle_register_theme( $code, $email ) {

		$response = $this->fetch_post_data(
			$this->get_endpoint_url( '/ruby-api/v1/register' ), [
			'purchase_code' => $code,
			'email'         => $email,
			'theme'         => self::$theme_id,
		] );

		$code    = $this->get_response_code( $response );
		$message = $this->get_response_message( $response );
		$this->license_activate( $response );

		if ( $code === 200 ) {
			wp_send_json_success( $message, 200 );
		} else {
			wp_send_json_error( $message, $code );
		}

		wp_die();
	}

	/**
	 * Retrieves the response code from an API response.
	 *
	 * Otherwise, it falls back to a general request error code (e.g., 500 for server errors).
	 *
	 * @param array $response The API response array.
	 *
	 * @return int The response code or a default error code in case of failure.
	 */
	public function get_response_code( $response ) {
		return ! empty( $response['code'] ) ? (int) $response['code'] : 500;
	}

	/**
	 * Retrieves the response message from an API response.
	 *
	 * This function checks if a 'message' key exists in the response array and returns
	 * its value after sanitization. If the message is not present, it returns a default
	 * error message.
	 *
	 * @param array $response The API response array.
	 *
	 * @return string The sanitized response message, or a default error message if none is found.
	 */
	public function get_response_message( $response ) {
		return ! empty( $response['message'] ) ? esc_html( $response['message'] )
			: esc_html__( 'An error occurred during the API response. Please contact the support team for assistance.', 'foxiz-core' );
	}

	/**
	 * Handles theme deregistration.
	 *
	 * This function verifies the nonce for security and processes the theme
	 * deregistration request.
	 *
	 * @return void
	 */
	public function deregister_theme() {

		$nonce = ( isset( $_POST['_nonce'] ) ) ? sanitize_key( $_POST['_nonce'] ) : '';

		if ( empty( $nonce ) || false === wp_verify_nonce( $nonce, self::$nonce ) ) {
			wp_send_json_error( esc_html__( 'Nonce validation failed. Please try again.', 'foxiz-core' ), 400 );

			wp_die();
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( esc_html__( 'Sorry, you are not allowed to perform this action.', 'foxiz-core' ), 403 );
			wp_die();
		}

		// handle_deregister_theme
		$this->handle_deregister_theme();
	}

	/**
	 * Handles the deregistration of the theme by sending a request
	 * to the API and deactivating the license.
	 *
	 * @return void
	 */
	public function handle_deregister_theme() {

		$response = $this->fetch_post_data(
			$this->get_endpoint_url( '/ruby-api/v1/deregister' ), [
				'theme' => self::$theme_id,
			]
		);

		$code    = $this->get_response_code( $response );
		$message = $this->get_response_message( $response );
		$this->license_deactivate( $response );

		if ( $code === 200 ) {
			wp_send_json_success( $message, 200 );
		} else {
			wp_send_json_error( $message, $code );
		}

		wp_die();
	}

	/**
	 * Deactivates the license
	 *
	 * @param array $data The license data to be processed.
	 */
	private function license_deactivate( $data = [] ) {

		$excluded_keys = [ 'code', 'success', 'error', 'message', 'retry_after', 'status' ];
		$data          = array_diff_key( (array) $data, array_flip( $excluded_keys ) );

		foreach ( $this->sanitize_data( $data ) as $key => $value ) {
			$this->delete_data( $key );
		}
	}

	/**
	 * Deletes an activation data.
	 *
	 * @param string $key The name to delete.
	 *
	 * @return void
	 */
	private function delete_data( $key ) {
		delete_option( $key );
	}

	/**
	 * Schedules a single event to refresh the token.
	 *
	 * If the event is not already, it schedules another execution.
	 *
	 * @return bool
	 */
	public function renew_token() {

		$nonce = ( isset( $_POST['_nonce'] ) ) ? sanitize_key( $_POST['_nonce'] ) : '';

		if ( empty( $nonce ) || false === wp_verify_nonce( $nonce, self::$nonce ) ) {
			wp_send_json_error( esc_html__( 'Nonce validation failed. Please try again.', 'foxiz-core' ), 400 );

			wp_die();
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( esc_html__( 'Sorry, you are not allowed to perform this action.', 'foxiz-core' ), 403 );
			wp_die();
		}

		! empty( self::$token ) ? $this->refresh_token( true ) : $this->try_recover_license();

		// clear update transient
		delete_site_transient( 'cache_foxiz_theme_version' );
		delete_site_transient( 'cache_foxiz_core_version' );

		wp_send_json_success( esc_html__( 'Successful refresh data.', 'foxiz-core' ), 200 );
		wp_die();

	}

	/**
	 * Checks for theme updates modifies the update transient.
	 *
	 * This function retrieves the latest theme version from a ThemeRuby server
	 * or makes an API request if necessary. If a new version is available,
	 * it updates the transient accordingly.
	 *
	 * @param object $transient The update transient object containing theme version data.
	 *
	 * @return object The modified transient with updated theme version details.
	 */
	public function update_theme( $transient ) {

		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$ver = get_site_transient( 'cache_foxiz_theme_version' );

		if ( empty( $ver ) ) {

			$response = $this->fetch_get_data( add_query_arg( [
				'item_id' => self::$theme_id,
				'quick'   => 'true',
			], $this->get_endpoint_url( '/ruby-api/v1/latest-version' ) ) );

			if ( ! is_wp_error( $response ) && ! empty( $response ) ) {
				$ver = $response;
			} else {
				$ver = 'error';
			}
			set_site_transient( 'cache_foxiz_theme_version', $ver, 6 * HOUR_IN_SECONDS );
		}

		if ( ! empty( $ver['new_version'] ) && version_compare( $ver['new_version'], $transient->checked[ self::$theme_id ], '>' ) ) {
			$transient->response[ self::$theme_id ] = $ver;
		}

		return $transient;
	}

	/**
	 * Performs a GET request to the specified URL with optional arguments.
	 *
	 * @param string $api_url The URL to send the request to.
	 * @param array $args Optional. Additional request arguments. Default is an empty array.
	 *
	 * @return array|WP_Error The response array on success, or a WP_Error on failure.
	 */
	private function fetch_get_data( $api_url, $args = [] ) {

		$defaults = [
			'headers' => $this->get_headers(),
			'timeout' => 30,
		];

		$args             = wp_parse_args( $args, $defaults );
		$response         = wp_remote_get( esc_url_raw( $api_url ), $args );
		$response_code    = wp_remote_retrieve_response_code( $response );
		$response_message = wp_remote_retrieve_response_message( $response );

		if ( empty( $response_message ) || 200 !== (int) $response_code ) {
			return new WP_Error( $response_code, $response_message );
		} else {
			return json_decode( wp_remote_retrieve_body( $response ), true );
		}
	}

	/**
	 * Checks for FOXIZ CORE plugin updates and modifies the update transient.
	 *
	 * This function retrieves the latest plugin version from a stored transient
	 * or makes an API request if necessary. If a new version is available,
	 * it updates the transient accordingly.
	 *
	 * @param object $transient The update transient object containing plugin version data.
	 *
	 * @return object The modified transient with updated plugin version details.
	 */
	public function update_core_plugin( $transient ) {

		$ver = get_site_transient( 'cache_foxiz_core_version' );
		if ( empty( $ver ) ) {

			$response = $this->fetch_get_data( add_query_arg( [
				'item_id' => self::$theme_id . '-core',
				'quick'   => 'true',
			], $this->get_endpoint_url( '/ruby-api/v1/latest-version' ) ) );

			if ( ! is_wp_error( $response ) && ! empty( $response ) ) {
				$response['slug'] = null;
				$ver              = (object) $response;
			} else {
				$ver = 'error';
			}

			set_site_transient( 'cache_foxiz_core_version', $ver, 6 * HOUR_IN_SECONDS );
		}

		if ( ! empty( $ver->new_version ) && version_compare( $ver->new_version, FOXIZ_CORE_VERSION, '>' ) ) {
			$transient->response[ self::$core_plugin_id ] = $ver;
		} else {
			unset( $transient->response[ self::$core_plugin_id ] );
		}

		return $transient;
	}

	/**
	 * Handles the installation or activation of a recommended plugin via AJAX.
	 *
	 * This function validates the request, retrieves plugin details,
	 * checks its status, and performs the appropriate action (install/activate).
	 *
	 * @return void Outputs a JSON response indicating success or failure.
	 */
	public function recommended_plugin() {

		$nonce = ( isset( $_POST['_nonce'] ) ) ? sanitize_key( $_POST['_nonce'] ) : '';

		if ( empty( $nonce ) || false === wp_verify_nonce( $nonce, self::$nonce ) ) {
			wp_send_json_error( esc_html__( 'Nonce validation failed. Please try again.', 'foxiz-core' ), 400 );

			wp_die();
		}

		$plugin              = ( isset( $_POST['plugin'] ) ) ? trim( sanitize_text_field( $_POST['plugin'] ) ) : '';
		$recommended_plugins = $this->get_recommended_plugins();

		if ( empty( $plugin ) || empty( $recommended_plugins[ $plugin ] ) ) {
			wp_send_json_error( esc_html__( 'Invalid plugin.', 'foxiz-core' ), 400 );

			wp_die();
		}

		$plugin = $this->get_plugin_info( $plugin );
		$status = $this->plugin_status( $plugin['file'] );

		switch ( $status ) {
			case 'not_found':
				$result = $this->install_plugin( $plugin );
				if ( $result ) {
					wp_send_json_success( [
						'status'      => 'is-inactive',
						'statusLabel' => esc_html__( 'Inactive', 'foxiz-core' ),
						'btnLabel'    => esc_html__( 'Activate', 'foxiz-core' ),
					], 200 );
				} else {
					wp_send_json_error( esc_html__( 'Plugin installation failed.', 'foxiz-core' ), 400 );
				}
				break;
			case  'inactive' :
				$result = $this->activate_plugin( $plugin );
				if ( $result ) {
					wp_send_json_success( [
						'status'      => 'is-active',
						'statusLabel' => esc_html__( 'Activated', 'foxiz-core' ),
						'btnLabel'    => esc_html__( 'Deactivate', 'foxiz-core' ),
					], 200 );
				} else {
					wp_send_json_error( esc_html__( 'Plugin Activation failed.', 'foxiz-core' ), 400 );
				}
				break;
			case  'active' :
				$result = $this->deactivate_plugin( $plugin['file'] );
				if ( $result ) {
					wp_send_json_success( [
						'status'      => 'is-inactive',
						'statusLabel' => esc_html__( 'Inactive', 'foxiz-core' ),
						'btnLabel'    => esc_html__( 'Activate', 'foxiz-core' ),
					], 200 );
				} else {
					wp_send_json_error( esc_html__( 'Plugin Deactivation failed.', 'foxiz-core' ), 400 );
				}
				break;
		}

		wp_send_json_error( esc_html__( 'Invalid action!', 'foxiz-core' ), 400 );
	}

	/**
	 * Retrieves information about a specific recommended plugin.
	 *
	 * This function checks if the requested plugin exists in the recommended plugins list
	 * and returns its details if available.
	 *
	 * @return array|null The plugin information if found, otherwise null.
	 */
	public function get_plugin_info( $plugin ) {

		$recommended_plugins = $this->get_recommended_plugins();

		if ( ! empty( $recommended_plugins[ $plugin ] ) ) {
			return $recommended_plugins[ $plugin ];
		}

		return [
			'slug' => $plugin,
			'file' => $plugin . '/' . $plugin . '.php',
		];
	}

	/**
	 * Installs a WordPress plugin from the repository.
	 *
	 * This function checks user permissions, downloads the plugin from WordPress.org,
	 * and installs it using the WordPress filesystem.
	 *
	 * @param array $plugin The slug of the plugin to install.
	 *
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public function install_plugin( $plugin ) {

		if ( ! current_user_can( 'install_plugins' ) ) {
			return false;
		}

		if ( ! function_exists( 'plugins_api' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
		}

		if ( ! class_exists( 'Plugin_Upgrader', false ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		}

		if ( ! empty( $plugin['package'] ) ) {
			$package = esc_url( $plugin['package'] );
		} else {
			$plugin = ! empty( $plugin['slug'] ) ? $plugin['slug'] : (string) $plugin;
			$api    = plugins_api( 'plugin_information', [
					'slug'   => sanitize_text_field( wp_unslash( $plugin ) ),
					'fields' => [ 'sections' => false ],
				]
			);

			if ( ! is_wp_error( $api ) ) {
				$package = $api->download_link;
			}
		}

		if ( empty( $package ) ) {
			return false;
		}

		$skin   = new WP_Ajax_Upgrader_Skin();
		$u      = new Plugin_Upgrader( $skin );
		$result = $u->install( $package, [ 'overwrite_package' => true ] );

		if ( is_wp_error( $result ) || is_wp_error( $skin->result ) || is_null( $result ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Activates a given plugin.
	 *
	 * This function attempts to activate a specified plugin and returns whether the activation was successful.
	 *
	 * @param array $plugin Plugin slug or path (e.g., 'plugin-folder/plugin-file.php').
	 *
	 * @return bool True if activation is successful, fasle on failure, or false if the user lacks permission.
	 */
	public function activate_plugin( $plugin ) {

		if ( ! current_user_can( 'install_plugins' ) ) {
			return false;
		}

		$plugin = ! empty( $plugin['file'] ) ? $plugin['file'] : (string) $plugin;

		if ( ! function_exists( 'activate_plugin' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$activate = activate_plugin( $this->get_plugin_file_path( $plugin ), '', false, false );

		if ( is_wp_error( $activate ) ) {
			return false;
		}

		return true;
	}

	/**
	 * deactivate a given plugin.
	 *
	 * This function attempts to activate a specified plugin and returns whether the activation was successful.
	 *
	 * @param string $plugin Plugin slug or path (e.g., 'plugin-folder/plugin-file.php').
	 *
	 * @return bool True if activation is successful, fasle on failure, or false if the user lacks permission.
	 */
	public function deactivate_plugin( $plugin ) {

		if ( ! current_user_can( 'install_plugins' ) ) {
			return false;
		}

		if ( ! function_exists( 'deactivate_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugin = $this->get_plugin_file_path( $plugin );
		deactivate_plugins( $plugin, true );

		return true;
	}

	/**
	 * Handles the installation or activation of a recommended plugin.
	 *
	 * This function checks if the plugin is in the recommended list, retrieves its details,
	 * and based on its current status, installs, activates, or skips it.
	 *
	 * @param string $plugin The  of the plugin to process.
	 *
	 * @return false|string WP_Error message or success status.
	 */
	public function importer_plugin( $plugin ) {

		$recommended_plugins = $this->get_recommended_plugins();

		if ( empty( $plugin ) || empty( $recommended_plugins[ $plugin ] ) ) {
			return false;
		}

		$plugin = $this->get_plugin_info( $plugin );
		$title  = ! empty( $plugin['title'] ) ? esc_html( $plugin['title'] ) : esc_html( ucwords( $plugin['slug'] ) );
		$status = $this->plugin_status( $plugin['file'] );

		switch ( $status ) {
			case 'not_found':
				// translators: %s is the plugin title being installed.
				Ruby_Importer::save_import_progress( sprintf( esc_html__( 'Installing Plugin: %s', 'foxiz-core' ), $title ), 8 );
				$result = $this->install_plugin( $plugin );
				if ( $result ) {
					// translators: %s is the plugin title being activated.
					Ruby_Importer::save_import_progress( sprintf( esc_html__( 'Activating Plugin: %s', 'foxiz-core' ), $title ), 2 );
					$result = $this->activate_plugin( $plugin );
				}
				break;
			case  'inactive' :
				// translators: %s is the plugin title being activated.
				Ruby_Importer::save_import_progress( sprintf( esc_html__( 'Activating Plugin: %s', 'foxiz-core' ), $title ), 2 );
				$result = $this->activate_plugin( $plugin );
				break;
		}

		if ( empty( $result ) ) {
			return false;
		}

		return $title;
	}

	/**
	 * Retrieves information about the WordPress environment.
	 *
	 * This function returns an array containing details such as the WordPress version,
	 * debug mode status, site URL, and multisite configuration.
	 *
	 * @return array An associative array of WordPress system details.
	 */
	public function get_wp_info() {

		return [
			'wp_version'    => [
				'title' => esc_html__( 'WordPress Version', 'foxiz-core' ),
				'value' => isset( $GLOBALS['wp_version'] ) ? $GLOBALS['wp_version'] : '',
			],
			'debug_mode'    => [
				'title'   => esc_html__( 'Debug Mode', 'foxiz-core' ),
				'value'   => ( WP_DEBUG ) ? 'Enabled' : 'Disabled',
				'passed'  => ( WP_DEBUG ) ? false : true,
				'warning' => esc_html__( 'Enabling WordPress debug mode might display details about your site\'s PHP code to visitors.', 'foxiz-core' ),
			],
			'debug_log'     => [
				'title' => esc_html__( 'Debug Log', 'foxiz-core' ),
				'value' => ( WP_DEBUG_LOG ) ? 'Enabled' : 'Disabled',
			],
			'theme_name'    => [
				'title' => esc_html__( 'Theme Name', 'foxiz-core' ),
				'value' => wp_get_theme()->Name,
			],
			'theme_version' => [
				'title' => esc_html__( 'Theme Version', 'foxiz-core' ),
				'value' => wp_get_theme()->Version,
			],
			'theme_author'  => [
				'title' => esc_html__( 'Theme Author', 'foxiz-core' ),
				'value' => '<a target="_blank" href="//1.envato.market/6bEx7Q">Theme-Ruby</a>',
			],
		];
	}

}

/** LOAD */
RB_ADMIN_CORE::get_instance();