<?php

use GFPDF\Plugins\PdfForGravityView\GravityView\MockSingleRequest;
use GFPDF\Plugins\PdfForGravityView\Pdf\Renderer;
use GV\Entry_Collection;
use GV\Mocks\Legacy_Context;
use GV\Multi_Entry;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * An easy-to-use API developers can use to work with PDF for GravityView
 *
 * @since 0.8
 */
final class PDFGV_API {

	/**
	 * @var Renderer
	 * @since 0.8.0
	 * @interal Holds the initiated PDF render class
	 */
	public static $_renderer; // phpcs:ignore PSR2.Classes.PropertyDeclaration.Underscore

	/**
	 * Generate a PDF for the Single Entry Layout of a View
	 * The PDF will be saved to a temporary directory by default, but you can change $output_type.
	 * If saving, be aware that Gravity PDF automatically cleans up the temporary directory daily.
	 * To keep the PDF on disk, move/copy the file to a permanent home on the filesystem.
	 *
	 * @param int    $view_id             The GravityView ID (the Post ID)
	 * @param string $entry_id            The ID of a single Gravity Forms Entry that is valid for the View.
	 *                                    It's only valid if the entry is linked to the same form used by the View.
	 *                                    Support for GV's Multiple Forms Extension is possible by passing a string of
	 *                                    comma-separated entry IDs.
	 *                                    Note: Multiple Forms has very specific logic for joining forms together, and you
	 *                                    cannot just pass in any old set of IDs. It has to be the same set of IDs used in the View.
	 * @param string $output_type         valid values: download|display|save
	 *
	 * @return string|WP_Error
	 *
	 * @since 0.8
	 */
	public static function create_single_entry_pdf( int $view_id, $entry_id, string $output_type = 'save' ) {
		/* Ensure this isn't called too early */
		if ( doing_action( 'init' ) || ! did_action( 'init' ) ) {
			_doing_it_wrong( __CLASS__, esc_html__( 'The PDF cannot be created until after the "init" WordPress hook.', 'gk-pdf-for-gravityview' ), '0.8' );

			return new \WP_Error( 'called_too_early_in_hook_cycle' );
		}

		/* Ensure the appropriate dependencies are loaded */
		if ( ! class_exists( 'GravityView_View' ) ) {
			gravityview()->plugin->include_legacy_frontend( true );
		}

		/* Grab the requested view */
		$view = \GV\View::by_id( $view_id );
		if ( ! $view ) {
			return new \WP_Error( 'invalid_view_id' );
		}

		/* Retrieve the entries for this view */
		try {
			if ( strpos( $entry_id, ',' ) === false ) {
				$gv_entry = \GV\GF_Entry::by_id( $entry_id );
				if ( ! $gv_entry ) {
					throw new InvalidArgumentException( sprintf( 'Entry #%d not valid for View #%d', $entry_id, $view_id ) );
				}
			} else {
				$entries = array_map(
					static function ( $id ) use ( $view_id ) {
						$entry = \GV\GF_Entry::by_id( trim( $id ) );
						if ( ! $entry ) {
							throw new InvalidArgumentException( sprintf( 'Entry #%d not valid for View #%d', esc_html( $id ), esc_html( $view_id ) ) );
						}

						return $entry;
					},
					explode( ',', $entry_id )
				);

				$gv_entry = Multi_Entry::from_entries( $entries );
			}
		} catch ( \Exception $e ) {
			return new \WP_Error( 'invalid_entry_id', $e->getMessage() );
		}

		/* Get the PDF Renderer and verify the entries are valid for this view */
		$pdf_renderer = self::$_renderer;
		if ( ! $pdf_renderer->is_view_entries_valid( $view, $gv_entry ) ) {
			return new \WP_Error( 'invalid_entry_id_for_view' );
		}

		try {
			/* Backup the current view info (if any) */
			$current_request = gravityview()->request;
			$current_view    = gravityview()->views->get();
			$current_context = Legacy_Context::freeze();

			/* Prepare the View / Context for PDF rendering */
			$request = new MockSingleRequest( $view, $gv_entry );

			gravityview()->request = $request;
			gravityview()->views->set( $view );
			$entries = new Entry_Collection();
			$entries->add( $gv_entry );
			Legacy_Context::push(
				[
					'view'    => $view,
					'entries' => $entries,
					'entry'   => $gv_entry,
					'request' => $request,
				]
			);

			/* Render and save the view */
			$pdf_path = $pdf_renderer->render_pdf( $view, $gv_entry, $output_type );

			return $pdf_path;
		} catch ( \Exception $e ) {
			return new \WP_Error( 'pdf_generation_error', $e->getMessage() );
		} finally {
			/* Reset the current View */
			gravityview()->request = $current_request;
			gravityview()->views->set( $current_view );
			Legacy_Context::thaw( $current_context );
		}
	}

	/**
	 * Get the settings for a Single Entry View PDF
	 *
	 * @param int $view_id
	 *
	 * @return array|WP_Error
	 *
	 * @since 0.8.0
	 */
	public static function get_single_entry_pdf_settings( int $view_id ) {
		$view = \GV\View::by_id( $view_id );
		if ( $view === null ) {
			return new \WP_Error( 'invalid_view_id' );
		}

		return self::$_renderer->get_single_entry_pdf_settings( $view );
	}
}
