<?php

/*
 * Plugin Name:       PDF for GravityView
 * Plugin URI:        https://gravitykit.com
 * Description:       Automatically generate and/or email PDFs for your GravityView Single Entry Layouts using Gravity PDF
 * Author:            Gravity PDF
 * Author URI:        https://gravitypdf.com
 * Version:           1.2.0
 * Requires PHP:      7.3
 * Text Domain:       gk-pdf-for-gravityview
 * Domain Path:       /languages
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Update URI:        https://gravitykit.com
 */

use GFPDF\Helper\Helper_Logger;
use GFPDF\Helper\Helper_Notices;
use GFPDF\Helper\Helper_Singleton;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/* If the plugin has been activated already, deactivate the original */
if ( defined( 'GFPDF_PDF_FOR_GV_FILE' ) ) {
	deactivate_plugins( plugin_basename( GFPDF_PDF_FOR_GV_FILE ), true );

	return;
}

define( 'GFPDF_PDF_FOR_GV_FILE', __FILE__ );
define( 'GFPDF_PDF_FOR_GV_VERSION', '1.2.0' );

/**
 * Class PdfForGravityViewChecks
 *
 * @since 0.1
 */
if ( ! class_exists( 'PdfForGravityViewChecks' ) ) {
	class PdfForGravityViewChecks {

		/**
		 * Holds any blocker error messages stopping plugin running
		 *
		 * @var array
		 *
		 * @since 0.1
		 */
		private $notices = [];

		/**
		 * @var string
		 *
		 * @since 0.1
		 */
		private $required_gravitypdf_version = '6.0.0';

		/**
		 * @var string
		 *
		 * @since 0.1
		 */
		private $required_gravityview_version = '2.16';

		/**
		 * Run our pre-checks and if it passes bootstrap the plugin
		 *
		 * @return void
		 *
		 * @since 0.1
		 */
		public function init(): void {

			/* Test the minimum version requirements are met */
			$this->check_gravitypdf();
			$this->check_gravityview();

			/* Check if any errors were thrown, enqueue them and exit early */
			if ( count( $this->notices ) > 0 ) {

				/* Only display on the plugin admin page */
				global $pagenow;
				if ( is_admin() && $pagenow === 'plugins.php' ) {
					add_action( 'admin_notices', [ $this, 'display_notices' ] );
				}

				return;
			}

			add_action(
				'gfpdf_fully_loaded',
				function () {
					require_once __DIR__ . '/src/bootstrap.php';

					/* Initialize Boostrap class */
					$bootstrap = '\GFPDF\Plugins\PdfForGravityView\Bootstrap';
					$name      = 'PDF for GravityView';
					$author    = 'Blue Liquid Designs';

					/* Use Bootstrap override if GravityKit distribution */
					$gravitykit_distribution_file = __DIR__ . '/src/bootstrap-gravitykit.php';
					if ( is_file( $gravitykit_distribution_file ) ) {
						require_once $gravitykit_distribution_file;
						$bootstrap .= 'GravityKit';
						$author     = 'Gravity PDF';

						define( 'GFPDF_PDF_FOR_GV_GRAVITYKIT', true );
					}

					$pdf_plugin = new $bootstrap(
						'gk-pdf-for-gravityview',
						$name,
						$author,
						GFPDF_PDF_FOR_GV_VERSION,
						GFPDF_PDF_FOR_GV_FILE,
						\GPDFAPI::get_data_class(),
						\GPDFAPI::get_options_class(),
						new Helper_Singleton(),
						new Helper_Logger( 'gk-pdf-for-gravityview', $name ),
						new Helper_Notices()
					);

					$pdf_plugin->set_edd_download_id( '119579' );
					$pdf_plugin->set_addon_documentation_slug( 'shop-plugin-pdf-for-gravityview-add-on' );
					$pdf_plugin->init();

					/* Use the action below to access our Bootstrap class, and any singletons saved in $plugin->singleton */
					do_action( 'gfpdf_gv_bootstrapped', $pdf_plugin );
				}
			);
		}

		/**
		 * Check if Gravity PDF is compatible with this add-on
		 *
		 * @return bool
		 *
		 * @since 0.1
		 */
		public function check_gravitypdf(): bool {

			/* Check if the Gravity PDF Minimum version requirements are met */
			if ( defined( 'PDF_EXTENDED_VERSION' ) &&
				 version_compare( PDF_EXTENDED_VERSION, $this->required_gravitypdf_version, '>=' )
			) {
				return true;
			}

			/* Try get the correct slug based on the text domain */
			$slug    = 'gravity-forms-pdf-extended/pdf.php';
			$plugins = function_exists( 'get_plugins' ) ? get_plugins() : [];
			foreach ( $plugins as $plugin_slug => $plugin ) {
				if ( ( $plugin['TextDomain'] ?? '' ) === 'gravity-forms-pdf-extended' ) {
					$slug = $plugin_slug;
					break;
				}
			}

			if ( defined( 'PDF_EXTENDED_VERSION' ) ) {
				/* Installed, but running unsupported version */
				$action          = $this->get_upgrade_action( $slug );
				$this->notices[] = sprintf( __( '%1$s version %2$s or higher is required to use this add-on.', 'gk-pdf-for-gravityview' ), 'Gravity PDF', $this->required_gravitypdf_version ) . ' ' . $action;
			} elseif ( isset( $plugins[ $slug ] ) ) {
				/* Installed, but not activated */
				$action          = $this->get_activation_action( $slug );
				$this->notices[] = sprintf( __( '%s is required to use this add-on.', 'gk-pdf-for-gravityview' ), 'Gravity PDF' ) . ' ' . $action;
			} else {
				/* Not installed */
				if ( current_user_can( 'install_plugins' ) ) {
					$slug   = 'gravity-forms-pdf-extended';
					$url    = admin_url( 'update.php?action=install-plugin&plugin=' . rawurlencode( $slug ) );
					$url    = wp_nonce_url( $url, 'install-plugin_' . $slug );
					$action = sprintf( '<a href="%s">' . __( 'Install the plugin to continue.', 'gk-pdf-for-gravityview' ) . '</a>', esc_url( $url ) );
				} else {
					$url    = 'https://wordpress.org/plugins/gravity-forms-pdf-extended/';
					$action = sprintf( '<a href="%s">' . __( 'Contact your Site Administrator to install the plugin from the WordPress.org repository.', 'gk-pdf-for-gravityview' ) . '</a>', esc_url( $url ) );
				}

				$this->notices[] = sprintf( __( '%s is required to use this add-on.', 'gk-pdf-for-gravityview' ), 'Gravity PDF' ) . ' ' . $action;
			}

			return false;
		}

		/**
		 * Check if Gravity PDF is compatible with this add-on
		 *
		 * @return bool
		 *
		 * @since 0.1
		 */
		public function check_gravityview(): bool {

			/* Check if the Gravity PDF Minimum version requirements are met */
			if ( defined( 'GV_PLUGIN_VERSION' ) &&
				 version_compare( GV_PLUGIN_VERSION, $this->required_gravityview_version, '>=' )
			) {
				return true;
			}

			/* Try get the correct slug based on the text domain */
			$slug    = 'gravityview/gravityview.php';
			$plugins = function_exists( 'get_plugins' ) ? get_plugins() : [];
			foreach ( $plugins as $plugin_slug => $plugin ) {
				if ( ( $plugin['TextDomain'] ?? '' ) === 'gk-gravityview' ) {
					$slug = $plugin_slug;
					break;
				}
			}

			if ( defined( 'GV_PLUGIN_VERSION' ) ) {
				/* Installed, but running unsupported version */
				$action          = $this->get_upgrade_action( $slug );
				$this->notices[] = sprintf( __( '%1$s version %2$s or higher is required to use this add-on.', 'gk-pdf-for-gravityview' ), 'GravityView', $this->required_gravitypdf_version ) . ' ' . $action;

			} elseif ( isset( $plugins[ $slug ] ) ) {
				/* Installed, but not activated */
				$action          = $this->get_activation_action( $slug );
				$this->notices[] = sprintf( __( '%s is required to use this add-on.', 'gk-pdf-for-gravityview' ), 'GravityView' ) . ' ' . $action;
			} else {
				/* Not installed */
				$action          = '<a href="https://www.gravitykit.com/extensions/gravityview/?ref=186">' . __( 'You can purchase and download GravityView from the GravityKit website', 'gk-pdf-for-gravityview' ) . '</a>';
				$this->notices[] = sprintf( __( '%s is required to use this add-on.', 'gk-pdf-for-gravityview' ), 'GravityView' ) . ' ' . $action;
			}

			return false;
		}

		/**
		 * Create a plugin activation link with an appropriate message, if the user has permission
		 *
		 * @param string $slug
		 *
		 * @return string
		 *
		 * @since 0.5
		 */
		protected function get_activation_action( string $slug ): string {
			if ( ! current_user_can( 'activate_plugins' ) ) {
				return __( 'Contact your Site Administrator to active the plugin.', 'gk-pdf-for-gravityview' );
			}

			$url = admin_url( 'plugins.php?action=activate&plugin=' . rawurlencode( $slug ) . '&plugin_status=all&paged=1&s' );
			$url = wp_nonce_url( $url, 'activate-plugin_' . $slug );

			return sprintf( '<a href="%s">' . __( 'Activate the plugin to continue.', 'gk-pdf-for-gravityview' ) . '</a>', esc_url( $url ) );
		}

		/**
		 * Create a plugin upgrade link with an appropriate message, if the user has permission
		 *
		 * @param string $slug
		 *
		 * @return string
		 *
		 * @since 0.5
		 */
		protected function get_upgrade_action( string $slug ): string {
			if ( ! current_user_can( 'update_plugins' ) ) {
				return __( 'Contact your Site Administrator to upgrade the plugin to the latest version.', 'gk-pdf-for-gravityview' );
			}

			$url = admin_url( 'update.php?action=upgrade-plugin&plugin=' . rawurlencode( $slug ) );
			$url = wp_nonce_url( $url, 'upgrade-plugin_' . $slug );

			return sprintf( '<a href="%s">' . __( 'Upgrade the plugin to continue.', 'gk-pdf-for-gravityview' ) . '</a>', esc_url( $url ) );
		}

		/**
		 * Helper function to easily display error messages
		 *
		 * @return void
		 *
		 * @since 0.1
		 */
		public function display_notices() {
			?>
			<div class="error">
				<p>
					<strong><?php esc_html_e( 'PDF for GravityView Installation Problem', 'gk-pdf-for-gravityview' ); ?></strong>
				</p>

				<ul style="padding-bottom: 0.5em">
					<?php foreach ( $this->notices as $notice ): ?>
						<li style="padding-left: 20px;list-style: inside"><?php echo wp_kses_post( $notice ); ?></li>
					<?php endforeach; ?>
				</ul>
			</div>
			<?php
		}
	}
}

/* Initialise the software */
add_action(
	'plugins_loaded',
	function () {
		( new PdfForGravityViewChecks() )->init();
	}
);

/* Delete the PDF Email GV form (if exists) */
register_deactivation_hook(
	GFPDF_PDF_FOR_GV_FILE,
	function () {
		if ( ! class_exists( 'GFAPI' ) || ! get_option( 'gfpdf_for_gv_pdf_email_form_id' ) ) {
			return;
		}

		\GFAPI::delete_form( get_option( 'gfpdf_for_gv_pdf_email_form_id' ) );
		delete_option( 'gfpdf_for_gv_pdf_email_form_id' );
	}
);
