<?php

namespace GFPDF\Plugins\PdfForGravityView\GravityView\Fields;

use GFPDF\Exceptions\GravityPdfDomainException;
use Psr\Log\LoggerInterface;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since 0.4
 */
class PdfEmail extends \GravityView_Field {

	protected $rendered = false;

	/**
	 * The name of the GravityView field type
	 * Example: `created_by`, `text`, `fileupload`, `address`, `entry_link`
	 *
	 * @var string
	 * @since 0.4
	 */
	public $name = 'pdf_email';

	/**
	 * The contexts in which a field is available. Some fields aren't editable, for example.
	 * - `singular` is an alias for both `single` and `edit`
	 * - `multiple` is an alias for `directory` (backward compatibility)
	 *
	 * @type array
	 * @since 0.4
	 */
	public $contexts = [ 'single' ];

	/**
	 * @type boolean Can the field be sorted in search?
	 * @since 0.4
	 */
	public $is_sortable = false;

	/**
	 * @type boolean Can the field be searched?
	 * @since 0.4
	 */
	public $is_searchable = false;

	/**
	 * `standard`, `advanced`, `post`, `pricing`, `meta`, `gravityview`, or `add-ons`
	 *
	 * @since 0.4
	 * @type string The group belongs to this field in the field picker
	 */
	public $group = 'add-ons';

	/**
	 * @var string An icon that represents the field type in the field picker.
	 *
	 * Supports these icon formats:
	 * - Gravity Forms icon class: The string starts with "gform-icon". Note: the site must be running GF 2.5+. No need to also pass "gform-icon".
	 * - Dashicons: The string starts with "dashicons". No need to also pass "dashicons".
	 * - Inline SVG: Starts with "data:"
	 * - If not matching those formats, the value will be used as a CSS class in a `<i>` element.
	 *
	 * @since 0.4
	 */
	public $icon = 'dashicons-pdf';

	/**
	 * @var LoggerInterface
	 * @since 0.8
	 */
	protected $log;

	/**
	 * @since 0.4
	 */
	public function __construct( LoggerInterface $log ) {
		$this->log = $log;

		$this->label       = esc_html__( 'Share Single Entry PDF', 'gk-pdf-for-gravityview' );
		$this->description = esc_html__( 'Sends the PDF as an email notification attachment.', 'gk-pdf-for-gravityview' );

		$this->add_hooks();

		parent::__construct();
	}

	/**
	 * @since 0.4
	 */
	protected function add_hooks() {
		add_filter( 'gravityview_entry_default_fields', [ $this, 'remove_pdf_email_field_from_directory' ], 10, 3 );
		add_filter( 'gravityview/field_output/args', [ $this, 'prevent_wpautop_on_field' ], 10, 3 );

		$form_id = get_option( 'gfpdf_for_gv_pdf_email_form_id' );

		/* Hide the primary validation message */
		add_filter( 'gform_form_validation_errors_markup_' . $form_id, '__return_empty_string' );

		/* Disable the view counter */
		add_filter( 'gform_disable_view_counter_' . $form_id, '__return_true' );

		/* Enable the Cc setting to prevent a deprecation notice in PHP8.1 */
		add_filter( 'gform_notification_enable_cc_' . $form_id, '__return_true' );

		/* Hide from list view */
		add_filter( 'gform_form_list_forms', [ $this, 'hide_form_in_list_view' ] );
		add_filter( 'gform_export_forms_forms', [ $this, 'hide_form_in_list_view' ] );
		add_filter( 'gform_form_switcher_forms', [ $this, 'hide_form_in_list_view' ] );
		add_filter( 'gk/gravityview/common/get_forms', [ $this, 'hide_form_in_list_view' ] );
		add_filter( 'gform_form_list_count', [ $this, 'fix_form_count' ] );
		add_action( 'admin_init', [ $this, 'redirect_on_form_access' ] );

		/* Override the labels, confirmation, and notification */
		add_filter( 'gform_pre_validation_' . $form_id, [ $this, 'populate_form' ] );
		add_filter( 'gform_pre_submission_filter_' . $form_id, [ $this, 'populate_form' ] );

		/* Generate the PDF */
		add_filter( 'gform_entry_post_save_' . $form_id, [ $this, 'save_view_pdf' ], 10, 2 );

		/* Attach PDF to Notification */
		add_filter( 'gform_notification_' . $form_id, [ $this, 'attach_view_pdf' ], 10, 3 );

		/* Delete entry after submission */
		add_action( 'gform_after_submission_' . $form_id, [ $this, 'cleanup_entry' ] );
	}

	/**
	 * Get the PDF Email settings from the View and populate the associated form during validation and form submission
	 *
	 * @param array $form
	 *
	 * @return array
	 *
	 * @since 0.4
	 */
	public function populate_form( $form ) {
		if ( ! rgpost( 'gform_submit' ) || (int) rgpost( 'gform_submit' ) !== (int) get_option( 'gfpdf_for_gv_pdf_email_form_id' ) ) {
			return $form;
		}

		$view_id = $this->decode_id( rgpost( 'input_5' ) );
		$nonce   = rgpost( 'input_6' );

		/* Add extra nonce check */
		if ( ! wp_verify_nonce( $nonce, 'gv-pdf-email-nonce-' . $view_id ) ) {
			$_POST['input_5'] = ''; // don't store the View ID if not valid

			return $form;
		}

		$view = \GV\View::by_id( $view_id );
		if ( ! $view ) {
			$_POST['input_5'] = ''; // don't store the View ID if not valid

			return $form;
		}

		/* Grab the first GV PDF Email field settings */
		$configuration = [];
		foreach ( $view->fields->all() as $field ) {
			if ( $field->ID !== 'pdf_email' ) {
				continue;
			}

			/* Verify the current user can view this field */
			if ( ! $field->is_visible( $view ) ) {
				$_POST['input_5'] = ''; // don't store the View ID if not valid

				return $form;
			}

			$configuration = $field->as_configuration();
		}

		if ( empty( $configuration ) ) {
			$_POST['input_5'] = ''; // don't store the View ID if not valid

			return $form;
		}

		/* Get the View Form and Entry */
		gravityview()->views->set( $view );

		$gv_form  = $view->form->form;
		$gv_entry = gravityview()->request->is_entry();
		$entry    = $gv_entry ? $gv_entry->as_entry() : false;
		if ( ! $entry ) {
			$_POST['input_5'] = ''; // don't store the View ID if not valid

			return $form;
		}

		/* Override labels */
		$form['description']        = \GFCommon::replace_variables( $configuration['pdf_email_instructions'] ?? '', $gv_form, $entry, false, false );
		$form['fields'][0]['label'] = \GFCommon::replace_variables( $configuration['pdf_email_email_label'] ?? '', $gv_form, $entry );
		$form['button']['text']     = \GFCommon::replace_variables( $configuration['pdf_email_submit_label'] ?? '', $gv_form, $entry );

		/* Override confirmation */
		$confirmation_id                                      = array_key_first( $form['confirmations'] );
		$form['confirmations'][ $confirmation_id ]['message'] = wp_kses_post( wpautop( \GFCommon::replace_variables( $configuration['pdf_email_confirmation_message'] ?? '', $gv_form, $entry, false, false, false ) ) );
		$form['confirmations'][ $confirmation_id ]['disableAutoformat'] = true;

		/* Override notification */
		$notification_id = array_key_first( $form['notifications'] );

		$form['notifications'][ $notification_id ]['isActive'] = true;
		$form['notifications'][ $notification_id ]['fromName'] = \GFCommon::replace_variables( $configuration['pdf_email_notification_from_name'] ?? '', $gv_form, $entry, false, false, false, 'text' );
		$form['notifications'][ $notification_id ]['from']     = \GFCommon::replace_variables( $configuration['pdf_email_notification_from_email'] ?? '', $gv_form, $entry, false, false, false, 'text' );
		$form['notifications'][ $notification_id ]['subject']  = \GFCommon::replace_variables( $configuration['pdf_email_notification_subject'] ?? '', $gv_form, $entry, false, false, false, 'text' );

		/* Convert {gv_entry_link} to an a direct link to the post/view using a short site URL for the text */
		$post_id = $this->decode_id( rgpost( 'input_7' ) );
		/* Use the Post ID if available (uses the URL the shortcode is embedded on) */
		add_filter( 'gravityview/entry_link/add_query_args', '__return_false' );
		$entry_link = gv_entry_link( $entry, $post_id ?: $view_id );
		remove_filter( 'gravityview/entry_link/add_query_args', '__return_false' );

		$gv_entry_link = sprintf( '<a href="%s">%s</a>', esc_url( $entry_link ), str_replace( [ 'http://', 'https://' ], '', site_url() ) );

		$message = $configuration['pdf_email_notification_message'] ?? '';
		$message = str_replace( '{gv_entry_link}', $gv_entry_link, $message );
		$message = \GFCommon::replace_variables( $message, $gv_form, $entry, false, false, false );
		$message = do_shortcode( $message );

		$form['notifications'][ $notification_id ]['message'] = wp_kses_post( $message );

		return $form;
	}

	/**
	 * After form submission, generate the PDF and save to a temporary file
	 *
	 * @param array $entry
	 *
	 * @return array
	 *
	 * @since 0.4
	 */
	public function save_view_pdf( $entry ) {
		try {
			/* Generate the PDF and store in the entry temporarily */
			$view_id  = $this->decode_id( $entry[5] );
			$entry_id = $this->decode_id( $entry[8] );

			$pdf_path = \PDFGV_API::create_single_entry_pdf( $view_id, $entry_id );
			if ( is_wp_error( $pdf_path ) ) {
				throw new GravityPdfDomainException( $pdf_path->get_error_message(), $pdf_path->get_error_code() );
			}

			/* Store the path in the entry object */
			$entry['gv_pdf_path'] = $pdf_path;
		} catch ( \Exception $e ) {
			$this->log->error(
				'PDF Generation Error',
				[
					'code'    => $e->getCode(),
					'message' => $e->getMessage(),
				]
			);
		}

		return $entry;
	}

	/**
	 * Attach GV PDF to Notification email. Disable notification if PDF is invalid
	 *
	 * @param array $notification
	 * @param array $form
	 * @param array $entry
	 *
	 * @return array
	 *
	 * @since 0.4
	 */
	public function attach_view_pdf( $notification, $form, $entry ) {
		if ( empty( $entry['gv_pdf_path'] ) || ! is_file( $entry['gv_pdf_path'] ) ) {
			/* Disable notification */
			$notification['subject'] = '';
			$notification['message'] = '';
		}

		$notification['attachments'] = [ $entry['gv_pdf_path'] ];

		return $notification;
	}

	/**
	 * Delete the created entry after submission
	 *
	 * @param array $entry
	 *
	 * @return void
	 *
	 * @since 0.4
	 */
	public function cleanup_entry( $entry ) {
		\GFAPI::delete_entry( $entry['id'] );
	}

	/**
	 * Remove the PDF Email Gravity Form from the form list view
	 *
	 * @param array $forms
	 *
	 * @return array
	 *
	 * @since 0.4
	 */
	public function hide_form_in_list_view( $forms ) {
		foreach ( $forms as $key => $form ) {
			$id = is_array( $form ) ? $form['id'] : $form->id;

			if ( (int) $id === (int) get_option( 'gfpdf_for_gv_pdf_email_form_id' ) ) {
				unset( $forms[ $key ] );
				break;
			}
		}

		return $forms;
	}

	/**
	 * Update the total and active form count so the PDF Email form is disregarded
	 *
	 * @param array $form_count
	 *
	 * @return array
	 *
	 * @since 0.4
	 */
	public function fix_form_count( $form_count ) {
		--$form_count['total'];
		--$form_count['active'];

		return $form_count;
	}

	/**
	 * Prevent direct access to the PDF Email Gravity Form in the admin area
	 *
	 * @return void
	 *
	 * @since 0.4
	 */
	public function redirect_on_form_access() {
		if ( \GFForms::is_gravity_page() && (int) rgget( 'id' ) === (int) get_option( 'gfpdf_for_gv_pdf_email_form_id' ) ) {
			wp_safe_redirect( admin_url( 'admin.php?page=gf_edit_forms' ) );
		}
	}

	/**
	 * If displaying Multi Entries Layout then remove PDF Email field from list
	 *
	 * @param array  $fields
	 * @param        $form
	 * @param string $zone
	 *
	 * @return array
	 */
	public function remove_pdf_email_field_from_directory( $fields, $form, $zone ) {
		if ( $zone === 'directory' && isset( $fields['pdf_email'] ) ) {
			unset( $fields['pdf_email'] );
		}

		return $fields;
	}

	/**
	 * Force `wpautop` to be disabled for this field
	 *
	 * @param array                $args
	 * @param array                $passed_args
	 * @param \GV\Template_Context $context
	 *
	 * @return array
	 *
	 * @since 0.4
	 */
	public function prevent_wpautop_on_field( $args, $passed_args, $context ) {
		if ( $context !== null && $context->field->ID === 'pdf_email' ) {
			$args['wpautop'] = false;

			/* If this field is already rendered, disable a second time */
			if ( $this->rendered ) {
				$args['value'] = '';
			}

			$this->rendered = true;
		}

		return $args;
	}

	/**
	 * Add settings to PDF Email field
	 *
	 * @param array  $field_options
	 * @param string $template_id
	 * @param string $field_id
	 * @param string $context
	 * @param string $input_type
	 * @param int    $form_id
	 *
	 * @return array
	 *
	 * @since 0.4
	 */
	public function field_options( $field_options, $template_id, $field_id, $context, $input_type, $form_id ) {
		$field_options['show_label']['value']   = true;
		$field_options['custom_label']['value'] = __( 'Share this Listing', 'gk-pdf-for-gravityview' );

		unset( $field_options['pdf_exclude'], $field_options['pdf_only'] );

		$add_options = [];

		$add_options['pdf_email_html_confirmation_settings'] = [
			'type'  => 'html',
			'desc'  => '<strong>' . __( 'Confirmation', 'gk-pdf-for-gravityview' ) . '</strong>',
			'group' => 'display',
		];

		$add_options['pdf_email_confirmation_message'] = [
			'type'       => 'textarea',
			'label'      => __( 'Message', 'gk-pdf-for-gravityview' ),
			'value'      => __( 'Thanks! An email has been sent to the address provided.', 'gk-pdf-for-gravityview' ),
			'merge_tags' => true,
			'group'      => 'display',
			'rows'       => 4,
		];

		$add_options['pdf_email_html_notification_settings'] = [
			'type'  => 'html',
			'desc'  => '<strong>' . __( 'Notification', 'gk-pdf-for-gravityview' ) . '</strong>',
			'group' => 'display',
		];

		$add_options['pdf_email_notification_from_name'] = [
			'type'       => 'text',
			'label'      => __( 'From Name', 'gk-pdf-for-gravityview' ),
			'desc'       => null,
			'tooltip'    => __( 'Enter the name you would like the notification email sent from.', 'gk-pdf-for-gravityview' ),
			'value'      => '',
			'merge_tags' => true,
			'group'      => 'display',
		];

		$add_options['pdf_email_notification_from_email'] = [
			'type'       => 'text',
			'label'      => __( 'From Email', 'gk-pdf-for-gravityview' ),
			'desc'       => null,
			'tooltip'    => __( 'Enter an authorized email address you would like the notification email sent from. To avoid deliverability issues, always use your site domain in the from email.', 'gk-pdf-for-gravityview' ),
			'value'      => '{admin_email}',
			'merge_tags' => true,
			'group'      => 'display',
		];

		$add_options['pdf_email_notification_subject'] = [
			'type'       => 'text',
			'label'      => __( 'Subject', 'gk-pdf-for-gravityview' ),
			'desc'       => null,
			'value'      => __( 'A PDF has been shared with you', 'gk-pdf-for-gravityview' ),
			'merge_tags' => true,
			'group'      => 'display',
		];

		$add_options['pdf_email_notification_message'] = [
			'type'       => 'textarea',
			'label'      => __( 'Message', 'gk-pdf-for-gravityview' ),
			'value'      => __( 'A PDF has been shared with you from {gv_entry_link} and is attached.', 'gk-pdf-for-gravityview' ),
			'merge_tags' => true,
			'group'      => 'display',
			'rows'       => 4,
		];

		$add_options['pdf_email_instructions'] = [
			'type'       => 'textarea',
			'label'      => __( 'Instructions', 'gk-pdf-for-gravityview' ),
			'value'      => __( 'Enter an email address and a PDF of this listing will be sent as an attachment.', 'gk-pdf-for-gravityview' ),
			'merge_tags' => true,
			'group'      => 'label',
			'rows'       => 4,
		];

		$add_options['pdf_email_html_field_settings'] = [
			'type'  => 'html',
			'desc'  => '<strong>' . __( 'Fields', 'gk-pdf-for-gravityview' ) . '</strong>',
			'group' => 'label',
		];

		$add_options['pdf_email_email_label'] = [
			'type'       => 'text',
			'label'      => __( 'Email Field Label', 'gk-pdf-for-gravityview' ),
			'desc'       => null,
			'value'      => __( 'Email', 'gk-pdf-for-gravityview' ),
			'merge_tags' => false,
			'group'      => 'label',
		];

		$add_options['pdf_email_submit_label'] = [
			'type'       => 'text',
			'label'      => __( 'Submit Button Label', 'gk-pdf-for-gravityview' ),
			'desc'       => null,
			'value'      => __( 'Share', 'gk-pdf-for-gravityview' ),
			'merge_tags' => false,
			'group'      => 'label',
		];

		return $add_options + $field_options;
	}

	/**
	 * @param string $string
	 *
	 * @return string
	 *
	 * @since 0.4
	 */
	protected function decode_id( string $text ): string {
		return substr( \GFCommon::openssl_decrypt( $text ), 8 );
	}
}
