<?php

namespace GFPDF\Plugins\PdfForGravityView\GravityView\Fields;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since 0.1
 */
class PdfLink extends \GravityView_Field {

	/**
	 * The name of the GravityView field type
	 * Example: `created_by`, `text`, `fileupload`, `address`, `entry_link`
	 *
	 * @var string
	 * @since 0.1
	 */
	public $name = 'pdf_link';

	/**
	 * The contexts in which a field is available. Some fields aren't editable, for example.
	 * - `singular` is an alias for both `single` and `edit`
	 * - `multiple` is an alias for `directory` (backward compatibility)
	 * @type array
	 * @since 0.1
	 */
	public $contexts = [ 'multiple', 'directory', 'single' ];

	/**
	 * @type boolean Can the field be sorted in search?
	 * @since 0.1
	 */
	public $is_sortable = false;

	/**
	 * @type boolean Can the field be searched?
	 * @since 0.1
	 */
	public $is_searchable = false;

	/**
	 * `standard`, `advanced`, `post`, `pricing`, `meta`, `gravityview`, or `add-ons`
	 *
	 * @since 0.1
	 * @type string The group belongs to this field in the field picker
	 */
	public $group = 'add-ons';

	/**
	 * @var string An icon that represents the field type in the field picker.
	 *
	 * Supports these icon formats:
	 * - Gravity Forms icon class: The string starts with "gform-icon". Note: the site must be running GF 2.5+. No need to also pass "gform-icon".
	 * - Dashicons: The string starts with "dashicons". No need to also pass "dashicons".
	 * - Inline SVG: Starts with "data:"
	 * - If not matching those formats, the value will be used as a CSS class in a `<i>` element.
	 *
	 * @since 0.1
	 */
	public $icon = 'dashicons-pdf';

	/**
	 * @since 0.1
	 */
	public function __construct() {
		$this->label       = esc_html__( 'Link to Single Entry PDF', 'gk-pdf-for-gravityview' );
		$this->description = esc_html__( 'A dedicated link to a PDF of the Single Entry.', 'gk-pdf-for-gravityview' );

		parent::__construct();
	}

	/**
	 * Add options to PDF Link field
	 *
	 * @param array  $field_options
	 * @param string $template_id
	 * @param string $field_id
	 * @param string $context
	 * @param string $input_type
	 * @param int    $form_id
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	public function field_options( $field_options, $template_id, $field_id, $context, $input_type, $form_id ) {

		/* Always a link */
		unset( $field_options['show_as_link'], $field_options['search_filter'] );

		$field_options['show_label']['value']   = true;
		$field_options['custom_label']['value'] = __( 'PDF', 'gk-pdf-for-gravityview' );

		$add_options                  = [];
		$add_options['pdf_link_text'] = [
			'type'       => 'text',
			'label'      => __( 'Link Text:', 'gk-pdf-for-gravityview' ),
			'desc'       => null,
			'value'      => __( 'View PDF', 'gk-pdf-for-gravityview' ),
			'merge_tags' => true,
			'group'      => 'display',
		];

		$add_options['pdf_link_action'] = [
			'type'  => 'checkbox',
			'label' => __( 'Force download?', 'gk-pdf-for-gravityview' ),
			'desc'  => __( 'Instead of opening the PDF in the browser, a save dialog box will open / the document will be downloaded automatically (depending on the browser settings).', 'gk-pdf-for-gravityview' ),
			'value' => '',
			'group' => 'display',
		];

		$add_options['pdf_link_expiration'] = [
			'type'       => 'text',
			'label'      => __( 'Link Expiration', 'gk-pdf-for-gravityview' ),
			'desc'       => __( 'For security, all links are signed and expire after the specified time. Valid options include: 30 minutes, 1 hour, 2 days etc.', 'gk-pdf-for-gravityview' ),
			'value'      => '1 day',
			'merge_tags' => false,
			'group'      => 'advanced',
		];

		$this->add_field_support( 'new_window', $field_options );

		return $add_options + $field_options;
	}
}
