<?php

namespace GFPDF\Plugins\PdfForGravityView\GravityView;

use GFPDF\Helper\Helper_Trait_Logger;
use GFPDF\Plugins\PdfForGravityView\GravityView\Fields\Exclusions;
use GFPDF\Plugins\PdfForGravityView\GravityView\Fields\PdfEmail;
use GFPDF\Plugins\PdfForGravityView\GravityView\Fields\PdfLink;
use GFPDF\Plugins\PdfForGravityView\Pdf\Renderer;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since 0.1
 */
class Registration {
	use Helper_Trait_Logger;

	/**
	 * @var array<int,array> A cache of the default PDF settings for a specific view
	 */
	protected $default_settings = [];

	/**
	 * @since 0.1
	 */
	public function init() {
		$this->register_view_fields();

		add_action( 'admin_enqueue_scripts', [ $this, 'add_scripts' ], 1000 );
		add_filter( 'gravityview/template/fields_template_paths', [ $this, 'register_field_template_path' ] );
		add_filter( 'gravityview_template_field_options', [ $this, 'register_single_template_field_options' ], 10, 5 );
		add_filter( 'gravityview/metaboxes/default', [ $this, 'register_pdf_metabox' ] );
		add_filter( 'gravityview/view/settings/defaults', [ $this, 'register_pdf_metabox_settings' ] );
		add_filter( 'gravityview/setting/class/rich_textarea', [ $this, 'register_rich_textarea_field' ] );
		add_filter( 'gravityview/setting/class/color_picker', [ $this, 'register_color_picker_field' ] );
		add_filter( 'gravityview/setting/class/upload', [ $this, 'register_upload_field' ] );
		add_filter( 'gravityview/setting/class/code_editor', [ $this, 'register_code_editor_field' ] );
		add_filter( 'gravityview/field/is_visible', [ $this, 'hide_pdf_only_fields' ], 10, 2 );
	}

	/**
	 * Register new GravityView layout fields
	 *
	 * @return void
	 *
	 * @since 0.1
	 */
	protected function register_view_fields(): void {
		new PdfLink();
		new PdfEmail( $this->get_logger() );

		$form_id = get_option( 'gfpdf_for_gv_pdf_email_form_id', 0 );
		$form    = \GFAPI::get_form( $form_id );

		/* Import the Gravity Form needed for the PDF Email field */
		if ( ! $form_id || ! $form ) {
			$form_id = \GFAPI::add_form( wp_json_file_decode( __DIR__ . '/../../assets/forms/pdf-email-form.json', [ 'associative' => true ] ) );

			if ( ! is_wp_error( $form_id ) ) {
				update_option( 'gfpdf_for_gv_pdf_email_form_id', $form_id );
			}
		} elseif ( isset( $form['is_active'] ) && ! $form['is_active'] ) {
			/* If the form state somehow got set to inactive, reset back to active */
			\GFAPI::update_form_property( $form_id, 'is_active', true );
		} elseif ( isset( $form['is_trash'] ) && $form['is_trash'] ) {
			/* If the form state somehow got set to trashed, reset back to active */
			\GFAPI::update_form_property( $form_id, 'is_trash', false );
		}
	}

	/**
	 * Register the Javascript required for the PDF settings on the Single View Layout admin page
	 *
	 * @param string $hook
	 *
	 * @return void
	 *
	 * @since 0.1
	 */
	public function add_scripts( $hook ) {
		if ( ! gravityview()->request->is_admin( $hook, 'single' ) ) {
			return;
		}

		wp_enqueue_media();
		wp_enqueue_editor();
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker' );

		wp_enqueue_style( 'gfpdf_gv_admin', plugins_url( 'assets/css/admin.css', GFPDF_PDF_FOR_GV_FILE ), [], GFPDF_PDF_FOR_GV_VERSION );

		wp_enqueue_script( 'gfpdf_gv_upload_field', plugins_url( 'assets/js/admin-upload-field.js', GFPDF_PDF_FOR_GV_FILE ), [ 'gravityview_views_scripts' ], GFPDF_PDF_FOR_GV_VERSION, true );

		wp_localize_script(
			'gfpdf_gv_upload_field',
			'gfpdfGvGlobals',
			[
				'mediaLabel' => esc_attr__( 'Choose File', 'gk-pdf-for-gravityview' ),
			]
		);
	}

	/**
	 * Register a GravityView template directory for the custom view(s)/field(s) registered
	 *
	 * @param array $paths
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	public function register_field_template_path( $paths ) {
		/* Ensure the priority key is unique */
		$key = 150;
		while ( isset( $paths[ $key ] ) ) {
			++$key;
		}

		$paths[ $key ] = __DIR__ . '/Templates';

		return $paths;
	}

	/**
	 * Register additional field(s) that can be displayed in the Single View Layout field settings
	 *
	 * @param array  $field_options Array of field options with `label`, `value`, `type`, `default` keys
	 * @param string $template_id   Table slug
	 * @param float  $field_id      GF Field ID - Example: `3`, `5.2`, `entry_link`, `created_by`
	 * @param string $context       What context are we in? Example: `single` or `directory`
	 * @param string $input_type    (textarea, list, select, etc.)
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	public function register_single_template_field_options( $field_options, $template_id, $field_id, $context, $input_type ) {
		if ( $context !== 'single' || in_array( $input_type, Exclusions::INTERNAL_FIELDS, true ) ) {
			return $field_options;
		}

		$field_options['pdf_exclude'] = [
			'type'     => 'checkbox',
			'label'    => __( 'Exclude from PDF?', 'gk-pdf-for-gravityview' ),
			'value'    => '',
			'priority' => 4200,
			'group'    => 'visibility',
		];

		$field_options['pdf_only'] = [
			'type'     => 'checkbox',
			'label'    => __( 'Only include in PDF (exclude from View)?', 'gk-pdf-for-gravityview' ),
			'value'    => '',
			'priority' => 4200,
			'group'    => 'visibility',
		];

		return $field_options;
	}

	/**
	 * Register a PDF metabox that will display new Single Entry PDF settings in the View admin area
	 *
	 * @param array $metaboxes
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	public function register_pdf_metabox( $metaboxes ) {
		$metaboxes[] = [
			'id'            => 'pdf',
			'title'         => __( 'Single Entry PDF', 'gk-pdf-for-gravityview' ),
			'file'          => __DIR__ . '/Templates/metabox-pdf.php',
			'icon-class'    => 'dashicons-pdf',
			'callback'      => '',
			'callback_args' => '',
		];

		return $metaboxes;
	}

	/**
	 * Register a Rich Text field used in the PDF metabox
	 *
	 * @param string $class_name
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	public function register_rich_textarea_field( $class_name ) {
		if ( ! class_exists( '\GravityView_FieldType_textarea' ) ) {
			require_once GRAVITYVIEW_DIR . 'includes/admin/field-types/type_textarea.php';
		}

		return class_exists( $class_name ) ? $class_name : 'GFPDF\Plugins\PdfForGravityView\GravityView\Fields\RichTextarea';
	}

	/**
	 * Register a Color Picker field used in the PDF metabox
	 *
	 * @param string $class_name
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	public function register_color_picker_field( $class_name ) {
		if ( ! class_exists( '\GravityView_FieldType_text' ) ) {
			require_once GRAVITYVIEW_DIR . 'includes/admin/field-types/type_text.php';
		}

		return class_exists( $class_name ) ? $class_name : 'GFPDF\Plugins\PdfForGravityView\GravityView\Fields\ColorPicker';
	}

	/**
	 * Register an Upload field used in the PDF metabox
	 *
	 * @param string $class_name
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	public function register_upload_field( $class_name ) {
		if ( ! class_exists( '\GravityView_FieldType_text' ) ) {
			require_once GRAVITYVIEW_DIR . 'includes/admin/field-types/type_text.php';
		}

		return class_exists( $class_name ) ? $class_name : 'GFPDF\Plugins\PdfForGravityView\GravityView\Fields\Upload';
	}

	/**
	 * Register an Code Editor field used in the PDF metabox
	 *
	 * @param string $class_name
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	public function register_code_editor_field( $class_name ) {
		if ( ! class_exists( '\GravityView_FieldType_textarea' ) ) {
			require_once GRAVITYVIEW_DIR . 'includes/admin/field-types/type_textarea.php';
		}

		return class_exists( $class_name ) ? $class_name : 'GFPDF\Plugins\PdfForGravityView\GravityView\Fields\CodeEditor';
	}

	/**
	 * Remove fields from a view if configured as "PDF Only"
	 *
	 * @param bool      $visible Whether the current field should be visible or not in a View
	 * @param \GV\Field $field
	 *
	 * @return bool
	 *
	 * @since 0.4
	 */
	public function hide_pdf_only_fields( $visible, $field ) {
		/* Skip if rendering the PDF */
		if ( Renderer::is_pdf_endpoint() ) {
			return $visible;
		}

		/* Hide any GV fields marked as PDF Only */
		$field_settings = $field->as_configuration();

		if ( ! empty( $field_settings['pdf_only'] ) ) {
			return false;
		}

		return $visible;
	}

	/**
	 * Register all the PDF metabox settings
	 *
	 * @param array $settings
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	public function register_pdf_metabox_settings( $settings ) {
		global $post;

		$view_id = $post->ID ?? 0;
		if ( ! empty( $this->default_settings[ $view_id ] ) ) {
			return $settings + $this->default_settings[ $view_id ];
		}

		$options = \GPDFAPI::get_options_class();

		$fonts       = $this->flatten_options_groups( $options->get_installed_fonts() );
		$paper_sizes = $this->flatten_options_groups( $options->get_paper_size() );
		unset( $paper_sizes['CUSTOM'] );

		/* Get the current form notifications */
		$notifications = [];
		$form_id       = gravityview_get_form_id( $view_id );
		$form          = ! empty( $form_id ) ? gravityview_Get_form( $form_id ) : false;
		if ( isset( $form['notifications'] ) ) {
			foreach ( $form['notifications'] as $notification ) {
				$notifications[ $notification['id'] ] = [ 'label' => $notification['name'] ];
			}
		}

		$pdf_settings = apply_filters(
			'gfpdf_gv_pdf_metabox_settings',
			[
				'gpdf_filename'         => [
					'label'             => __( 'Filename', 'gk-pdf-for-gravityview' ),
					'tooltip'           => __( 'Set the filename for the generated PDF (excluding the .pdf extension). If left blank the View name will be used. Invalid characters <code>/ \ " * ? | : &lt; &gt;</code> are converted to an underscore.', 'gk-pdf-for-gravityview' ),
					'group'             => 'pdf',
					'type'              => 'text',
					'value'             => '',
					'placeholder'       => $post->post_title ?? '',
					'show_in_shortcode' => false,
					'merge_tags'        => 'force',
				],

				'gpdf_size'             => [
					'label'             => __( 'Paper Size', 'gk-pdf-for-gravityview' ),
					'tooltip'           => __( 'Set the paper size used when generating PDFs.', 'gk-pdf-for-gravityview' ),
					'type'              => 'select',
					'group'             => 'pdf',
					'options'           => $paper_sizes,
					'value'             => $options->get_option( 'default_pdf_size', 'A4' ),
					'show_in_shortcode' => false,
				],

				'gpdf_orientation'      => [
					'label'             => esc_html__( 'Paper Orientation', 'gk-pdf-for-gravityview' ),
					'type'              => 'select',
					'group'             => 'pdf',
					'options'           => [
						'portrait'  => __( 'Portrait', 'gk-pdf-for-gravityview' ),
						'landscape' => __( 'Landscape', 'gk-pdf-for-gravityview' ),
					],
					'value'             => 'portrait',
					'show_in_shortcode' => false,
				],

				'gpdf_page_margin'      => [
					'label'             => __( 'Page Margin', 'gk-pdf-for-gravityview' ),
					'type'              => 'text',
					'tooltip'           => __(
						'You may specify the margin using one, two, three, or four values (much like CSS shorthand). The first value is the top margin, the second is the right, the third is the bottom, and the forth is the left margin eg. <code>20mm</code>, <code>1em 2em</code>, <code>1in 1.5in 3in</code>, <code>50px 100px 75px 25px</code>.',
						'gk-pdf-for-gravityview'
					),
					'group'             => 'pdf',
					'value'             => '20mm',
					'show_in_shortcode' => false,
				],

				'gpdf_font'             => [
					'label'             => __( 'Font', 'gk-pdf-for-gravityview' ),
					'type'              => 'select',
					'tooltip'           => __(
						'Select the font to use in the PDF. You can install additional fonts by navigating to Forms -> Settings -> PDF -> Tools in your admin area.',
						'gk-pdf-for-gravityview'
					),
					'group'             => 'pdf',
					'options'           => $fonts,
					'value'             => $options->get_option( 'default_font' ),
					'show_in_shortcode' => false,
				],

				'gpdf_font_size'        => [
					'label'             => __( 'Font Size', 'gk-pdf-for-gravityview' ),
					'type'              => 'number',
					'group'             => 'pdf',
					'value'             => $options->get_option( 'default_font_size', 10 ),
					'show_in_shortcode' => false,
				],

				'gpdf_font_color'       => [
					'label'             => __( 'Font Color', 'gk-pdf-for-gravityview' ),
					'type'              => 'color_picker',
					'desc'              => __( 'Set the font color to use in the PDF.', 'gk-pdf-for-gravityview' ),
					'group'             => 'pdf',
					'value'             => $options->get_option( 'default_font_colour', '#222222' ),
					'show_in_shortcode' => false,
				],

				'gpdf_header'           => [
					'label'             => __( 'Header', 'gk-pdf-for-gravityview' ),
					'type'              => 'rich_textarea',
					'group'             => 'pdf',
					'desc'              => sprintf(
						__(
							'The header is included at the top of each page. For simple columns %1$stry this HTML table snippet%2$s.',
							'gk-pdf-for-gravityview'
						),
						'<a href="https://gist.github.com/jakejackson1/997b5dedf0a5e665e8ef">',
						'</a>'
					),
					'size'              => 125,
					'show_in_shortcode' => false,
					'value'             => '',
					'merge_tags'        => 'force',
					'full_width'        => true,
				],

				'gpdf_footer'           => [
					'label'             => __( 'Footer', 'gk-pdf-for-gravityview' ),
					'type'              => 'rich_textarea',
					'group'             => 'pdf',
					'desc'              => sprintf(
						__(
							'The footer is included at the bottom of every page. For simple text footers use the left, center and right alignment buttons in the editor. The special %1$s{PAGENO}%2$s and %1$s{nbpg}%2$s tags can also be used to display page numbering.',
							'gk-pdf-for-gravityview'
						),
						'<em>',
						'</em>'
					),
					'size'              => 125,
					'show_in_shortcode' => false,
					'value'             => '',
					'merge_tags'        => 'force',
					'full_width'        => true,
				],

				'gpdf_background_color' => [
					'label'             => __( 'Background Color', 'gk-pdf-for-gravityview' ),
					'type'              => 'color_picker',
					'group'             => 'pdf',
					'value'             => '#FFFFFF',
					'show_in_shortcode' => false,
				],

				'gpdf_background_image' => [
					'label'             => __( 'Background Image', 'gk-pdf-for-gravityview' ),
					'type'              => 'upload',
					'group'             => 'pdf',
					'value'             => '',
					'desc'              => __( 'For optimal results, use an image the same dimensions as the paper size, and put it through an image optimization tool.', 'gk-pdf-for-gravityview' ),
					'show_in_shortcode' => false,
				],

				'gpdf_show_title'       => [
					'label'             => __( 'Show Title?', 'gk-pdf-for-gravityview' ),
					'type'              => 'checkbox',
					'group'             => 'pdf',
					'value'             => '1',
					'desc'              => __( 'Display the Single Entry Title (or the View Title, if empty) in the PDF', 'gk-pdf-for-gravityview' ),
					'show_in_shortcode' => false,
				],

				'gpdf_notifications'    => [
					'label'   => __( 'Form Notifications', 'gk-pdf-for-gravityview' ),
					'type'    => 'checkboxes',
					'tooltip' => __( 'When a selected Gravity Forms notification email is sent, the PDF will be included as an email attachment.', 'gk-pdf-for-gravityview' ),
					'group'   => 'pdf',
					'value'   => 0,
					'options' => $notifications,
				],

				'gpdf_css_editor'       => [
					'label'      => __( 'Additional CSS', 'gk-pdf-for-gravityview' ),
					'desc'       => sprintf( __( '%1$sInspect a PDF\'s HTML%3$s to help create your selectors. Note: %2$sonly a subset of CSS is supported%3$s.', 'gk-pdf-for-gravityview' ), '<a href="https://docs.gravitypdf.com/v6/developers/helper-parameters#html1">', '<a href="https://docs.gravitypdf.com/v6/developers/pdf-features/supported-html-and-css#css-support">', '</a>' ),
					'type'       => 'code_editor',
					'syntax'     => 'css',
					'group'      => 'pdf',
					'full_width' => true,
					'value'      => "/* \n * Useful CSS snippets can be found in the documentation: \n * https://docs.gravitypdf.com/v6/extensions/pdf-for-gravityview-add-on#css \n */ \n\n\n\n",
				],
			],
			$settings
		);

		/* Remove form notifications setting if no options exist, or this is a joined form */
		if ( empty( $pdf_settings['gpdf_notifications']['options'] ) || ! empty( get_post_meta( $view_id, '_gravityview_form_joins', true ) ) ) {
			unset( $pdf_settings['gpdf_notifications'] );
		}

		$this->default_settings[ $view_id ] = $pdf_settings;

		return $settings + $this->default_settings[ $view_id ];
	}

	/**
	 * Convert a multi-dimensional array to a single-dimension array
	 *
	 * @param array $array_to_flatten A multi-dimensional array that goes two levels deep
	 *
	 * @return array A single-dimensional array
	 *
	 * @since 0.1
	 */
	protected function flatten_options_groups( $array_to_flatten ) {
		$new_array = [];
		foreach ( $array_to_flatten as $group ) {
			$new_array += $group;
		}

		return $new_array;
	}
}
