<?php

namespace GFPDF\Plugins\PdfForGravityView\Pdf;

use GFPDF\Helper\Helper_Trait_Logger;
use GFPDF\Helper\Helper_Url_Signer;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since 0.1
 */
class Authentication {
	use Helper_Trait_Logger;

	/**
	 * @since 0.1
	 */
	public function init() {
		add_filter( 'gravityview/view/can_render', [ $this, 'authenticate_request' ], 10, 4 );
	}

	/**
	 * Check if a valid PDF request
	 *
	 * @param bool|\WP_Error $result  The result. Default: null.
	 * @param \GV\View       $view    The view.
	 * @param string[]       $context See \GV\View::can_render
	 * @param \GV\Request    $request The request.
	 *
	 * @since 0.1
	 */
	public function authenticate_request( $result, $view, $context, $request ) {
		/* Do nothing if not a PDF request */
		if ( ! in_array( 'pdf', $context, true ) ) {
			return $result;
		}

		 /* Missing key information to validate the request */
		 /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
		if ( ! isset( $_GET['expires'], $_GET['signature'], $_SERVER['HTTP_HOST'], $_SERVER['REQUEST_URI'] ) ) {
			add_filter( 'gravityview/entry_link/add_query_args', '__return_false' );
			$message = sprintf(
				__( 'There was a problem processing the request. %1$sGo back%2$s.', 'gk-pdf-for-gravityview' ),
				'<a href="' . esc_url( gv_entry_link( $request->is_entry()->as_entry(), $view->ID ) ) . '">',
				'</a>'
			);
			remove_filter( 'gravityview/entry_link/add_query_args', '__return_false' );

			$this->get_logger()->error(
				'The PDF request did not contain all the required superglobals',
				[
					'GET'    => $_GET, /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
					'SERVER' => $_SERVER, /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
				]
			);

			return new \WP_Error( 'invalid_pdf_url', $message );
		}

		$protocol = isset( $_SERVER['HTTPS'] ) && $_SERVER['HTTPS'] === 'on' ? 'https://' : 'http://';
		$domain   = $_SERVER['HTTP_HOST'];
		$uri      = $_SERVER['REQUEST_URI'];

		$url = esc_url_raw( $protocol . $domain . $uri );

		/* Remove PDF Helper arguments without breaking signing auth */
		$url = remove_query_arg( [ 'data', 'html' ], $url );

		$signer = new Helper_Url_Signer();
		if ( $signer->verify( $url ) ) {
			return $result;
		}

		$this->get_logger()->warning(
			'The PDF request has an expired (or invalid) signature',
			[
				'url'    => $url,
				'GET'    => $_GET, /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
				'SERVER' => $_SERVER, /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
			]
		);

		return new \WP_Error( 'expired_pdf_url', __( 'The PDF is no longer accessible.', 'gk-pdf-for-gravityview' ) );
	}
}
