<?php

use GFPDF\Statics\Kses;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! isset( $gravityview ) || empty( $gravityview->template ) ) {
	gravityview()->log->error( '{file} template loaded without context', [ 'file' => __FILE__ ] );

	return;
}

/** @var \GV\Field $field */
$field         = $gravityview->field;
$display_value = $gravityview->display_value;
$input_id      = gravityview_get_input_id_from_id( $field->ID );

// Backward compatibility for the `score` field setting checkbox before migrating to `choice_display` radio
$default_display = $field->score ? 'score' : 'default';

$choice_display = \GV\Utils::get( $field, 'choice_display', $default_display );

switch ( $gravityview->field->field->inputType ) {
	case 'text':
	case 'textarea':
	case 'radio':
	case 'rank':
	case 'select':
	default:
		Kses::output( $display_value );

		return;  // Return early
	case 'checkbox':
		// Display the <ul>
		if ( ! $input_id ) {
			Kses::output( $display_value );

			return;
		}

		$display_value = GFFormsModel::get_choice_text( $field->field, $gravityview->value, $field->ID );

		if ( 'tick' === $choice_display || 'default' === $choice_display ) {
			/**
			 * Filter is defined in /templates/fields/field-checkbox-html.php
			 */
			Kses::output( $display_value !== '' ? apply_filters( 'gravityview_field_tick', '<span class="dashicons dashicons-yes"></span>', $gravityview->entry, $field->as_configuration(), $gravityview ) : '' );

			return; // Return early
		}

		Kses::output( $display_value );

		return; // Return early
	case 'likert':
		// Gravity Forms-generated Likert table output
		if ( 'default' === $choice_display || empty( $choice_display ) ) {

			// Default is the likert table; show it and return early.
			if ( $field->field->gsurveyLikertEnableMultipleRows && ! $input_id ) {
				// If Survey plugin enabled this will be a string and we should output
				Kses::output( ! is_array( $display_value ) ? $display_value : '' );

				return;  // Return early
			}
		}

		// Force the non-multirow fields into the same formatting (row:column)
		$raw_value = is_array( $gravityview->value ) ? $gravityview->value : [ $field->ID => ':' . $gravityview->value ];

		$output_values = [];
		foreach ( $raw_value as $row => $row_values ) {
			[ $_likert_row, $row_value ] = array_pad( explode( ':', $row_values ), 2, '' );

			// If we're displaying a single row, don't include other row values
			if ( $input_id && $row !== $field->ID ) {
				continue;
			}

			switch ( $choice_display ) {
				case 'score':
					$output_values[] = GravityView_Field_Survey::get_choice_score( $field->field, $row_value, $row );
					break;
				case 'text':
					$output_values[] = GFFormsModel::get_choice_text( $field->field, $row_value, $row );
					break;
				case 'default':
				default:
					// When displaying a single input, render as if multiple rows were disabled
					/** @var GF_Field_Likert $single_input_field */
					$single_input_field                                  = clone $field->field;
					$single_input_field->id                              = $field->ID;
					$single_input_field->gsurveyLikertEnableMultipleRows = false;
					$output_values[]                                     = $single_input_field->get_field_input( [ 'id' => $field->form_id ], $row_value );
					break;
			}
		}

		/**
		 * @filter `gravityview/template/field/survey/glue` The value used to separate multiple values in the Survey field output
		 *
		 * @param string The glue. Default: "; " (semicolon with a trailing space)
		 * @param \GV\Template_Context The context.
		 */
		$glue = apply_filters( 'gravityview/template/field/survey/glue', '; ', $gravityview );

		Kses::output( implode( $glue, $output_values ) );

		return; // Return early

	case 'rating':
		$choice_text = GFFormsModel::get_choice_text( $field->field, $gravityview->value, $input_id );

		if ( ! in_array( $choice_display, [ 'stars', 'dashicons', 'emoji' ], true ) ) {
			Kses::output( $choice_text );

			return;
		}

		$choices       = $field->field->choices;
		$choice_values = wp_list_pluck( $choices, 'value', $gravityview->value );
		$starred_index = array_search( $gravityview->value, $choice_values, true );

		$star = apply_filters(
			'gfpdf_gv_field_star_empty',
			'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCA1MiA1MiI+PGNpcmNsZSBjeD0iMjUuOCIgY3k9IjI1LjgiIHI9IjE3LjkiIGZpbGw9IiNlZWUiLz48cGF0aCBmaWxsPSIjY2NjIiBmaWxsLXJ1bGU9Im5vbnplcm8iIGQ9Ik0yNS44IDUxLjZDMTEuNiA1MS42IDAgNDAuMSAwIDI1LjhTMTEuNiAwIDI1LjggMEM0MCAwIDUxLjYgMTEuNiA1MS42IDI1LjhjMCAxNC4yLTExLjUgMjUuOC0yNS44IDI1LjhabTAtNDYuNkMxNC4zIDUgNSAxNC4zIDUgMjUuOHM5LjMgMjAuOCAyMC44IDIwLjggMjAuOC05LjMgMjAuOC0yMC44UzM3LjMgNSAyNS44IDVaIi8+PHBhdGggZmlsbD0iI2NjYyIgZmlsbC1ydWxlPSJub256ZXJvIiBkPSJtMzAuNiAyNy42IDQuOC00LjYtNi42LTEtMy02LTMgNi02LjYgMSA0LjggNC42LTEuMSA2LjYgNS45LTMuMSA1LjkgMy4xLTEuMS02LjZabTguMi01LjZjMCAuMi0uMS41LS40LjhsLTUuNyA1LjUgMS4zIDcuOHYuM2MwIC41LS4yLjgtLjYuOC0uMiAwLS40LS4xLS42LS4ybC03LTMuNy03IDMuN2MtLjIuMS0uNC4yLS42LjItLjIgMC0uNC0uMS0uNS0uMi0uMS0uMS0uMi0uMy0uMi0uNnYtLjNsMS4zLTcuOC01LjctNS41Yy0uMy0uMy0uNC0uNS0uNC0uOCAwLS40LjMtLjYuOS0uN2w3LjgtMS4xIDMuNS03LjFjLjItLjQuNS0uNi44LS42LjMgMCAuNi4yLjguNmwzLjUgNy4xIDcuOCAxLjFjLjcuMSAxIC4zIDEgLjdaIi8+PC9zdmc+',
			$gravityview->entry,
			$field->as_configuration(),
			$gravityview
		);

		$star_filled = apply_filters(
			'gfpdf_gv_field_star_filled',
			'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCA1MiA1MiI+PGNpcmNsZSBjeD0iMjUuOCIgY3k9IjI1LjgiIHI9IjE3LjkiIGZpbGw9IiNlZWUiLz48cGF0aCBmaWxsPSIjY2NjIiBmaWxsLXJ1bGU9Im5vbnplcm8iIGQ9Ik0yNS44IDUxLjZDMTEuNiA1MS42IDAgNDAuMSAwIDI1LjhTMTEuNiAwIDI1LjggMEM0MCAwIDUxLjYgMTEuNiA1MS42IDI1LjhjMCAxNC4yLTExLjUgMjUuOC0yNS44IDI1LjhabTAtNDYuNkMxNC4zIDUgNSAxNC4zIDUgMjUuOHM5LjMgMjAuOCAyMC44IDIwLjggMjAuOC05LjMgMjAuOC0yMC44UzM3LjMgNSAyNS44IDVaIi8+PHBhdGggZmlsbD0iI2ZmOTgwMCIgZmlsbC1ydWxlPSJub256ZXJvIiBkPSJNMzguOCAyMi4xYzAgLjItLjEuNS0uNC44bC01LjcgNS41IDEuMyA3Ljh2LjNjMCAuMi0uMS40LS4yLjYtLjEuMi0uMy4yLS41LjJzLS40LS4xLS42LS4ybC03LTMuNy03IDMuN2MtLjIuMS0uNC4yLS42LjItLjIgMC0uNC0uMS0uNS0uMi0uMS0uMi0uMi0uMy0uMi0uNnYtLjNsMS4zLTcuOC01LjctNS41Yy0uMy0uMy0uNC0uNS0uNC0uOCAwLS40LjMtLjYuOS0uN2w3LjgtMS4xIDMuNS03LjFjLjItLjQuNS0uNi44LS42LjMgMCAuNi4yLjguNmwzLjUgNy4xIDcuOCAxLjFjLjguMSAxLjEuMyAxLjEuN1oiLz48L3N2Zz4=',
			$gravityview->entry,
			$field->as_configuration(),
			$gravityview
		);


		foreach ( $choices as $current_index => $choice_value ) {
			// Have we already shown the last filled-in star?
			$empty     = ( $current_index > $starred_index );
			$css_class = 'gv-field-survey-star-' . ( $empty ? 'empty' : 'filled' );

			printf(
				'<img class="gv-field-survey-star %1$s" src="%2$s" />',
				esc_attr( $css_class ),
				esc_attr( $empty ? $star : $star_filled )
			);
		}

		return;
}
