<?php

namespace GFPDF\Plugins\PdfForGravityView\Pdf;

use GFPDF\Helper\Helper_Trait_Logger;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since 1.1
 */
class Notifications {
	use Helper_Trait_Logger;

	/**
	 * @since 1.1
	 */
	public function init() {
		add_filter( 'gform_notification', [ $this, 'attachments' ], 9999, 3 );
	}

	/**
	 * Get associated views and attach configured PDFs to the notification
	 *
	 * @param array $notification Gravity Forms Notification Array
	 * @param array $form         Form object
	 * @param array $entry        Entry object
	 *
	 * @return array
	 *
	 * @internal let the GPDF core scheduled cleanup delete the document from disk after 12hrs
	 *
	 * @since 1.1
	 */
	public function attachments( $notification, $form, $entry ) {
		/* Exit early if unexpected input provided */
		if ( empty( $entry['id'] ) || empty( $form['id'] ) || ! is_array( $notification ) ) {
			return $notification;
		}

		/* Ensure attachments key exists */
		$notification['attachments'] = $notification['attachments'] ?? [];

		/* Get all views assigned to this form (do not include views with joined forms) */
		if ( version_compare( GV_PLUGIN_VERSION, '2.19', '>=' ) ) {
			$views = gravityview_get_connected_views( $form['id'], [], false );
		} else {
			$views = gravityview_get_connected_views( $form['id'] );
		}

		foreach ( $views as $view ) {
			/** @var \WP_Post $view */

			/* Get the PDF Notification setting for the view (if any) and filter the selected options only */
			$single_entry_pdf_notifications = gravityview_get_template_setting( $view->ID, 'gpdf_notifications' );
			$single_entry_pdf_notifications = is_array( $single_entry_pdf_notifications ) ? array_keys( array_filter( $single_entry_pdf_notifications ) ) : [];

			if ( ! in_array( $notification['id'], $single_entry_pdf_notifications, true ) ) {
				continue;
			}

			/* Skip joined forms */
			if ( ! empty( get_post_meta( $view->ID, '_gravityview_form_joins', true ) ) ) {
				continue;
			}

			/* Generate and attach the PDF to the Notification */
			$pdf_path = \PDFGV_API::create_single_entry_pdf( $view->ID, $entry['id'] );

			/* Log error and continue */
			if ( is_wp_error( $pdf_path ) ) {
				$this->logger->error(
					'PDF Generation Error',
					[
						'code'         => $pdf_path->get_error_code(),
						'message'      => $pdf_path->get_error_message(),
						'form'         => $form['id'],
						'entry'        => $entry['id'],
						'view'         => $view->ID,
						'notification' => $notification,
					]
				);

				continue;
			}

			/* Success! Attach to notification */
			$notification['attachments'][] = $pdf_path;
		}

		$this->logger->notice(
			'Gravity Forms Attachments',
			[
				'form'         => $form['id'],
				'entry'        => $entry['id'],
				'attachments'  => $notification['attachments'],
				'notification' => $notification,
			]
		);

		return $notification;
	}
}
