<?php

namespace GFPDF\Plugins\PdfForGravityView\Pdf;

use GFPDF\Exceptions\GravityPdfException;
use GFPDF\Helper\Helper_Abstract_Form;
use GFPDF\Helper\Helper_Data;
use GFPDF\Helper\Helper_Form;
use GFPDF\Helper\Helper_Misc;
use GFPDF\Helper\Helper_PDF;
use GFPDF\Helper\Helper_QueryPath;
use GFPDF\Helper\Helper_Templates;
use GFPDF\Helper\Helper_Trait_Logger;
use GFPDF\Model\Model_PDF;
use GFPDF\Statics\Kses;
use GPDFAPI;
use GV\Entry;
use GV\Entry_Collection;
use GV\Entry_Template;
use GV\GF_Entry;
use GV\GF_Form;
use GV\Mocks\Legacy_Context;
use GV\Template_Context;
use GV\View;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * @since 0.1
 */
class Renderer {
	use Helper_Trait_Logger;

	/**
	 * @var Settings
	 * @since 0.1
	 */
	protected $settings;

	/**
	 * @var Helper_Form
	 * @since 0.1
	 */
	protected $gform;

	/**
	 * @var Helper_Data
	 * @since 0.1
	 */
	protected $data;

	/**
	 * @var Helper_Misc
	 * @since 0.1
	 */
	protected $misc;

	/**
	 * @var Helper_Templates
	 * @since 0.1
	 */
	protected $templates;

	/**
	 * @var bool Whether the PDF is currently being rendered
	 * @since 0.2
	 */
	protected static $rendering_pdf = false;

	/**
	 * @since 0.1
	 */
	public function __construct( Settings $settings, Helper_Abstract_Form $gform, Helper_Data $data, Helper_Misc $misc, Helper_Templates $templates ) {
		$this->settings  = $settings;
		$this->gform     = $gform;
		$this->data      = $data;
		$this->misc      = $misc;
		$this->templates = $templates;
	}

	/**
	 * Whether the PDF is currently being rendered, or not
	 *
	 * @return bool
	 * @since 0.2
	 */
	public static function is_pdf_endpoint(): bool {
		return self::$rendering_pdf;
	}

	/**
	 * @since 0.1
	 */
	public function init() {
		add_action( 'template_redirect', [ $this, 'maybe_render_pdf' ] );
	}

	/**
	 * Check if a valid PDF request and then generate the document
	 *
	 * @return void
	 *
	 * @since 0.1
	 */
	public function maybe_render_pdf(): void {
		/* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
		$is_pdf = $_GET['gv-pdf'] ?? '';

		/*
		 * Check if a valid request
		 */
		if ( ! $is_pdf ) {
			return;
		}

		/*
		 * Pull the View ID from the URL parameter
		 * The PDF URL is signed (tamper resistant), and security checks later in the process will
		 * stop rendering if the ID is changed by the user.
		 */
		$view_id = (int) ( $_GET['view'] ?? 0 ); /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
		$view    = \GV\View::by_id( $view_id );

		gravityview()->views->set( $view );
		/** @var Entry $entry */

		$entry = gravityview()->request->is_entry();

		if ( ! $view || ! $entry ) {
			$this->get_logger()->error(
				'Invalid View or Entry',
				[
					'view_id' => $view_id,
					'SERVER'  => $_SERVER, /* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
				]
			);

			$this->die( __( 'There was a problem with the request.', 'gk-pdf-for-gravityview' ), __( 'PDF Not Found', 'gk-pdf-for-gravityview' ), 404 );

			return;
		}

		/*
		 * Do security check
		 */
		$can_render = $view->can_render( [ 'pdf' ] );

		if ( is_wp_error( $can_render ) ) {
			$this->get_logger()->error(
				'Not rendering PDF: ' . wp_strip_all_tags( $can_render->get_error_message() ),
				[
					'view'  => $view->ID,
					'entry' => $entry->ID,
				]
			);

			$this->die( $can_render->get_error_message(), __( 'Invalid Request', 'gk-pdf-for-gravityview' ), 403 );
		}

		/* Verify entry or entries belong to this view. */
		if ( ! $this->is_view_entries_valid( $view, $entry ) ) {
			$this->die( __( 'There was a problem with the request.', 'gk-pdf-for-gravityview' ), __( 'Invalid Request', 'gk-pdf-for-gravityview' ), 400 );
			return;
		}

		/*
		 * A valid PDF request
		 */
		$entries = new Entry_Collection();
		$entries->add( $entry );
		Legacy_Context::push(
			[
				'view'    => $view,
				'entries' => $entries,
				'entry'   => $entry,
				'request' => gravityview()->request,
			]
		);

		$this->render_pdf( $view, $entry );
	}

	/**
	 * Check if the entries for the current view are all considered valid
	 *
	 * @param View  $view
	 * @param Entry $entry
	 *
	 * @return bool
	 *
	 * @since 0.4
	 */
	public function is_view_entries_valid( View $view, Entry $entry ): bool {
		if ( $view->joins ) {
			$form_ids = [];
			foreach ( $view->joins as $join ) {
				$form_ids[] = (int) $join->join->ID;
				$form_ids[] = (int) $join->join_on->ID;
			}
			foreach ( $entry->entries as $e ) {
				if ( ! in_array( (int) $e['form_id'], $form_ids, true ) ) {
					$this->get_logger()->error(
						'The requested entry does not belong to this View.',
						[
							'view'  => $view->ID,
							'entry' => $e->ID,
						]
					);

					return false;
				}
			}
		} elseif ( $view->form && (int) $view->form->ID !== (int) $entry['form_id'] ) {
			$this->get_logger()->error(
				'The requested entry does not belong to this View.',
				[
					'view'  => $view->ID,
					'entry' => $entry->ID,
				]
			);

			return false;
		}

		return true;
	}

	/**
	 * Render the PDF based on the current View/Entry
	 *
	 * @param View   $view
	 * @param Entry  $entry
	 * @param string $output_type valid values: download|display|save
	 *
	 * @return void|string
	 *
	 * @throws GravityPdfException
	 * @since 0.1
	 */
	public function render_pdf( View $view, Entry $entry, $output_type = null ) {
		self::$rendering_pdf = true;

		$this->modify_settings_for_pdf_generation();

		$pdf_settings = $this->get_single_entry_pdf_settings( $view );
		$pdf_settings = apply_filters( 'gfpdf_gv_pdf_settings', $pdf_settings, $view, $entry );

		try {
			$pdf_generator = new Helper_PDF(
				$entry->as_entry(),
				$pdf_settings,
				$this->gform,
				$this->data,
				$this->misc,
				$this->templates,
				$this->get_logger()
			);

			/** @var Model_PDF $model */
			$model    = \GPDFAPI::get_pdf_class( 'model' );
			$filename = $model->get_pdf_name( $pdf_settings, $entry->as_entry() );
			$pdf_generator->set_filename( apply_filters( 'gfpdf_gv_pdf_filename', $filename, $view, $entry ) );

			$pdf_generator->init();

			/* Determine how the PDF should be output */
			if ( ! $output_type ) {
				/* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
				$download_pdf = (int) ( $_GET['download'] ?? 0 );

				$output_type = $download_pdf ? 'download' : 'display';
			}

			$output_type = strtolower( $output_type );
			$pdf_generator->set_output_type( $output_type );

			/* Generate the HTML for the PDF */
			$args = $this->get_template_arguments( $pdf_settings, $view, $entry );
			$pdf_generator->render_html( $args );

			/* Clean up / remove the _pdf suffix on the template slug */
			$view_template        = $args['view_template'];
			$view_template::$slug = substr( $view_template::$slug, 0, -4 );
			unset( $args );

			if ( $output_type !== 'save' ) {
				/* Display or download */
				$pdf_generator->generate();
			} else {
				/* Save PDF to file and return path */
				$pdf = $pdf_generator->generate();

				self::$rendering_pdf = false;

				$this->restore_settings_after_pdf_generation();

				return $pdf_generator->save_pdf( $pdf );
			}
		} catch ( \Exception $e ) {
			$this->get_logger()->error(
				'PDF Generation Error',
				[
					'entry'     => $entry,
					'settings'  => $pdf_settings,
					'exception' => $e->getMessage(),
				]
			);

			$message = sprintf(
				'%s in %s on line %s',
				$e->getMessage(),
				$e->getFile(),
				$e->getLine()
			);

			/* If saving the PDF, rethrow the exception so it can be handled later */
			if ( $output_type === 'save' ) {
				$this->restore_settings_after_pdf_generation();

				throw new GravityPdfException( esc_html( $message ) );
			}

			if ( $this->gform->has_capability( 'gravityforms_view_entries' ) ) {
				$message = sprintf(
					'%s in %s on line %s',
					$e->getMessage(),
					$e->getFile(),
					$e->getLine()
				);

				$this->die( $message, __( 'PDF Generation Error', 'gk-pdf-for-gravityview' ), 500 );
			}

			$this->die( __( 'There was a problem generating the PDF', 'gk-pdf-for-gravityview' ), __( 'PDF Generation Error', 'gk-pdf-for-gravityview' ), 500 );
		}
	}

	/**
	 * Register the directory that contain the GravityView PDF templates
	 *
	 * @param string $path
	 * @param string $template_id
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	public function register_template_directory( $path, $template_id ) {
		$base_path    = __DIR__ . '/Templates/';
		$path_to_test = realpath( $base_path . $template_id . '.php' );
		if ( $path_to_test !== false && strpos( $path_to_test, realpath( $base_path ) ) === 0 ) {
			return $path_to_test;
		}

		return $path;
	}

	/**
	 * Register the GV/Gravity PDF hooks needed to successfully generate the PDFs
	 *
	 * @return void
	 *
	 * @since 0.1
	 */
	protected function modify_settings_for_pdf_generation(): void {
		/* Register the GV template directory */
		add_filter( 'gfpdf_fallback_template_path_by_id', [ $this, 'register_template_directory' ], 10, 2 );

		/* Prevent PDF arguments polluting view entry links */
		add_filter( 'gravityview/entry_link/add_query_args', '__return_false' );

		/* Ensure email is never encoded */
		add_filter( 'gravityview_email_prevent_encrypt', '__return_true' );

		/* Override the pre-processing of Header/Footer PDF settings */
		$model = \GPDFAPI::get_pdf_class( 'model' );
		remove_filter( 'gfpdf_template_args', [ $model, 'preprocess_template_arguments' ] );
		add_filter( 'gfpdf_template_args', [ $this, 'preprocess_template_arguments' ] );

		/* Remove default list styles */
		add_filter( 'gfpdf_include_list_styles', '__return_false' );

		/* Register a new GravityView template override folder */
		add_filter( 'gravityview/template/fields_template_paths', [ $this, 'register_pdf_field_template_path' ] );

		/* Allow inline <style> tags */
		add_filter( 'gfpdf_wp_kses_allowed_html', [ $this, 'allow_inline_styles' ] );

		/* Change font icon checkmark for a tick html entity */
		add_filter( 'gravityview_field_tick', [ $this, 'change_checkmark_mark_up' ] );
	}

	/**
	 * Restore the modified hooks back to the original state
	 *
	 * @return void
	 *
	 * @since 1.1.0
	 */
	protected function restore_settings_after_pdf_generation(): void {
		/* Register the GV template directory */
		remove_filter( 'gfpdf_fallback_template_path_by_id', [ $this, 'register_template_directory' ] );

		/* Prevent PDF arguments polluting view entry links */
		remove_filter( 'gravityview/entry_link/add_query_args', '__return_false' );

		/* Ensure email is never encoded */
		remove_filter( 'gravityview_email_prevent_encrypt', '__return_true' );

		/* Override the pre-processing of Header/Footer PDF settings */
		$model = \GPDFAPI::get_pdf_class( 'model' );
		add_filter( 'gfpdf_template_args', [ $model, 'preprocess_template_arguments' ] );
		remove_filter( 'gfpdf_template_args', [ $this, 'preprocess_template_arguments' ] );

		/* Remove default list styles */
		remove_filter( 'gfpdf_include_list_styles', '__return_false' );

		/* Register a new GravityView template override folder */
		remove_filter( 'gravityview/template/fields_template_paths', [ $this, 'register_pdf_field_template_path' ] );

		/* Allow inline <style> tags */
		remove_filter( 'gfpdf_wp_kses_allowed_html', [ $this, 'allow_inline_styles' ] );

		/* Change font icon checkmark for a tick html entity */
		remove_filter( 'gravityview_field_tick', [ $this, 'change_checkmark_mark_up' ] );
	}

	/**
	 * Register a new GravityView fields path, so we can easily change the mark-up for specific
	 * fields when rendering in the PDF>
	 *
	 * @param array $paths
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	public function register_pdf_field_template_path( $paths ) {
		/* Ensure the priority key is unique and early */
		$key = 1;
		while ( isset( $paths[ $key ] ) ) {
			++$key;
		}

		$paths[ $key ] = __DIR__ . '/Fields';

		return $paths;
	}

	/**
	 * Allow inline <style> tags in PDFs
	 *
	 * @param array $args
	 *
	 * @return array
	 */
	public function allow_inline_styles( $args ) {
		$args['style'] = [];

		return $args;
	}

	/**
	 * Add support for individual Choices in Views
	 *
	 * @return string
	 *
	 * @since 0.2
	 */
	public function change_checkmark_mark_up() {
		return '<span class="dashicons dashicons-yes">&#10004;</span>';
	}

	/**
	 * Ensure the Header/Footer mark-up is sanitized appropriately
	 *
	 * @internal these methods are redefined so the image width/height isn't auto-removed, which give users better control over image sizes in the PDF.
	 *
	 * @param array $args
	 *
	 * @return array
	 *
	 * @since    0.1
	 */
	public function preprocess_template_arguments( $args ) {

		if ( isset( $args['settings']['header'] ) ) {
			$args['settings']['header'] = $this->gform->process_tags( $args['settings']['header'], $args['form'], $args['entry'] );
			$args['settings']['header'] = $this->fix_header_footer( $args['settings']['header'] );
		}

		if ( isset( $args['settings']['footer'] ) ) {
			$args['settings']['footer'] = $this->gform->process_tags( $args['settings']['footer'], $args['form'], $args['entry'] );
			$args['settings']['footer'] = $this->fix_header_footer( $args['settings']['footer'] );
		}

		return $args;
	}

	/**
	 * Sanitize the mark-up for Header/Footers
	 *
	 * @param $html
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	protected function fix_header_footer( $html ) {
		$html = Kses::parse( $html );
		$html = trim( wpautop( $html ) );
		$html = $this->fix_header_footer_images( $html );

		/* Strip page breaks */
		$html = preg_replace( '/<pagebreak(.+?)?\/?>/', '', $html );
		$html = preg_replace( '/page-break-(before|after):( +)?(always|left|right|auto|avoid)/', '', $html );

		return $html;
	}

	/**
	 * Load local images over the network, and include support for center aligned images in the editor
	 *
	 * @param string $html
	 *
	 * @return string
	 *
	 * @since 0.3
	 */
	protected function fix_header_footer_images( string $html ): string {
		try {
			/* Get the <img> from the DOM and extract required details */
			$qp      = new Helper_QueryPath();
			$wrapper = $qp->html5( $html );

			$images = $wrapper->find( 'img' );

			if ( count( $images ) > 0 ) {
				/* Loop through each matching element */
				foreach ( $images as $image ) {

					/* Get current image src */
					$image_src      = trim( $image->attr( 'src' ) );
					$image_src_path = $this->misc->convert_url_to_path( $image_src );

					if ( false !== $image_src_path ) {
						$image->attr( 'src', $image_src_path );
					}

					/* Get the current image classes */
					$image_classes = $image->attr( 'class' );

					/*
					 * If center-aligning, wrap in a new <p> tag to ensure the PDF renderer can do it.
					 * If the direct parent is a link, we'll wrap the link instead.
					 */
					if ( strpos( $image_classes, 'aligncenter' ) !== false ) {
						$parent_node = $image->parent()->get( 0 );
						if ( $parent_node instanceof \DOMElement && $parent_node->nodeName === 'a' ) {
							$image->parent()->wrap( '<p class="' . esc_attr( $image_classes ) . '"></p>' );
						} else {
							$image->wrap( '<p class="' . esc_attr( $image_classes ) . '"></p>' );
						}
					}
				}

				$html = $wrapper->top( 'html' )->innerHTML();
			}

			return $html;

		} catch ( \Exception $e ) {
			/* if there was any issues we'll just return the $html */
			return $html;
		}
	}

	/**
	 * Get the appropriate PDF template file based on the View type
	 *
	 * @param View $view
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	protected function get_pdf_template( View $view ) {
		$template_slug = $this->get_view_template_slug( $view );
		$template      = 'gravityview-single-entry-' . $template_slug . '-template';

		/* Try get a template for this specific View */
		try {
			$this->templates->get_template_path_by_id( $template . '-' . $view->ID );
			$template .= '-' . $view->ID;
		} catch ( \Exception $e ) {
			// Do nothing
		}

		/* Fallback to the layout-specific templates */
		try {
			$this->templates->get_template_path_by_id( $template );
		} catch ( \Exception $e ) {
			$this->get_logger()->warning(
				'The current View uses a layout that is not yet supported. Using table fallback',
				[
					'view'   => $view->ID,
					'layout' => $view->settings->get( 'template' ),
				]
			);

			/* No layout-specific template found. Use the Table layout */
			$template = 'gravityview-single-entry-table-template';
		}

		return $template;
	}

	/**
	 * Get all the arguments to pass to the PDF template
	 *
	 * @param array $pdf_settings
	 * @param View  $view
	 * @param Entry $entry
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	protected function get_template_arguments( array $pdf_settings, View $view, Entry $entry ): array {
		$form = $this->gform->get_form( $view->form->ID );

		$args = $this->templates->get_template_arguments(
			$form,
			$this->misc->get_fields_sorted_by_id( $form['id'] ),
			$entry->as_entry(),
			GPDFAPI::get_form_data( $entry->ID ),
			$pdf_settings,
			$this->templates->get_config_class( $pdf_settings['template'] ),
			$this->misc->get_legacy_ids( $entry->ID, $pdf_settings )
		);

		unset( $args['lead_ids'], $args['lead'] );

		$args['view']          = $view;
		$args['view_entry']    = $entry;
		$args['view_template'] = $this->get_view_template( $view, $entry );
		$args['view_fields']   = $this->get_view_fields( $view, $entry, $args['view_template'] );

		$pdf_view = GPDFAPI::get_pdf_class();
		$pdf_view->maybe_view_form_data( $args['form_data'] ?? [] );

		return apply_filters( 'gfpdf_gv_template_arguments', $args, $view, $entry, $pdf_settings );
	}

	/**
	 * Pre-process all the Single Layout fields into groups, filtering out any that should not be displayed in the PDF
	 *
	 * @param View           $view
	 * @param Entry          $entry
	 * @param Entry_Template $template
	 *
	 * @return array
	 *
	 * @since 0.1
	 */
	protected function get_view_fields( View $view, Entry $entry, Entry_Template $template ): array {
		$view_fields = [];
		$fields      = $view->fields->by_position( 'single_*' )->by_visible( $view );
		foreach ( $fields->all() as $field ) {
			$field_form  = GF_Form::by_id( $field->form_id ) ?: $view->form;
			$field_entry = $entry->from_field( $field );

			if ( ! $field_entry ) {
				continue;
			}

			$context = Template_Context::from_template( $template, [ 'field' => $field ] );

			/* Allow fields to be programmatically skipped */
			$middleware = apply_filters( 'gfpdf_gv_field_middleware', false, $field, $field_entry, $field_form, $view, $context );

			if ( $middleware ) {
				continue;
			}

			/* Allow field to be programmatically manipulated */
			$field = apply_filters( 'gfpdf_gv_view_field', $field, $field_entry, $field_form, $view, $context );
			$field = apply_filters( 'gfpdf_gv_view_field_' . $field->type, $field, $field_entry, $field_form, $view, $context );

			/* Group the field by the position */
			if ( ! isset( $view_fields[ $field->position ] ) ) {
				$view_fields[ $field->position ] = [];
			}

			$view_fields[ $field->position ][] = $field;
		}

		return $view_fields;
	}

	/**
	 * Get the GravityView Entry Template for the current View layout
	 *
	 * @param View  $view
	 * @param Entry $entry
	 *
	 * @return Entry_Template
	 *
	 * @since 0.1
	 */
	protected function get_view_template( View $view, Entry $entry ): Entry_Template {
		$request       = gravityview()->request;
		$template_slug = $this->get_view_template_slug( $view );

		$class = apply_filters( 'gravityview/template/entry/class', sprintf( '\GV\Entry_%s_Template', ucwords( $template_slug, '_' ) ), $entry, $view, $request );
		if ( ! $class || ! class_exists( $class ) ) {
			$this->get_logger()->warning(
				'The current View does not have a dedicated entry template. Falling back to \GV\Entry_Legacy_Template',
				[
					'view'   => $view->ID,
					'layout' => $view->settings->get( 'template' ),
				]
			);

			$class = '\GV\Entry_Legacy_Template';
		}

		$template         = new $class( $entry, $view, $request );
		$template::$slug .= '_pdf';

		return $template;
	}

	/**
	 * Get the correct template to use for the view
	 *
	 * @param View $view
	 *
	 * @return string
	 *
	 * @since 0.1
	 */
	protected function get_view_template_slug( View $view ) {
		/* Add support for GV 2.24, which allows a different template per layout */
		if ( function_exists( 'gravityview_get_single_entry_template_id' ) ) {
			switch ( gravityview_get_single_entry_template_id( $view->ID ) ) {
				case 'default_list':
					return 'list';
				case 'map':
					return 'map';
				case 'diy':
					return 'diy';
				case 'default_table':
				case 'datatables_table':
					return 'table';
				case 'gravityview-layout-builder':
					return 'layout_builder';
			}
		}

		return apply_filters( 'gravityview_template_slug_' . $view->settings->get( 'template' ), 'table', 'single' );
	}

	/**
	 * End the execution cycle and output the message
	 *
	 * @param string $message
	 * @param string|int $title
	 * @param array|int  $args
	 *
	 * @return void
	 *
	 * @since 0.1
	 */
	protected function die( string $message, $title = '', $args = [] ): void {
		$allowed_tags = [
			'a' => [ 'href' => [] ],
		];

		/* if response code is passed as an int, turn it back into an array */
		if ( is_int( $args ) ) {
			$args = [
				'response' => $args,
			];
		}

		/* Shows a JS-powered back link */
		$args['back_link'] = true;

		$title = is_string( $title ) ? esc_html( $title ) : $title;

		wp_die( wp_kses( $message, $allowed_tags ), $title, $args ); /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */
	}

	/**
	 * Get the PDF settings for the Single Layout View
	 *
	 * @param View $view
	 *
	 * @return array
	 *
	 * @since 0.8.0
	 */
	public function get_single_entry_pdf_settings( View $view ): array {
		$pdf_settings             = $this->settings->get_pdf_settings( $view );
		$pdf_settings['filename'] = $pdf_settings['filename'] ?: $view->get_post()->post_title;
		$pdf_settings['template'] = $this->get_pdf_template( $view );

		return $pdf_settings;
	}
}
