<?php

/*
 * Template Name: GravityView Layout Builder
 * Version: 0.1
 * Description: A PDF template designed specifically for outputting the GravityView Single Entry Layout Builder.
 * Author: Gravity PDF
 * Author URI: https://gravitypdf.com
 * Group: GravityView
 * License: GPLv2
 * Required PDF Version: 6.0
 */

use GFPDF\Statics\Kses;
use GV\Grid;
use GV\Template_Context;

/* Prevent direct access to the template */
if ( ! class_exists( 'GFForms' ) ) {
	return;
}

/**
 * The GravityView PDF Template has access to the following:
 *
 * @var \GV\View           $view          The current GravityView object
 * @var \GV\Entry          $view_entry    The current GravityView entry object
 * @var \GV\Entry_Template $view_template The original template class used to output the Single Entry Layout in GravityView
 * @var array              $view_fields   A curated array of Single Entry Layout fields, grouped by position
 *
 * @var array              $form          The current Gravity Form array
 * @var array              $entry         The raw entry data
 * @var array              $form_data     The processed entry data stored in an array
 * @var array              $settings      The current PDF configuration
 * @var object             $config        The initialized template config class (if any)
 * @var object             $gfpdf         The main Gravity PDF object containing all the helper classes
 * @var array              $args          Contains an array of all variables - the ones being described right now - passed to the template
 */

?>
	<style>
	  @page {
		margin: <?php echo esc_attr( $settings['margin'] ); ?>;
	  }

	  /* Header / Footer */
	  #header p,
	  .footer p {
		margin: 0;
	  }

	  /* Basic Styles */
	  h1,
	  h2,
	  h3,
	  h4,
	  h5,
	  h6 {
		margin: 0.5rem 0;
		padding: 0;
	  }

	  ul,
	  ol {
		margin: 0;
		padding: 0 0 0 5mm;
	  }

	  p {
		margin: 0 0 3.5mm;
		padding: 0
	  }

	  /*
	   * Image alignment
	   */
	  .alignleft {
		float: left;
	  }

	  .alignright {
		float: right;
	  }

	  .aligncenter {
		text-align: center;
	  }

	  p.alignleft {
		text-align: left;
		float: none;
	  }

	  p.alignright {
		text-align: right;
		float: none;
	  }

	  /* Single View Layout Title */
	  #form_title {
		margin: 0 0 3mm;
		padding: 0;
		font-size: 150%;
	  }

	  /* List Field */
	  .gfield_list th {
		text-align: left;
	  }

	  .gfield_list th,
	  .gfield_list td {
		padding: .3em;
	  }

	  /* Notes Field */
	  .entry-detail-note {
		margin-top: -1px; /* Fixes border overlap */
		padding: 1em .5em 0 1em;
		border: 1px solid #eee;
	  }

	  .gv-note-avatar {
		width: 42px;
		height: 42px;
		float: left;
		padding-right: 10px;
	  }

	  .gv-note-author {
		font-weight: bold;
		font-size: 1.1rem;
		line-height: 1.1;
		margin: 0.75mm 0;
		padding: 0;
	  }

	  .gv-note-added-on {
		font-weight: normal;
		font-size: .875rem;
	  }

	  .gv-note-content {
		clear: both;
		margin: 0;
		padding: 1em 0 0 0;
		line-height: 1.8em;
	  }

	  /*
	   * Checkmark
	   */
	  .dashicons-yes {
		font-family: dejavusanscondensed;
		font-size: 120%;
	  }

	  /*
	   * Likert
	   */
	  .gsurvey-likert-choice {
		text-align: center;
		padding: .3em;
	  }

	  /*
	   * Rating
	   */
	  .gv-field-survey-star {
		width: 6mm;
	  }

	  /* Single Entry Layout Builder */
	  body, th, td, li, a {
		line-height: 1.5;
	  }

	  .gv-layout-builder-view--entry {
		border: 1px solid #E3E6EF;
		background: #FFF;
		padding: 16px 0;
	  }

	  .gv-grid-row {
		clear: both;
	  }

	  .gv-grid-col {
		float: left;
		border-right: 1px solid #E3E6EF;
	  }

	  .gv-grid-col.right {
		border-right: none;
	  }

	  .gv-grid-col .inner-container {
		padding: 0 16px;
	  }

	  .gv-grid-col-1-1 {
		float: none;
		border-right: none;
	  }

	  .gv-grid-col-1-2 {
		width: 49.5%;
	  }

	  .gv-grid-col-1-4 {
		width: 24.68%;
	  }

	  .gv-grid-col-1-3 {
		width: 33%;
	  }

	  .gv-grid-col-2-3 {
		width: 66%;
		padding-right: 1px;
	  }

	  .gv-field-wrapper {
		margin-top: 16px;
		border-top: 1px solid #E3E6EF;
		padding-top: 16px;
	  }

	  .gv-field-wrapper.first {
		margin-top: 0;
		padding-top: 0;
		border-top: none;
	  }

	  .gv-field-label {
		color: #565D67;
		font-size: .8em;
	  }

	  <?php
		/* Add Additional CSS, if present */
		echo esc_html( wp_filter_nohtml_kses( $settings['additional_css'] ?? '' ) );

		do_action( 'gfpdf_gv_template_styles', $view, $view_entry );
		do_action( 'gfpdf_gv_template_styles_' . $view->ID, $view, $view_entry );
		do_action( 'gfpdf_gv_template_styles_layout_builder', $view, $view_entry );
		?>
	</style>

<?php

/* Output the page title, if configured */
$single_title  = $view->settings->get( 'single_title', '' );
$single_title  = ! empty( $single_title ) ? $single_title : $view->get_post()->post_title;
$form['title'] = $single_title;

/** @var \GFPDF\View\View_PDF $pdf_view */
$pdf_view = \GPDFAPI::get_pdf_class();
$pdf_view->show_form_title( $settings['show_title'], $form );

/* Output the Single Entry Layout Builder View */
$context = Template_Context::from_template( $view_template );

$zone = 'single';
$rows = Grid::prefixed(
	GravityView_Layout_Builder::ID,
	static function () use ( $zone, $context ) {
		return Grid::get_rows_from_collection( $context->fields, $zone );
	}
);

/*
 * If all fields in the row are filtered by the PDF settings then remove the entire row from display
 */
$filtered_rows = array_filter(
	$rows,
	function ( $row ) use ( $zone, $context, $view_fields ) {
		$does_row_have_fields = false;
		foreach ( $row as $areas ) {
			foreach ( $areas as $area ) {
				$position_id = $zone . '_' . $area['areaid'];

				$area_fields = $context->fields->by_position( $position_id )->all();
				foreach ( $area_fields as $field ) {
					foreach ( ( $view_fields[ $position_id ] ?? [] ) as $view_field ) {
						if ( $field->UID === $view_field->UID ) {
							$does_row_have_fields = true;
							break 4;
						}
					}
				}
			}
		}

		return $does_row_have_fields;
	}
);

?>
	<div class="gv-layout-builder-view gv-layout-builder-view--entry gv-grid">
		<?php foreach ( $filtered_rows as $r => $row ): ?>
			<div class="gv-grid-row">
				<?php
				foreach ( $row as $column => $areas ):
					?>
					<div class="gv-grid-col <?php echo esc_attr( 'gv-grid-col-' . $column ); ?>">
						<div class="inner-container">
							<?php
							if ( empty( $areas ) ) {
								echo '&nbsp;';

								continue;
							}

							foreach ( $areas as $area ) {
								$position_id = $zone . '_' . $area['areaid'];

								$area_fields = $context->fields->by_position( $position_id )->all();
								if ( empty( $area_fields ) ) {
									echo '&nbsp;';

									continue;
								}

								foreach ( $area_fields as $k => $field ) {
									/* skip if the field isn't found in our filtered fields */
									$match = false;
									foreach ( ( $view_fields[ $position_id ] ?? [] ) as $view_field ) {
										if ( $field->UID === $view_field->UID ) {
											$match = true;

											/* replace the field because it could be filtered with `gfpdf_gv_view_field` */
											$field = $view_field;

											break;
										}
									}

									if ( ! $match ) {
										echo '&nbsp;';

										continue;
									}

									$field->custom_class = $r === 0 && $k === 0 ? 'first gv-field-wrapper' : 'gv-field-wrapper';

									$content = $view_template->the_field( $field, $view_entry );
									if ( empty( $content ) ) {
										echo '&nbsp;';
										continue;
									}

									/* Convert label SPAN to a DIV */
									$qp         = new \GFPDF\Helper\Helper_QueryPath();
									$label_tags = $qp->html5( $content )->find( 'span.gv-field-label' );

									foreach ( $label_tags as $label_tag ) {
										/** @var $label_tag \GFPDF_Vendor\QueryPath\DOMQuery */
										$label_tag->replaceWith( sprintf( '<div class="gv-field-label">%s</div>', $label_tag->innerHTML5() ) );
									}

									$content = $label_tags->top( 'html' )->innerHTML5();

									echo ! empty( $content ) ? Kses::parse( $content ) : '&nbsp;'; /* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */

								}
							}
							?>
						</div>
					</div>
				<?php endforeach; // $row ?>
			</div>
		<?php endforeach; // $rows ?>
	</div>
<?php
