<?php

/*
 * Template Name: GravityView Single Entry Table Layout
 * Version: 0.1
 * Description: A PDF template designed specifically for outputting the GravityView Single Entry Table and Datatable Layout.
 * Author: Gravity PDF
 * Author URI: https://gravitypdf.com
 * Group: GravityView
 * License: GPLv2
 * Required PDF Version: 6.0
 */

use GFPDF\Statics\Kses;
use GV\GF_Form;
use GV\Template_Context;

/* Prevent direct access to the template */
if ( ! class_exists( 'GFForms' ) ) {
	return;
}

/**
 * The GravityView PDF Template has access to the following:
 *
 * @var \GV\View           $view          The current GravityView object
 * @var \GV\Entry          $view_entry    The current GravityView entry object
 * @var \GV\Entry_Template $view_template The original template class used to output the Single Entry Layout in GravityView
 * @var array              $view_fields   A curated array of Single Entry Layout fields, grouped by position
 *
 * @var array              $form          The current Gravity Form array
 * @var array              $entry         The raw entry data
 * @var array              $form_data     The processed entry data stored in an array
 * @var array              $settings      The current PDF configuration
 * @var object             $config        The initialized template config class (if any)
 * @var object             $gfpdf         The main Gravity PDF object containing all the helper classes
 * @var array              $args          Contains an array of all variables - the ones being described right now - passed to the template
 */

$show_label = apply_filters( 'gfpdf_gv_show_label', true, $view, $view_entry, 'table' );
$show_label = apply_filters( 'gfpdf_gv_show_label_' . $view->ID, $show_label, $view, $view_entry, 'table' );
$show_label = apply_filters( 'gfpdf_gv_show_label_table', $show_label, $view, $view_entry, 'table' );

?>

<style>
  @page {
	margin: <?php echo esc_attr( $settings['margin'] ); ?>;
  }

  /* Header / Footer */
  #header p,
  .footer p {
	margin: 0;
  }

  /* Basic Styles */
  h1,
  h2,
  h3,
  h4,
  h5,
  h6 {
	margin: 0.5rem 0;
	padding: 0;
  }

  ul,
  ol {
	margin: 0;
	padding: 0 0 0 5mm;
  }

  p {
	margin: 0 0 3.5mm;
	padding: 0
  }

  .field-type-textarea-p td,
  .field-type-textarea-ul td,
  .field-type-textarea-ol td {
	padding-bottom: 3.5mm;
  }

  .field-type-textarea-h1 td,
  .field-type-textarea-h2 td,
  .field-type-textarea-h3 td,
  .field-type-textarea-h4 td,
  .field-type-textarea-h5 td,
  .field-type-textarea-h6 td {
	padding: 0.5rem 0;
  }

  /*
   * Image alignment
   */
  .alignleft {
	float: left;
  }

  .alignright {
	float: right;
  }

  .aligncenter {
	text-align: center;
  }

  p.alignleft {
	text-align: left;
	float: none;
  }

  p.alignright {
	text-align: right;
	float: none;
  }

  /* Single View Layout Title */
  #form_title {
	margin: 0 0 3mm;
	padding: 0;
	font-size: 150%;
  }

  /* Section Break Field */
  .field-type-section .label {
	font-size: 125%;
	margin: 1.5mm 0 0.5mm;
	padding: 0;
  }

  /* List Field */
  .gfield_list th {
	text-align: left;
  }

  .gfield_list th,
  .gfield_list td {
	padding: .3em;
  }

  /* Notes Field */
  .entry-detail-note {
	margin-top: -1px; /* Fixes border overlap */
	padding: 1em .5em 0 1em;
	border: 1px solid #eee;
  }

  .gv-note-avatar {
	width: 42px;
	height: 42px;
	float: left;
	padding-right: 10px;
  }

  .gv-note-author {
	font-weight: bold;
	font-size: 1.1rem;
	line-height: 1.1;
	margin: 0.75mm 0;
	padding: 0;
  }

  .gv-note-added-on {
	font-weight: normal;
	font-size: .875rem;
  }

  .gv-note-content {
	clear: both;
	margin: 0;
	padding: 1em 0 0 0;
	line-height: 1.8em;
  }

  /*
   * Checkmark
   */
  .dashicons-yes {
	font-family: dejavusanscondensed;
	font-size: 120%;
  }

  /*
   * Likert
   */
  .gsurvey-likert-choice {
	text-align: center;
	padding: .3em;
  }

  /*
   * Rating
   */
  .gv-field-survey-star {
	width: 6mm;
  }

  /* Common Layout */
  .row-separator td {
	padding: 1.5mm;
  }

  .inner-container {

  }

  .label {
	font-weight: bold;
  }

  th, td {
	vertical-align: top;
	border: none;
  }

  /* Table View Layout */
  <?php if ( $show_label ): ?>
  .label {
	width: 23%;
  }
  <?php endif; ?>

  /* Fix nested table padding issue */
  .row-separator td td {
	padding-left: 0;
	padding-right: 0;
  }

  <?php
	/* Add Additional CSS, if present */
	echo esc_html( wp_filter_nohtml_kses( $settings['additional_css'] ?? '' ) );

	do_action( 'gfpdf_gv_template_styles', $view, $view_entry );
	do_action( 'gfpdf_gv_template_styles_' . $view->ID, $view, $view_entry );
	do_action( 'gfpdf_gv_template_styles_table', $view, $view_entry );
	?>
</style>

<?php

/* Output the page title, if configured */
$single_title  = $view->settings->get( 'single_title', '' );
$single_title  = ! empty( $single_title ) ? $single_title : $view->get_post()->post_title;
$form['title'] = $single_title;

/** @var \GFPDF\View\View_PDF $pdf_view */
$pdf_view = \GPDFAPI::get_pdf_class();
$pdf_view->show_form_title( $settings['show_title'], $form );

/* Output the Single Entry Table View */
$fields  = $view_fields['single_table-columns'] ?? [];
$context = Template_Context::from_template( $view_template );
$fields  = apply_filters( 'gravityview/template/table/fields', $fields, $context );

echo '<table class="view" autosize="1">';

foreach ( $fields as $field ) {
	/** @var $field \GV\GF_Field */

	/* A view may be comprised of multiple forms (using the Multiple Forms extension) and/or entries */
	$field_form  = GF_Form::by_id( $field->form_id ) ?: $view->form;
	$field_entry = $view_entry->from_field( $field );

	if ( ! $field_entry ) {
		continue;
	}

	$context    = Template_Context::from_template( $view_template, [ 'field' => $field ] );
	$hide_empty = apply_filters( 'gravityview/render/hide-empty-zone', $view->settings->get( 'hide_empty_single', false ), $context );
	$label      = apply_filters( 'gravityview/template/field/label', $field->get_label( $view, $field_form, $field_entry ), $context );

	/* Get the field markup and output */
	$output = gravityview_field_output(
		[
			'label'      => $label,
			'value'      => $view_template->the_field( $field ),
			'markup'     => sprintf(
				'<tr id="field-%1$s" class="row-separator field-type-%2$s {{ class }}">
									' . ( $show_label ? '<td class="label">{{ label }}</td>' : '' ) . '
									<td class="value">{{ value }}</td>
							</tr>',
				$field->UID,
				$field->type
			),
			'hide_empty' => $hide_empty,
			'zone_id'    => 'single_table-columns',
		],
		$context
	);

	/* Fix textarea/rich text spacing */
	try {
		$qp                    = new \GFPDF\Helper\Helper_QueryPath();
		$textarea_inner_fields = $qp->html5( $output )
									->find( '.field-type-textarea .value' )
									->find( 'p,ol,ul,h1,h2,h3,h4,h5,h6' );

		foreach ( $textarea_inner_fields as $inner_field ) {
			$inner_field->wrap(
				sprintf(
					'<table class="field-type-textarea-%1$s"><tr><td class="%2$s" style="%3$s"></td></tr></table>',
					$inner_field->tag(),
					$inner_field->attr( 'class' ),
					$inner_field->attr( 'style' )
				)
			);
		}

		$output = $textarea_inner_fields->top( 'html' )->innerHTML5();
	} catch ( \Exception $e ) {
		/* Do nothing */
	}

	Kses::output( $output );
}

echo '</table>';
