<?php

namespace GFPDF\Plugins\PdfForGravityView;

use GFPDF\Plugins\PdfForGravityView\GravityView\Registration;
use GFPDF\Plugins\PdfForGravityView\Pdf\Authentication;
use GFPDF\Plugins\PdfForGravityView\Pdf\FieldFormat;
use GFPDF\Plugins\PdfForGravityView\Pdf\FieldMiddleware;
use GFPDF\Plugins\PdfForGravityView\Pdf\Notifications;
use GFPDF\Plugins\PdfForGravityView\Pdf\Renderer;
use GFPDF\Plugins\PdfForGravityView\Pdf\Settings;

use GFPDF\Helper\Licensing\EDD_SL_Plugin_Updater;
use GFPDF\Helper\Helper_Abstract_Addon;
use GFPDF\Helper\Helper_Singleton;
use GFPDF\Helper\Helper_Logger;
use GFPDF\Helper\Helper_Notices;
use GFPDF\Plugins\PdfForGravityView\Pdf\ThirdParty\GravityWizNestedForms;

/**
 * @package     PDF for GravityView
 * @copyright   Copyright (c) 2025, Blue Liquid Designs
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/* Exit if accessed directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/* Load Composer */
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../api.php';

/**
 * @package GFPDF\Plugins\PdfForGravityView
 */
class Bootstrap extends Helper_Abstract_Addon {

	/**
	 * @param array $classes
	 *
	 * @return void
	 *
	 * @since 0.1
	 */
	public function init( $classes = [] ) {
		$this->maybe_do_upgrade_routine();

		$pdf_renderer = new Renderer(
			new Settings(),
			\GPDFAPI::get_form_class(),
			\GPDFAPI::get_data_class(),
			\GPDFAPI::get_misc_class(),
			\GPDFAPI::get_templates_class()
		);

		$classes = array_merge(
			$classes,
			[
				new Registration(),
				new Authentication(),
				new FieldMiddleware(),
				new FieldFormat(),
				new Notifications(),
				new GravityWizNestedForms(),
				$pdf_renderer,
			]
		);

		/* Run the setup */
		parent::init( $classes );

		/* Give the API access to the PDF rendering object */
		\PDFGV_API::$_renderer = $pdf_renderer;
	}

	/**
	 * Add a plugin upgrade routine
	 *
	 * @return void
	 *
	 * @since 0.8.0
	 */
	public function maybe_do_upgrade_routine() {
		$db_version     = get_option( 'gfpdf_for_gv_current_version', '0.0.0' );
		$plugin_version = $this->get_version();

		/* Do nothing if the versions are the same */
		if ( version_compare( $plugin_version, $db_version, '==' ) ) {
			return;
		}

		$update_versions = [
			'0.8.0' => [ $this, 'do_upgrade_0_8' ],
		];

		foreach ( $update_versions as $version => $callback ) {
			/* Only run when the DB version is less than $version, and the plugin version is more than or equal to $version */
			if ( version_compare( $db_version, $version, '<' ) && version_compare( $plugin_version, $version, '>=' ) ) {
				call_user_func( $callback );
			}
		}

		/* Store the new version */
		update_option( 'gfpdf_for_gv_current_version', $plugin_version );
	}

	/**
	 * @return void
	 *
	 * @since 0.8
	 */
	public function do_upgrade_0_8() {
		/* Upgrade the PDF Email Gravity Form */
		$form_id = get_option( 'gfpdf_for_gv_pdf_email_form_id' );
		if ( ! $form_id ) {
			return;
		}

		\GFAPI::update_form( wp_json_file_decode( __DIR__ . '/../assets/forms/pdf-email-form.json', [ 'associative' => true ] ), $form_id );
	}

	/**
	 * Check the plugin's license is active and initialise the EDD Updater
	 *
	 * @since 0.1
	 */
	public function plugin_updater(): void {
		/* Skip over this addon if license status isn't active */
		$license_info = $this->get_license_info();

		new EDD_SL_Plugin_Updater(
			$this->data->store_url,
			$this->get_main_plugin_file(),
			[
				'version'   => $this->get_version(),
				'license'   => $license_info['license'],
				'item_name' => $this->get_short_name(),
				'author'    => $this->get_author(),
				'beta'      => false,
			]
		);
	}
}
