<?php

use GP_Email_Validator\Validators\Validator;

if ( ! class_exists( 'GP_Plugin' ) ) {
	return;
}

class GP_Email_Validator extends GP_Plugin {

	private static $instance = null;

	/**
	 * @var array<string, Validator> Validator instances
	 */
	protected $validators = [];

	protected $_version     = GP_EMAIL_VALIDATOR_VERSION;
	protected $_path        = 'gp-email-validator/gp-email-validator.php';
	protected $_full_path   = __FILE__;
	protected $_slug        = 'gp-email-validator';
	protected $_title       = 'GP Email Validator';
	protected $_short_title = 'Email Validator';

	/**
	 * @var string $_capabilities_settings_page The capability needed to access the Add-On settings page.
	 */
	protected $_capabilities_settings_page = 'gp-email-validator_settings';

	/**
	 * @var string $_capabilities_form_settings The capability needed to access the Add-On form settings page.
	 */
	protected $_capabilities_form_settings = 'gp-email-validator_form_settings';

	/**
	 * @var string $_capabilities_uninstall The capability needed to uninstall the Add-On.
	 */
	protected $_capabilities_uninstall = 'gp-email-validator_uninstall';

	public static function get_instance() {
		if ( self::$instance === null ) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	/**
	 * Give the form settings and plugin settings panels a nice shiny icon.
	 */
	public function get_menu_icon() {
		return $this->get_base_url() . '/assets/images/icon.svg';
	}

	public function minimum_requirements() {
		return [
			'php'          => [
				'version' => '7.2',
			],
			'gravityforms' => [
				'version' => '2.8',
			],
			'wordpress'    => [
				'version' => '6.0',
			],
			'plugins'      => [
				'gravityperks/gravityperks.php',
			],
		];
	}

	public function pre_init() {
		parent::pre_init();
		$this->register_validators();
	}

	public function init() {
		parent::init();

		$this->init_active_validator();

		load_plugin_textdomain( $this->_slug, false, basename( dirname( __file__ ) ) . '/languages/' );

		add_action( 'gperk_field_settings', [ $this, 'field_settings_ui' ] );
		add_action( 'gform_editor_js', [ $this, 'field_settings_js' ] );
		add_filter( 'gform_field_css_class', [ $this, 'add_field_class' ], 10, 3 );
	}

	/**
	 * Register validator classes
	 */
	protected function register_validators() {
		$validator_classes = [
			'basic'      => 'GP_Email_Validator\Validators\Basic_Validator',
			'kickbox'    => 'GP_Email_Validator\Validators\Kickbox_Validator',
			'zerobounce' => 'GP_Email_Validator\Validators\Zerobounce_Validator',
		];

		foreach ( $validator_classes as $service => $class ) {
			/** @var class-string<Validator> $class */
			$this->validators[ $service ] = new $class();
		}
	}

	/**
	 * Initialize only the active validator
	 */
	protected function init_active_validator() {
		$validator = $this->get_validator();
		$validator->init();
	}

	/**
	 * Get the appropriate validator based on settings
	 *
	 * @return Validator
	 */
	protected function get_validator() {
		$service = $this->get_plugin_setting( 'validator_service' );
		return isset( $this->validators[ $service ] ) ? $this->validators[ $service ] : $this->validators['basic'];
	}

	public function init_admin() {
		parent::init_admin();

		GravityPerks::enqueue_field_settings();
	}

	public function scripts() {
		$scripts = [
			[
				'handle'    => 'gp_email_validator_admin',
				'src'       => $this->get_base_url() . '/js/built/gp-email-validator-admin.js',
				'version'   => $this->_version,
				'deps'      => [ 'jquery' ],
				'in_footer' => true,
				'enqueue'   => [
					[
						'admin_page' => [ 'plugin_settings', 'entry_view', 'entry_list' ],
						'tab'        => $this->_slug,
					],
				],
			],
			[
				'handle'    => 'gp_email_validator',
				'src'       => $this->get_base_url() . '/js/built/gp-email-validator.js',
				'version'   => $this->_version,
				'in_footer' => true,
				'strings'   => [
					'nonce'      => wp_create_nonce( 'gpev_get_suggestion' ),
					'ajaxUrl'    => admin_url( 'admin-ajax.php' ),
					'didYouMean' => esc_html__( 'Did you mean ', 'gp-email-validator' ),
				],
				'enqueue'   => [
					[ $this, 'should_enqueue_frontend' ],
				],
			],
		];

		return array_merge( parent::scripts(), $scripts );
	}

	/**
	 * Determines if the frontend JavaScript/styles should be enqueued.
	 *
	 * @param GFForm $form
	 *
	 * @return bool
	 */
	public function should_enqueue_frontend( $form ) {
		if ( ! is_array( $form ) || empty( $form['fields'] ) ) {
			return false;
		}

		// If suggestions are NOT enabled for the current validator, don't enqueue the frontend JS.
		if ( ! $this->get_validator()->should_show_suggestions() ) {
			return false;
		}

		// Check that there is a GPEV field in the form.
		foreach ( $form['fields'] as $field ) {
			if ( $this->is_email_validator_field( $field ) ) {
				return true;
			}
		}

		return false;
	}

	public function styles() {
		$scripts = [
			[
				'handle'  => 'gp_email_validator_admin',
				'src'     => $this->get_base_url() . '/assets/css/admin.css',
				'version' => $this->_version,
				'enqueue' => [
					[
						'admin_page' => [ 'plugin_settings', 'entry_view', 'entry_list' ],
						'tab'        => $this->_slug,
					],
				],
			],
			[
				'handle'  => 'gp_email_validator',
				'src'     => $this->get_base_url() . '/assets/css/frontend.css',
				'version' => $this->_version,
				'enqueue' => [
					[ $this, 'should_enqueue_frontend' ],
				],
			],
		];

		return array_merge( parent::scripts(), $scripts );
	}

	/**
	 * @param GF_Field $field
	 *
	 * @return boolean
	 */
	public function is_email_validator_field( $field ) {
		return rgar( $field, 'type' ) === 'email' && ! rgar( $field, 'gpevDisable', false );
	}

	/**
	 * Add 'gpev-field' class to email fields that have GPEV enabled.
	 *
	 * @param string $classes
	 * @param GF_Field $field
	 * @param GFForm $form
	 *
	 * @return string
	 */
	public function add_field_class( $classes, $field, $form ) {
		if ( ! $this->is_email_validator_field( $field ) ) {
			return $classes;
		}

		$classes .= ' gpev-field';
		return $classes;
	}

	## Admin Field Settings

	public function field_settings_ui( $position ) {
		?>

		<li class="gpev-field-setting field_setting" style="display:none;">
			<input type="checkbox" value="1" id="gpev-disable" onchange="SetFieldProperty( 'gpevDisable', ! this.checked );" />
			<label for="gpev-disable" class="inline">
				<?php _e( 'Enable Email Validator', 'gp-email-validator' ); ?>
				<?php gform_tooltip( $this->_slug . '_toggle' ); ?>
			</label>
		</li>

		<?php
	}

	public function field_settings_js() {
		?>
		<script type="text/javascript">
			( function( $ ) {

				$( document ).ready( function(){
					for( fieldType in fieldSettings ) {
						if( fieldSettings.hasOwnProperty( 'email' ) ) {
							fieldSettings[ 'email' ] += ', .gpev-field-setting';
						}
					}
				} );

				$( document ).bind( 'gform_load_field_settings', function( event, field, form ) {
					$( '#gpev-disable' ).prop( 'checked', ! field['gpevDisable'] );
				} );

			} )( jQuery );
		</script>

		<?php
	}

	public function tooltips( $tooltips ) {
		$tooltips[ $this->_slug . '_toggle' ] = sprintf(
			'<h6>%s</h6> %s',
			__( 'GP Email Validator', 'gp-email-validator' ),
			__( 'Toggle GP Email Validator for the current field.', 'gp-email-validator' )
		);

		return $tooltips;
	}

	/**
	 * Override get_plugin_settings to allow easy filtering.
	 *
	 * Returns the currently saved plugin settings
	 *
	 * @return array|false Returns the plugin settings or false if the settings haven't been saved yet.
	 */
	public function get_plugin_settings() {
		$settings = parent::get_plugin_settings();

		/**
		 * Filter GP Email Validator's settings.
		 *
		 * @param array $settings The plugin settings.
		 *
		 * @since 1.0
		 */
		return apply_filters( 'gp_email_validator_settings', $settings );
	}

	/**
	 * Defines the settings for the plugin's global settings such as API key. Accessible via Forms » Settings
	 *
	 * @return array[]
	 */
	public function plugin_settings_fields() {
		$fields = [
			[
				'name' => 'automatically_enabled_notice',
				'type' => 'html',
				'html' => esc_html__( 'Email Validator is automatically enabled on all Email fields by default. You can disable per field via the \'Enable Email Validator\' field setting.', 'gp-email-validator' ),
			],
			[
				'name'          => 'validator_service',
				'tooltip'       => __( 'Choose which validation service to use. Basic provides syntax and DNS validation, while Kickbox and ZeroBounce offer advanced validation through their APIs.', 'gp-email-validator' ),
				'label'         => __( 'Validator Service', 'gp-email-validator' ),
				'type'          => 'radio',
				'default_value' => 'basic',
				'horizontal'    => true,
				'choices'       => [
					[
						'value' => 'basic',
						'label' => __( 'Basic', 'gp-email-validator' ),
					],
					[
						'value' => 'kickbox',
						'label' => __( 'Kickbox', 'gp-email-validator' ),
					],
					[
						'value' => 'zerobounce',
						'label' => __( 'ZeroBounce', 'gp-email-validator' ),
					],
				],
			],
		];

		// Get settings from all validators
		foreach ( $this->validators as $service => $validator ) {
			$validator_settings = $validator->get_settings();
			foreach ( $validator_settings as $setting ) {
				// Add dependency to show only when this validator is selected
				if ( ! isset( $setting['dependency'] ) ) {
					$setting['dependency'] = [
						'live'   => true,
						'fields' => [
							[
								'field'  => 'validator_service',
								'values' => [ $service ],
							],
						],
					];
				} else {
					$setting['dependency']['fields'][] = [
						'field'  => 'validator_service',
						'values' => [ $service ],
					];
				}

				$fields[] = $setting;
			}
		}

		return [
			[
				'title'  => __( 'Email Validator', 'gp-email-validator' ),
				'fields' => $fields,
			],
			[
				'title'       => __( 'Domain Validator', 'gp-email-validator' ),
				'description' => __( 'Create a list of email domains that are automatically blocked or allowed.', 'gs-email-validator' ),
				'fields'      => [
					[
						'name'          => 'domain_validator_mode',
						'tooltip'       => __( 'Choose how to handle specific email domains. "Allow" mode only accepts emails from listed domains, while "Block" mode rejects emails from listed domains. "None" disables domain validation.', 'gp-email-validator' ),
						'label'         => __( 'Domain Validator Mode', 'gp-email-validator' ),
						'type'          => 'radio',
						'default_value' => 'none',
						'horizontal'    => true,
						'choices'       => [
							[
								'value' => 'none',
								'label' => __( 'None', 'gp-email-validator' ),
							],
							[
								'value' => 'allow',
								'label' => __( 'Allow', 'gp-email-validator' ),
							],
							[
								'value' => 'block',
								'label' => __( 'Block', 'gp-email-validator' ),
							],
						],
						'fields'        => [
							[
								'name'       => 'allowed_domains',
								'label'      => __( 'Email Domains', 'gp-email-validator' ),
								'type'       => 'textarea',
								'tooltip'    => __( 'Enter one domain per line. Only email addresses from these domains will be accepted. Example: company.com', 'gp-email-validator' ),
								'dependency' => [
									'live'   => true,
									'fields' => [
										[
											'field'  => 'domain_validator_mode',
											'values' => [ 'allow' ],
										],
									],
								],
							],
							[
								'name'       => 'blocked_domains',
								'label'      => __( 'Email Domains', 'gp-email-validator' ),
								'type'       => 'textarea',
								'tooltip'    => __( 'Enter one domain per line. Email addresses from these domains will be rejected. Example: disposable-email.com', 'gp-email-validator' ),
								'dependency' => [
									'live'   => true,
									'fields' => [
										[
											'field'  => 'domain_validator_mode',
											'values' => [ 'block' ],
										],
									],
								],
							],
						],
					],
				],
			],
		];
	}
}

function gp_email_validator() {
	return GP_Email_Validator::get_instance();
}

GFAddOn::register( 'GP_Email_Validator' );
