<?php

namespace GP_Email_Validator\Results;

use GP_Email_Validator\Validators\Validator;

/**
 * Abstract class for validation results.
 *
 * Provides a consistent interface for handling validation results
 * across different validation services (Basic, Kickbox, ZeroBounce).
 *
 * @template Metadata
 *
 * @phpstan-type ValidationReason array{
 *   code: string,
 *   message: string,
 *   details?: string
 * }
 */
abstract class Validation_Result {
	/**
	 * @var string The email address that was validated
	 */
	protected $email;

	/**
	 * @var bool Whether the email is valid
	 */
	protected $is_valid;

	/**
	 * @var string Primary validation status (valid, invalid, etc.)
	 */
	protected $status;

	/**
	 * @var ValidationReason[] Reasons for validation result
	 */
	protected $reasons = [];

	/**
	 * @var string|null Suggested correction for typos
	 */
	protected $suggestion;

	/**
	 * @var string|null Sanitized version of email (e.g. Gmail without plus)
	 */
	protected $sanitized_email;

	/**
	 * @var Metadata Additional metadata from the validator
	 */
	protected $metadata;

	/**
	 * @var Validator The validator instance
	 */
	protected $validator;

	/**
	 * Create a new validation result
	 *
	 * @param string $email The email address that was validated
	 * @param array|null $response The raw validation response
	 * @param Validator $validator The validator instance
	 */
	final public function __construct( $email, $response, Validator $validator, $from_array = null ) {
		$this->email     = $email;
		$this->validator = $validator;

		if ( $from_array ) {
			foreach ( $from_array as $key => $value ) {
				if ( ! property_exists( $this, $key ) ) {
					continue;
				}

				$this->{$key} = $value;
			}

			return;
		}

		$this->normalize( $response );
	}

	/**
	 * Setter for reasons.
	 *
	 * @param ValidationReason[] $reasons
	 *
	 * @return void
	 */
	public function set_reasons( array $reasons ) {
		$this->reasons = $reasons;
	}

	/**
	 * Sets is_valid to true.
	 *
	 * @return void
	 */

	public function mark_valid() {
		$this->is_valid = true;
	}

	/**
	 * Sets is_valid to false.
	 */
	public function mark_invalid() {
		$this->is_valid = false;
	}

	/**
	 * Setter for status
	 *
	 * @param string $status
	 *
	 * @return void
	 */
	public function set_status( string $status ) {
		$this->status = $status;
	}

	/**
	 * Create a validation result from array data
	 *
	 * @param array $data The stored validation data
	 * @param Validator $validator The validator instance
	 * @return static|Error_Validation_Result
	 */
	public static function from_array( array $data, Validator $validator ) {
		if ( $data['status'] === 'error' ) {
			return new Error_Validation_Result( $data['email'], null, $validator, $data );
		}

		return new static( $data['email'], null, $validator, $data );
	}

	/**
	 * Convert validation result to array for storage
	 *
	 * @return array
	 */
	public function to_array(): array {
		$array = [];

		foreach ( get_object_vars( $this ) as $key => $value ) {
			if ( is_object( $value ) ) {
				continue;
			}

			$array[ $key ] = $value;
		}

		return $array;
	}

	/**
	 * Normalize the validation response into result fields
	 *
	 * Each validator result class must implement this to handle its specific response format
	 * and set appropriate validation reasons based on the validator's settings.
	 *
	 * @param array $response The validation response
	 */
	abstract protected function normalize( array $response);

	/**
	 * Add a validation reason
	 *
	 * @param string $code The reason code
	 * @param string $message The reason message
	 * @param string|null $details Additional details
	 */
	protected function add_reason( $code, $message, $details = null ) {
		/** @var ValidationReason $reason */
		$reason = [
			'code'    => $code,
			'message' => $message,
		];

		if ( $details ) {
			$reason['details'] = $details;
		}

		$this->is_valid  = false;
		$this->reasons[] = $reason;
	}

	/**
	 * Get whether the email is valid
	 *
	 * @return bool
	 */
	public function is_valid() {
		return $this->is_valid;
	}

	/**
	 * Get the validation status
	 *
	 * @return string
	 */
	public function get_status() {
		return $this->status;
	}

	/**
	 * Get the validation reasons
	 *
	 * @return ValidationReason[]
	 */
	public function get_reasons() {
		return $this->reasons;
	}

	/**
	 * Get suggested correction for typos
	 *
	 * @return string|null
	 */
	public function get_suggestion() {
		return $this->suggestion;
	}

	/**
	 * Get sanitized version of email
	 *
	 * @return string|null
	 */
	public function get_sanitized_email() {
		return $this->sanitized_email;
	}

	/**
	 * Get additional metadata from the validator
	 *
	 * @return Metadata
	 */
	public function get_metadata() {
		return $this->metadata;
	}

	/**
	 * Get the validated email address
	 *
	 * @return string
	 */
	public function get_email() {
		return $this->email;
	}

	/**
	 * Get a formatted error message
	 *
	 * @return string
	 */
	public function get_error_message() {
		if ( empty( $this->reasons ) ) {
			return __( 'Email validation failed.', 'gp-email-validator' );
		}

		$reason   = reset( $this->reasons );
		$messages = $this->validator->get_error_messages();

		if ( ! empty( $reason['message'] ) ) {
			$message = $reason['message'];
		} elseif ( isset( $messages[ $reason['code'] ] ) ) {
			$message = $messages[ $reason['code'] ];
		} else {
			$message = $reason['code'];
		}

		if ( $this->suggestion ) {
			// translators: %s is the suggested email address
			$message .= ' ' . sprintf( __( 'Did you mean %s?', 'gp-email-validator' ), $this->suggestion );
		}

		return $message;
	}
}
