<?php

namespace GP_Email_Validator\Results;

use GP_Email_Validator\Validators\Zerobounce_Validator;

/**
 * ZeroBounce validation result.
 *
 * @phpstan-type ZeroBounceStatus 'valid'|'invalid'|'catch-all'|'unknown'|'spamtrap'|'abuse'|'do_not_mail'
 * @phpstan-type ZeroBounceSubStatus 'alternate'|'antispam_system'|'greylisted'|'mail_server_temporary_error'|'forcible_disconnect'|'mail_server_did_not_respond'|'timeout_exceeded'|'failed_smtp_connection'|'mailbox_quota_exceeded'|'exception_occurred'|'possible_trap'|'role_based'|'global_suppression'|'mailbox_not_found'|'no_dns_entries'|'failed_syntax_check'|'possible_typo'|'unroutable_ip_address'|'leading_period_removed'|'does_not_accept_mail'|'alias_address'|'role_based_catch_all'|'disposable'|'toxic'
 * @phpstan-type ZeroBounceMetadata array{
 *   address: string,
 *   status: ZeroBounceStatus,
 *   sub_status: ZeroBounceSubStatus|null,
 *   account: string,
 *   domain: string,
 *   did_you_mean: string|null,
 *   domain_age_days: int|null,
 *   active_in_days: string|null,
 *   free_email: bool,
 *   mx_found: bool,
 *   mx_record: string|null,
 *   smtp_provider: string|null,
 *   firstname: string|null,
 *   lastname: string|null,
 *   gender: string|null,
 *   city: string|null,
 *   region: string|null,
 *   zipcode: string|null,
 *   country: string|null,
 *   processed_at: string
 * }
 *
 * @extends Validation_Result<ZeroBounceMetadata>
 */
class Zerobounce_Validation_Result extends Validation_Result {
	/**
	 * @var Zerobounce_Validator
	 */
	protected $validator;

	protected function normalize( array $response ) {
		$rejection_settings = $this->validator->get_rejection_settings();

		// Set basic fields, we always start with is_valid being true until going through rejection reasons.
		$this->is_valid   = ! empty( $response['processed_at'] );
		$this->status     = $response['status'] ?? 'unknown';
		$this->suggestion = $response['did_you_mean'] ?? null;

		// Set metadata
		/** @var ZeroBounceMetadata $metadata */
		$metadata = [
			'address'         => $response['address'] ?? $this->email,
			'status'          => $response['status'] ?? 'unknown',
			'sub_status'      => $response['sub_status'] ?? null,
			'account'         => $response['account'] ?? '',
			'domain'          => $response['domain'] ?? '',
			'did_you_mean'    => $response['did_you_mean'] ?? null,
			'domain_age_days' => $response['domain_age_days'] ?? null,
			'active_in_days'  => $response['active_in_days'] ?? null,
			'free_email'      => $response['free_email'] ?? false,
			'mx_found'        => $response['mx_found'] ?? false,
			'mx_record'       => $response['mx_record'] ?? null,
			'smtp_provider'   => $response['smtp_provider'] ?? null,
			'firstname'       => $response['firstname'] ?? null,
			'lastname'        => $response['lastname'] ?? null,
			'gender'          => $response['gender'] ?? null,
			'city'            => $response['city'] ?? null,
			'region'          => $response['region'] ?? null,
			'zipcode'         => $response['zipcode'] ?? null,
			'country'         => $response['country'] ?? null,
			'processed_at'    => $response['processed_at'] ?? '',
		];

		$this->metadata = $metadata;

		$messages = $this->validator->get_error_messages();

		// Check if result should be rejected based on settings and status alone.
		if ( isset( $rejection_settings[ $metadata['status'] ] ) && $rejection_settings[ $metadata['status'] ] ) {
			$this->add_reason(
				$metadata['status'],
				$messages[ $metadata['status'] ],
				// Add sub-status details if available
				isset( $metadata['sub_status'] ) ? $this->validator->get_sub_status_messages()[ $metadata['sub_status'] ] ?? null : null
			);
		}

		// If response is coming in with a "reason", use it.
		if ( ! empty( $response['reason'] ) ) {
			$this->add_reason(
				$metadata['status'],
				$response['reason']
			);
		}
	}
}
