<?php

namespace GP_Email_Validator\Validators;

use GP_Email_Validator\Results\Validation_Result;
use GP_Email_Validator\Results\Error_Validation_Result;
use GP_Email_Validator\Results\Domain_Validation_Result;

/**
 * Abstract base class for email validators.
 *
 * @template VR
 */
abstract class Validator {
	/**
	 * @var string The name of the validation service
	 */
	protected $service_name;

	/**
	 * @var array<int, VR> Validation results keyed by field ID
	 */
	protected $validation_results = [];

	/**
	 * Initialize the validator
	 */
	public function init() {
		add_filter( 'gform_entry_meta', [$this, 'register_entry_meta'], 10, 2 );
		add_action( 'gform_entry_created', [$this, 'store_validation_result'], 10, 2 );
		add_filter( 'gform_entry_field_value', [$this, 'display_entry_value'], 10, 4 );
		add_filter( 'gform_entries_field_value', [$this, 'display_list_value'], 10, 4 );
		add_filter( 'gform_field_validation', [$this, 'validate_field'], 10, 4 );

		// Add an AJAX endpoint for getting suggested email addresses that will show when blurring an email field.
		if ( $this->should_show_suggestions() ) {
			add_action( 'wp_ajax_nopriv_gpev_get_suggestion', [$this, 'ajax_get_suggestion'] );
			add_action( 'wp_ajax_gpev_get_suggestion', [$this, 'ajax_get_suggestion'] );
		}
	}

	/**
	 * Get the validation result class for this validator
	 *
	 * @return class-string<VR>
	 */
	protected function get_result_class(): string {
		$validator_class_parts = explode( '\\', get_class( $this ) );
		$validator_name        = end( $validator_class_parts );
		$validator_name        = str_replace( '_Validator', '', $validator_name );

		return 'GP_Email_Validator\Results\\' . $validator_name . '_Validation_Result';
	}

	/**
	 * List out the allowed statuses, used for filtering and conditional logic.
	 *
	 * @return string[]
	 */
	abstract public function get_allowed_statuses();

	/**
	 * Validate an email address
	 *
	 * @param string $value The email address to validate
	 * @param int|null $field_id The field ID being validated
	 * @return VR|null Returns null if email is empty
	 */
	abstract public function validate( $value, $field_id = null);

	/**
	 * Gets the meta key for the entire result for an email field.
	 *
	 * @param int $field_id The field ID
	 */
	public function get_meta_key( $field_id ) {
		return "gpev_{$this->service_name}_result_{$field_id}";
	}

	/**
	 * Gets the meta key for the result status for an email field.
	 *
	 * @param int $field_id The field ID
	 */
	public function get_status_meta_key( $field_id ) {
		return "gpev_{$this->service_name}_status_{$field_id}";
	}

	/**
	 * Validate an email address. Calls abstract validate method along with handling domain validation.
	 *
	 * @param string|array{string, string} $value The email address to validate
	 * @param int|null $field_id The field ID being validated
	 * @return VR|null Returns null if email is empty
	 */
	public function perform_validation( $value, $field_id = null ) {
		if ( rgblank( $value ) ) {
			return null;
		}

		// If the email field is an array containing two strings, it is an Email field with confirmation enabled.
		// Return null if the email field is empty or the email confirmation does not match.
		if ( is_array( $value ) ) {
			if ( rgblank( $value[0] ) || $value[0] !== $value[1] ) {
				return null;
			}

			$value = $value[0];
		}

		// Check domain validation first
		$domain_validation = $this->check_domain_validation( $value );
		if ( $domain_validation !== null ) {
			return $this->create_validation_result( $value, $domain_validation );
		}

		// Proceed with actual validator-specific validation
		return $this->validate( $value, $field_id );
	}

	/**
	 * Get error messages for each type of validation failure
	 *
	 * @return array<string, string>
	 */
	abstract public function get_error_messages();

	/**
	 * Get technical details for tooltip
	 *
	 * @param VR $result The validation result
	 * @return array Array of technical detail strings
	 */
	abstract protected function get_technical_details( $result): array;

	/**
	 * Get additional details for tooltip
	 *
	 * @param VR $result The validation result
	 * @return array Array of additional detail strings
	 */
	protected function get_additional_details( $result ): array {
		return [];
	}

	/**
	 * Get validation badge status
	 *
	 * @param VR $result The validation result
	 * @return string One of: 'error', 'success', 'warning'
	 */
	protected function get_validation_badge_status( $result ): string {
		if ( $result instanceof Error_Validation_Result ) {
			return 'error';
		}

		if ( in_array( $result->get_status(), $this->get_success_statuses(), true ) ) {
			return 'success';
		}

		return 'warning';
	}

	/**
	 * Get statuses that should show a success indicator
	 *
	 * @return array<string>
	 */
	protected function get_success_statuses(): array {
		return ['valid'];
	}

	/**
	 * Get the service name
	 *
	 * @return string
	 */
	public function get_service_name() {
		return $this->service_name;
	}

	/**
	 * Get settings for this validator
	 *
	 * @return array
	 */
	public function get_settings() {
		return [];
	}

	/**
	 * Format an error message for display
	 *
	 * @param string $error_message The specific error message
	 * @return string
	 */
	protected function format_error_message( $error_message ) {
		$message = __( 'The email address entered cannot be used. Please enter a different email address.', 'gp-email-validator' );
		if ( \GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
			$message .= '<br>[Only Visible to Form Editors: ' . $error_message . ']';
		}

		return $message;
	}

	/**
	 * Create a validation result instance
	 *
	 * @param string $value The email address that was validated
	 * @param array $response The validation response
	 * @return VR|Error_Validation_Result|Domain_Validation_Result
	 */
	protected function create_validation_result( $value, array $response ) {
		// If response contains an error, return error result
		if ( isset( $response['error'] ) ) {
			return new Error_Validation_Result( $value, $response, $this );
		}

		// Domain validation errors get a specific result class
		if ( rgar( $response, 'result' ) === 'failed_domain_validation' ) {
			return new Domain_Validation_Result( $value, $response, $this );
		}

		// Use the validator-specific result class for normal validations
		$result_class = $this->get_result_class();
		return new $result_class( $value, $response, $this );
	}

	/**
	 * Get rejection settings for this validator
	 *
	 * Each validator should implement this to return an array of settings that determine
	 * what should cause a validation failure. For example:
	 * - Basic validator: disposable emails, free emails, etc.
	 * - Kickbox validator: risky, unknown, undeliverable results
	 * - ZeroBounce validator: invalid, catch-all, spamtrap results
	 *
	 * @return array<string, bool>
	 */
	abstract public function get_rejection_settings(): array;

	/**
	 * Check if suggestions should be shown
	 *
	 * @return bool
	 */
	public function should_show_suggestions(): bool {
		return false;
	}

	/**
	 * Register entry meta for storing validation results
	 *
	 * @param array $entry_meta The existing entry meta
	 * @param int $form_id The form ID
	 * @return array
	 */
	public function register_entry_meta( $entry_meta, $form_id ) {
		$form = \GFAPI::get_form( $form_id );

		foreach ( $form['fields'] as $field ) {
			if ( ! gp_email_validator()->is_email_validator_field( $field ) ) {
				continue;
			}

			$status_choices = array_map( function( $status ) {
				return [
					'text'  => ucfirst( $status ),
					'value' => $status,
				];
			}, $this->get_allowed_statuses() );

			$label = \GFCommon::get_label( $field );
			$entry_meta[ $this->get_status_meta_key( $field->id ) ] = [
				// translators: %1$s is the field label, %2$s is the validator service name
				'label'                      => sprintf( __( '%1$s (%2$s Result)', 'gp-email-validator' ), $label, ucfirst( $this->service_name ) ),
				'is_numeric'                 => false,
				'is_default_column'          => false,
				'update_entry_meta_callback' => null,
				'filter'                     => [
					'operators' => [ 'is', 'isnot' ],
					'choices'   => $status_choices,
				], // Allows filtering and conditional logic
			];
		}

		return $entry_meta;
	}

	/**
	 * Store validation result in entry meta
	 *
	 * @param array $entry The entry
	 * @param array $form The form
	 */
	public function store_validation_result( $entry, $form ) {
		foreach ( $form['fields'] as $field ) {
			if ( ! gp_email_validator()->is_email_validator_field( $field ) ) {
				continue;
			}

			// Use stored validation result if available
			if ( isset( $this->validation_results[ $field->id ] ) ) {
				$result = $this->validation_results[ $field->id ];
			} else {
				// Fall back to validating again if no stored result
				$email = rgar( $entry, $field['id'] );
				if ( ! $email ) {
					continue;
				}

				$result = $this->perform_validation( $email, $field->id );
			}

			if ( $result ) {
				$data = $result->to_array();
				gform_update_meta( $entry['id'], $this->get_meta_key( $field->id ), $data );
				gform_update_meta( $entry['id'], $this->get_status_meta_key( $field->id ), $data['status'] );
			}
		}
	}

	/**
	 * Format validation result for display in entry list
	 *
	 * @param VR $result The validation result
	 * @return string
	 */
	protected function format_list_value( $result ): string {
		$status  = ucfirst( $result->get_status() );
		$reasons = $result->get_reasons();

		if ( ! empty( $reasons ) ) {
			$reason  = reset( $reasons );
			$status .= ' - ' . $reason['message'];
		}

		return $status;
	}

	/**
	 * Format detailed validation info for tooltip
	 *
	 * @param VR $result The validation result
	 * @return string
	 */
	protected function format_tooltip_details( $result ): string {
		$details = [];

		// Validation Status
		$details[] = sprintf( '<strong>Status:</strong> %s', esc_html( ucwords( $result->get_status() ) ) );

		// Validation Reasons
		$reasons = $result->get_reasons();
		if ( ! empty( $reasons ) ) {
			$reason_texts = [];
			foreach ( $reasons as $reason ) {
				$reason_text = $reason['message'];
				if ( ! empty( $reason['details'] ) ) {
					$reason_text .= ' (' . $reason['details'] . ')';
				}
				$reason_texts[] = '• ' . $reason_text;
			}
			$details[] = sprintf( '<br /><strong>Reasons:</strong>%s', implode( '<br>', array_map( 'esc_html', $reason_texts ) ) );
		}

		if ( get_class( $result ) === $this->get_result_class() ) {
			// Additional Details (like Gmail Plus info)
			$additional_details = $this->get_additional_details( $result );
			if ( ! empty( $additional_details ) ) {
				$details = array_merge( $details, $additional_details );
			}

			// Technical Details
			$tech_details = $this->get_technical_details( $result );
			if ( ! empty( $tech_details ) ) {
				$details[] = '<br><strong>Technical Details:</strong>'
					. implode( '<br>', array_map( 'esc_html', $tech_details ) );
			}
		}

		return implode( '<br>', $details );
	}

	/**
	 * Get validation badge HTML
	 *
	 * @param VR $result The validation result
	 * @param string $tooltip The tooltip text
	 * @param string $prefix Optional prefix before badge
	 * @return string
	 */
	protected function get_validation_badge( $result, string $tooltip, string $prefix = '&nbsp;' ): string {
		$status = $this->get_validation_badge_status( $result );

		switch ( $status ) {
			case 'error':
				$icon  = 'gform-icon--circle-notice';
				$color = '#e93934';
				break;
			case 'success':
				$icon  = 'gform-icon--verified';
				$color = '#22a753';
				break;
			case 'warning':
			default:
				$icon  = 'gform-icon--info';
				$color = '#ffbe03';
				break;
		}

		return sprintf(
			'%s<button onclick="return false;" onkeypress="return false;" class="gf_tooltip gpev_tooltip tooltip-%s" aria-label="%s" style="background:none;color:%s">
				<span class="gform-icon %s" aria-hidden="true"></span>
			</button>',
			$prefix,
			sanitize_html_class( strtolower( $icon ) ),
			esc_attr( $tooltip ),
			esc_attr( $color ),
			esc_attr( $icon )
		);
	}

	/**
	 * Display validation result in entry details
	 *
	 * @param string $value The field value
	 * @param \GF_Field $field The field
	 * @param array $entry The entry
	 * @param array $form The form
	 * @return string
	 */
	public function display_entry_value( $value, $field, $entry, $form ) {
		if ( ! gp_email_validator()->is_email_validator_field( $field ) ) {
			return $value;
		}

		$data = gform_get_meta( $entry['id'], $this->get_meta_key( $field->id ) );
		if ( ! $data ) {
			return $value;
		}

		$result_class = rgar( $data, 'status' ) !== 'error' ? $this->get_result_class() : Error_Validation_Result::class;
		$result       = $result_class::from_array( $data, $this );

		$tooltip = $this->format_tooltip_details( $result );
		return $value . $this->get_validation_badge( $result, $tooltip );
	}

	/**
	 * Display validation result in entry list
	 *
	 * @param string|array $value The field value
	 * @param int $form_id The form ID
	 * @param string|int $field_id The field ID
	 * @param array $entry The entry

	 * @return string
	 */
	public function display_list_value( $value, $form_id, $field_id, $entry ) {
		// Check if this is one of our meta fields
		$meta_key = "gpev_{$this->service_name}_status_";
		if ( ! is_string( $field_id ) || strpos( $field_id, $meta_key ) !== 0 ) {
			return $value;
		}

		// Get the actual field ID from the meta key
		$email_field_id = (int) str_replace( $meta_key, '', $field_id );
		if ( ! $email_field_id ) {
			return $value;
		}

		// We only register the status meta field, so get the actual result meta
		$data = gform_get_meta( $entry['id'], $this->get_meta_key( $email_field_id ) );

		if ( empty( $data['status'] ) ) {
			return $value;
		}

		$result_class = $this->get_result_class();
		$result       = $result_class::from_array( $data, $this );

		return $this->get_validation_badge( $result, $this->format_tooltip_details( $result ) );
	}

	/**
	 * Validate a field
	 *
	 * @param array $result The validation result
	 * @param string $value The field value
	 * @param array $form The form
	 * @param \GF_Field_Email $field The field
	 * @return array
	 */
	public function validate_field( $result, $value, $form, $field ) {
		if ( ! $result['is_valid'] || ! gp_email_validator()->is_email_validator_field( $field ) || rgblank( $value ) ) {
			return $result;
		}

		$validation_result = $this->perform_validation( $value, $field->id );

		/**
		 * Filter the validation result.
		 *
		 * @param Validation_Result|null $validation_result The validation result
		 * @param string $value The email address being validated
		 * @param \GF_Field_Email $field The email field
		 * @param array $form The form
		 * @param Validator $validator The validator instance
		 *
		 * @since 1.0
		 */
		$validation_result = gf_apply_filters( [ 'gpev_validation_result', $form['id'], $field->id ], $validation_result, $value, $field, $form, $this );

		if ( ! $validation_result ) {
			return $result;
		}

		// Store validation result for later use
		$this->validation_results[ $field->id ] = $validation_result;

		// For error results, allow submission but store the result
		if ( $validation_result instanceof Error_Validation_Result ) {
			return $result;
		}

		$result['is_valid'] = $validation_result->is_valid();

		if ( ! $result['is_valid'] ) {
			$result['message'] = $this->format_error_message( $validation_result->get_error_message() );

			gp_email_validator()->log(
				'Validation failed for email address: ' . json_encode( [
					'email'     => $value,
					'validator' => $this->service_name,
					'result'    => $validation_result->to_array(),
				], JSON_PRETTY_PRINT )
			);
		}

		return $result;
	}

	/**
	 * Check if the email domain is allowed or blocked based on plugin settings
	 *
	 * @param string $email The email address to check
	 * @return array|null Returns null if domain is allowed, or an error response if blocked
	 */
	protected function check_domain_validation( $email ) {
		$domain_mode = gp_email_validator()->get_plugin_setting( 'domain_validator_mode' );

		if ( $domain_mode === 'none' ) {
			return null;
		}

		$email_domain = $this->extract_domain( $email );

		if ( $domain_mode === 'allow' ) {
			$allowed_domains = preg_split( '/\r\n|\r|\n/',
				gp_email_validator()->get_plugin_setting( 'allowed_domains' )
			);
			$allowed_domains = array_filter( array_map( 'trim', $allowed_domains ) );

			if ( empty( $allowed_domains ) || ! in_array( $email_domain, $allowed_domains ) ) {
				return [
					'result'  => 'failed_domain_validation',
					'reason'  => 'domain_not_allowed',
					'message' => sprintf(
						// translators: %s is a comma-separated list of allowed domains
						__( 'Only emails from the following domains are allowed: %s', 'gp-email-validator' ),
						implode( ', ', $allowed_domains )
					),
				];
			}
		}

		if ( $domain_mode === 'block' ) {
			$blocked_domains = preg_split( '/\r\n|\r|\n/',
				gp_email_validator()->get_plugin_setting( 'blocked_domains' )
			);
			$blocked_domains = array_filter( array_map( 'trim', $blocked_domains ) );

			if ( in_array( $email_domain, $blocked_domains ) ) {
				return [
					'result'  => 'failed_domain_validation',
					'reason'  => 'domain_blocked',
					'message' => sprintf(
						// translators: %s is a comma-separated list of blocked domains
						__( 'Emails from the following domains are not allowed: %s', 'gp-email-validator' ),
						implode( ', ', $blocked_domains )
					),
				];
			}
		}

		return null;
	}

	/**
	 * Extract domain from an email address
	 *
	 * @param string $email The email address
	 * @return string The domain part of the email
	 */
	protected function extract_domain( $email ) {
		$parts = explode( '@', $email );
		return isset( $parts[1] ) ? strtolower( trim( $parts[1] ) ) : '';
	}

	/**
	 * AJAX callback to get email suggestion
	 */
	public function ajax_get_suggestion() {
		// Check nonce
		check_ajax_referer( 'gpev_get_suggestion', 'nonce' );

		$email = rgpost( 'email' );

		// Validate email
		$validation_result = $this->perform_validation( $email );

		// Get suggestion
		$suggestion = $validation_result->get_suggestion();

		wp_send_json_success( [
			'checked_email' => $email,
			'suggestion'    => $suggestion,
		] );
	}
}
