<?php

namespace GP_Email_Validator\Validators;

use EmailValidator\EmailValidator;
use EmailValidator\EmailAddress;
use GP_Email_Validator\Results\Validation_Result;
use GP_Email_Validator\Results\Basic_Validation_Result;

/**
 * Basic email validator that checks for valid email syntax, MX records,
 * and optionally blocks disposable/free email providers.
 *
 * @extends Validator<Basic_Validation_Result>
 */
class Basic_Validator extends Validator {
	protected $service_name = 'basic';

	/**
	 * @var EmailValidator|null
	 */
	protected $email_validator;

	/**
	 * Initialize the validator by registering hooks
	 *
	 * The sanitize_gmail hook allows us to modify Gmail addresses that use the plus trick
	 * (e.g., user+tag@gmail.com) when saving the field value, if that option is enabled.
	 */
	public function init() {
		parent::init();
		add_filter( 'gform_save_field_value', [$this, 'sanitize_gmail'], 10, 3 );
		add_filter( 'gform_value_pre_duplicate_check', [$this, 'sanitize_gmail_duplicate_value_check'], 10, 3 );
	}

	public function get_allowed_statuses() {
		return [
			'valid',
			'invalid',
		];
	}

	/**
	 * Get the email validator instance
	 *
	 * @param string $value The email being validated
	 * @return EmailValidator
	 */
	public function get_validator( $value ) {
		if ( $this->email_validator === null ) {
			$this->email_validator = new EmailValidator( $this->get_config( $value ) );
		}
		return $this->email_validator;
	}

	public function validate( $value, $field_id = null ) {
		if ( rgblank( $value ) ) {
			return null;
		}

		$response = $this->get_api_response( $value );
		return $this->create_validation_result( $value, $response );
	}

	/**
	 * Get the API response for an email address
	 *
	 * @param string $value The email address to validate
	 * @return array
	 */
	protected function get_api_response( $value ) {
		$validator = $this->get_validator( $value );
		$validator->validate( $value );

		$error_code = $validator->getErrorCode();

		/*
		 * Instead of using the output of $validator->isValid(), we know we have a "valid" email if there is either no
		 * error, it's a free email provider, or it's a disposable email provider.
		 *
		 * If we didn't do this, we'd have to set up a new validator instance to check for basic validity and then
		 * what we're actually GF validating against which will be done in the Basic_Validation_Result class.
		 */
		$is_valid = $error_code === EmailValidator::NO_ERROR ||
			$error_code === EmailValidator::FAIL_FREE_PROVIDER ||
			$error_code === EmailValidator::FAIL_DISPOSABLE_DOMAIN;

		$response = [
			'is_valid'          => $is_valid,
			'status'            => $is_valid ? 'valid' : 'invalid',
			'has_mx_records'    => $error_code !== EmailValidator::FAIL_MX_RECORD,
			'is_disposable'     => $error_code === EmailValidator::FAIL_DISPOSABLE_DOMAIN,
			'is_free_email'     => $error_code === EmailValidator::FAIL_FREE_PROVIDER,
			'is_invalid_format' => $error_code === EmailValidator::FAIL_BASIC,
			'is_gmail_plus'     => $validator->isGmailWithPlusChar(),
			'reason'            => $is_valid ? null : $validator->getErrorReason(),
		];

		// Handle Gmail plus addressing
		if ( $response['is_gmail_plus'] && $this->should_sanitize_gmail() ) {
			$response['is_valid']        = true;
			$response['sanitized_email'] = $validator->getGmailAddressWithoutPlus();
		}

		return $response;
	}

	/**
	 * Get error messages for each type of validation failure
	 *
	 * @return array<string, string>
	 */
	public function get_error_messages() {
		return [
			'invalid_format'   => __( 'Email address format is invalid.', 'gp-email-validator' ),
			'no_mx_record'     => __( 'Email domain does not have a valid MX record.', 'gp-email-validator' ),
			'disposable_email' => __( 'Disposable email addresses are not allowed.', 'gp-email-validator' ),
			'free_email'       => __( 'Free email accounts are not allowed.', 'gp-email-validator' ),
		];
	}

	protected function get_technical_details( $result ): array {
		$result_metadata = $result->get_metadata();

		$details = [
			sprintf( '• MX Records: %s', $result_metadata['has_mx_records'] ? 'Yes' : 'No' ),
			sprintf( '• Disposable Email: %s', $result_metadata['is_disposable'] ? 'Yes' : 'No' ),
			sprintf( '• Free Email: %s', $result_metadata['is_free_email'] ? 'Yes' : 'No' ),
			sprintf( '• Invalid Format: %s', $result_metadata['is_invalid_format'] ? 'Yes' : 'No' ),
		];

		return $details;
	}

	protected function get_additional_details( $result ): array {
		$details         = [];
		$result_metadata = $result->get_metadata();

		if ( ! empty( $result_metadata['is_gmail_plus'] ) ) {
			$details[] = '<br /><strong>Gmail Plus:</strong>'
				. sprintf( '• Original: %s', esc_html( $result->get_email() ) );
			$details[] = sprintf( '• Sanitized: %s', esc_html( $result->get_sanitized_email() ) );
		}

		return $details;
	}

	/**
	 * Define settings for the Basic validator
	 *
	 * These settings appear in the plugin settings page and allow admins to:
	 * - Choose between default and custom validation rules
	 * - Configure which checks to perform (MX, disposable, free email)
	 * - Enable Gmail plus address sanitization
	 *
	 * @return array
	 */
	public function get_settings() {
		return [
			[
				'name'          => 'validator_rules',
				'tooltip'       => __( 'Validator Rules', 'gp-email-validator' ),
				'label'         => __( 'Validator Rules', 'gp-email-validator' ),
				'type'          => 'radio',
				'default_value' => 'default',
				'horizontal'    => true,
				'choices'       => [
					[
						'value' => 'default',
						'label' => __( 'Use default rules', 'gp-email-validator' ),
					],
					[
						'value' => 'custom',
						'label' => __( 'Use custom rules', 'gp-email-validator' ),
					],
				],
				'fields'        => [
					[
						'name'       => 'rules',
						'type'       => 'checkbox',
						'choices'    => [
							[
								'name'          => 'mx_record',
								'label'         => __( 'Require configured MX record', 'gp-email-validator' ),
								'tooltip'       => __( 'Checks if the email domain has valid MX records configured.', 'gp-email-validator' ),
								'default_value' => 1,
							],
							[
								'name'          => 'disposable_email',
								'label'         => __( 'Block disposable emails', 'gp-email-validator' ),
								'tooltip'       => __( 'Prevents temporary or disposable email addresses from being used.', 'gp-email-validator' ),
								'default_value' => 1,
							],
							[
								'name'    => 'free_email',
								'label'   => __( 'Block free email accounts', 'gp-email-validator' ),
								'tooltip' => __( 'Prevents free email services (like Gmail, Yahoo, etc.) from being used.', 'gp-email-validator' ),
							],
							[
								'name'    => 'sanitize_gmail',
								'label'   => __( 'Sanitize Gmail email addresses that use the "plus trick".', 'gp-email-validator' ),
								'tooltip' => __( 'Removes the plus part from Gmail addresses (e.g., user+tag@gmail.com → user@gmail.com).', 'gp-email-validator' ),
							],
						],
						'dependency' => [
							'live'   => true,
							'fields' => [
								[
									'field'  => 'validator_rules',
									'values' => ['custom'],
								],
							],
						],
					],
				],
			],
		];
	}

	/**
	 * Get validation configuration based on settings and domain rules
	 *
	 * The configuration determines which checks to perform:
	 * - MX record validation
	 * - Disposable email blocking
	 * - Free email provider blocking
	 *
	 * Allowed domains bypass all checks.
	 *
	 * @param string $value The email address being validated
	 * @return array
	 */
	protected function get_config( $value ) {
		$config = [
			'checkMxRecords'       => true,
			'checkDisposableEmail' => true,
			'checkFreeEmail'       => true,
		];

		return $config;
	}

	public function get_rejection_settings(): array {
		$validator_rules = gp_email_validator()->get_plugin_setting( 'validator_rules' );

		if ( $validator_rules === 'custom' ) {
			return [
				'mx_record'        => (bool) gp_email_validator()->get_plugin_setting( 'mx_record' ),
				'disposable_email' => (bool) gp_email_validator()->get_plugin_setting( 'disposable_email' ),
				'free_email'       => (bool) gp_email_validator()->get_plugin_setting( 'free_email' ),
			];
		}

		// Default rules
		return [
			'mx_record'        => true,
			'disposable_email' => true,
			'free_email'       => false,
		];
	}

	/**
	 * Check if Gmail sanitization is enabled in settings
	 *
	 * When enabled, Gmail addresses with plus signs (e.g., user+tag@gmail.com)
	 * will be accepted and sanitized to their base form (e.g., user@gmail.com).
	 *
	 * @return bool
	 */
	public function should_sanitize_gmail() {
		return gp_email_validator()->get_plugin_setting( 'validator_rules' ) == 'custom'
			&& (bool) gp_email_validator()->get_plugin_setting( 'sanitize_gmail' );
	}

	/**
	 * Sanitize Gmail addresses by removing the plus part
	 *
	 * This hook runs when saving the field value and removes the "plus" part
	 * of Gmail addresses if that feature is enabled. For example:
	 * - user+anything@gmail.com becomes user@gmail.com
	 *
	 * @param string $value The email address to sanitize
	 * @param array $entry The form entry
	 * @param \GF_Field_Email $field The email field
	 * @return string
	 */
	public function sanitize_gmail( $value, $entry, $field ) {
		if ( ! gp_email_validator()->is_email_validator_field( $field ) || rgblank( $value ) ) {
			return $value;
		}

		$email = new EmailAddress( $value );

		if ( $this->should_sanitize_gmail() && $email->isGmailWithPlusChar() ) {
			return $email->getGmailAddressWithoutPlus();
		}

		return $value;
	}

	/**
	 * Sanitizes Gmail addresses when checking for duplicate values. Uses Basic_Validator::sanitize_gmail(), but
	 * this has a different signature as it's on a different hook.
	 *
	 * @param string $value The email address to sanitize
	 * @param \GF_Field_Email $field The field being checked
	 * @param int $form_id The form ID
	 */
	public function sanitize_gmail_duplicate_value_check( $value, $field, $form_id ) {
		if ( ! gp_email_validator()->is_email_validator_field( $field ) || rgblank( $value ) ) {
			return $value;
		}

		return $this->sanitize_gmail( $value, [], $field );
	}
}
