<?php

namespace GP_Email_Validator\Validators;
use GP_Email_Validator\Results\Validation_Result;
use GP_Email_Validator\Results\Kickbox_Validation_Result;

/**
 * Kickbox email validator that uses the Kickbox API for advanced validation.
 *
 * API Endpoint: https://api.kickbox.com/v2/verify
 * Documentation: https://docs.kickbox.com/docs/single-verification-api
 *
 * @extends Validator<Kickbox_Validation_Result>
 */
class Kickbox_Validator extends Validator {
	protected $service_name = 'kickbox';
	protected $api_endpoint = 'https://api.kickbox.com/v2/verify';

	protected function get_success_statuses(): array {
		return ['deliverable'];
	}

	public function validate( $value, $field_id = null ) {
		if ( rgblank( $value ) ) {
			return null;
		}

		$response = $this->get_api_response( $value );
		return $this->create_validation_result( $value, $response );
	}

	public function get_allowed_statuses() {
		return [
			'deliverable',
			'undeliverable',
			'risky',
			'unknown',
		];
	}

	public function get_error_messages() {
		return [
			'invalid_email'      => __( 'Email address has invalid syntax.', 'gp-email-validator' ),
			'invalid_domain'     => __( 'Domain does not exist.', 'gp-email-validator' ),
			'rejected_email'     => __( 'Email address was rejected by the mail server and does not exist.', 'gp-email-validator' ),
			'low_quality'        => __( 'Email address has quality issues that may make it risky or low-value.', 'gp-email-validator' ),
			'low_deliverability' => __( 'Email address appears deliverable, but delivery cannot be guaranteed.', 'gp-email-validator' ),
			'no_connect'         => __( 'Could not connect to mail server.', 'gp-email-validator' ),
			'timeout'            => __( 'Mail server connection timed out.', 'gp-email-validator' ),
			'invalid_smtp'       => __( 'Mail server returned an unexpected response.', 'gp-email-validator' ),
			'unavailable_smtp'   => __( 'Mail server was unavailable to process the request.', 'gp-email-validator' ),
			'unexpected_error'   => __( 'An unexpected error occurred during validation.', 'gp-email-validator' ),
		];
	}

	protected function get_technical_details( $result ): array {
		$result_metadata = $result->get_metadata();

		return [
			sprintf( '• Result: %s', ucfirst( $result_metadata['result'] ) ),
			sprintf( '• Reason: %s', ucfirst( $result_metadata['reason'] ) ),
			sprintf( '• Role Address: %s', $result_metadata['role'] ? 'Yes' : 'No' ),
			sprintf( '• Free Email: %s', $result_metadata['free'] ? 'Yes' : 'No' ),
			sprintf( '• Disposable: %s', $result_metadata['disposable'] ? 'Yes' : 'No' ),
			sprintf( '• Accept All: %s', $result_metadata['accept_all'] ? 'Yes' : 'No' ),
			sprintf( '• Sendex Score: %s', $result_metadata['sendex'] ),
		];
	}

	/**
	 * Define settings for the Kickbox validator
	 *
	 * @return array
	 */
	public function get_settings() {
		return [
			[
				'name'          => 'kickbox_mode',
				'tooltip'       => __( 'Choose between live and sandbox mode. Use sandbox for testing.', 'gp-email-validator' ),
				'label'         => __( 'Mode', 'gp-email-validator' ),
				'type'          => 'radio',
				'default_value' => 'test',
				'horizontal'    => true,
				'choices'       => [
					[
						'value' => 'live',
						'label' => __( 'Live', 'gp-email-validator' ),
					],
					[
						'value' => 'test',
						'label' => __( 'Sandbox', 'gp-email-validator' ),
					],
				],
			],
			[
				'name'    => 'kickbox_rejected_results',
				'label'   => __( 'Rejection Criteria', 'gp-email-validator' ),
				'tooltip' => __( 'Select which Kickbox results should cause the email to be rejected.', 'gp-email-validator' ),
				'type'    => 'checkbox',
				'choices' => [
					[
						'name'    => 'kickbox_result_risky',
						'label'   => __( 'Risky', 'gp-email-validator' ),
						'tooltip' => __( 'Addresses with quality issues like Accept-All domains, disposable emails, or role addresses.', 'gp-email-validator' ),
					],
					[
						'name'    => 'kickbox_result_unknown',
						'label'   => __( 'Unknown', 'gp-email-validator' ),
						'tooltip' => __( 'Addresses that could not be verified due to temporary server issues.', 'gp-email-validator' ),
					],
					[
						'name'          => 'kickbox_result_undeliverable',
						'label'         => __( 'Undeliverable', 'gp-email-validator' ),
						'tooltip'       => __( 'Addresses that do not exist or cannot receive mail.', 'gp-email-validator' ),
						'default_value' => 1,
					],
				],
			],
			[
				'name'    => 'kickbox_show_suggestions',
				'label'   => __( 'Email Suggestions', 'gp-email-validator' ),
				'type'    => 'checkbox',
				'choices' => [
					[
						'name'    => 'kickbox_show_suggestions',
						'label'   => __( 'Show "Did you mean?" suggestions for misspelled emails', 'gp-email-validator' ),
						'tooltip' => __( 'Suggests corrections for common typos (e.g., user@gamil.com → user@gmail.com).', 'gp-email-validator' ),
					],
				],
			],
			[
				'name'       => 'kickbox_live_api_key',
				'tooltip'    => __( 'API key for live mode from your Kickbox account.', 'gp-email-validator' ),
				'label'      => __( 'Live API Key', 'gp-email-validator' ),
				'type'       => 'text',
				'input_type' => 'password',
				'dependency' => [
					'live'   => true,
					'fields' => [
						[
							'field'  => 'kickbox_mode',
							'values' => ['live'],
						],
					],
				],
			],
			[
				'name'       => 'kickbox_test_api_key',
				'tooltip'    => __( 'API key for sandbox testing from your Kickbox account.', 'gp-email-validator' ),
				'label'      => __( 'Test API Key', 'gp-email-validator' ),
				'type'       => 'text',
				'input_type' => 'password',
				'dependency' => [
					'live'   => true,
					'fields' => [
						[
							'field'  => 'kickbox_mode',
							'values' => ['test'],
						],
					],
				],
			],
		];
	}

	/**
	 * Get the API response for an email address
	 *
	 * @param string $value The email address to validate
	 * @return array
	 */
	protected function get_api_response( $value ) {
		$mode    = gp_email_validator()->get_plugin_setting( 'kickbox_mode' );
		$api_key = gp_email_validator()->get_plugin_setting( "kickbox_{$mode}_api_key" );

		if ( empty( $api_key ) ) {
			return [
				'error' => __( 'API key not configured.', 'gp-email-validator' ),
			];
		}

		$url = add_query_arg([
			'email'   => urlencode( $value ),
			'apikey'  => $api_key,
			'timeout' => 6000,
		], $this->api_endpoint);

		$response = wp_remote_get( $url );

		if ( is_wp_error( $response ) ) {
			return [
				'error' => $response->get_error_message(),
			];
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( ! $data ) {
			return [
				'error' => __( 'Invalid response from Kickbox API', 'gp-email-validator' ),
			];
		}

		return $data;
	}

	public function get_rejection_settings(): array {
		$settings = [];

		// Get rejected results from settings
		$results = ['undeliverable', 'risky', 'unknown'];
		foreach ( $results as $result ) {
			$settings[ $result ] = (bool) gp_email_validator()->get_plugin_setting( "kickbox_result_{$result}" );
		}

		return $settings;
	}

	public function should_show_suggestions(): bool {
		return (bool) gp_email_validator()->get_plugin_setting( 'kickbox_show_suggestions' );
	}
}
