<?php

namespace GP_Email_Validator\Validators;
use GP_Email_Validator\Results\Validation_Result;
use GP_Email_Validator\Results\Zerobounce_Validation_Result;

/**
 * ZeroBounce email validator that uses the ZeroBounce API for advanced validation.
 *
 * API Endpoint: https://api.zerobounce.net/v2/validate
 *
 * @extends Validator<Zerobounce_Validation_Result>
 */
class Zerobounce_Validator extends Validator {
	protected $service_name = 'zerobounce';
	protected $api_endpoint = 'https://api.zerobounce.net/v2/validate';

	/**
	 * List of valid sandbox emails
	 *
	 * @var string[]
	 */
	private $sandbox_emails = [
		'disposable@example.com',
		'invalid@example.com',
		'valid@example.com',
		'toxic@example.com',
		'donotmail@example.com',
		'spamtrap@example.com',
		'abuse@example.com',
		'unknown@example.com',
		'catch_all@example.com',
		'antispam_system@example.com',
		'does_not_accept_mail@example.com',
		'exception_occurred@example.com',
		'failed_smtp_connection@example.com',
		'failed_syntax_check@example.com',
		'forcible_disconnect@example.com',
		'global_suppression@example.com',
		'greylisted@example.com',
		'leading_period_removed@example.com',
		'mail_server_did_not_respond@example.com',
		'mail_server_temporary_error@example.com',
		'mailbox_quota_exceeded@example.com',
		'mailbox_not_found@example.com',
		'no_dns_entries@example.com',
		'possible_trap@example.com',
		'possible_typo@example.com',
		'role_based@example.com',
		'timeout_exceeded@example.com',
		'unroutable_ip_address@example.com',
		'free_email@example.com',
		'role_based_catch_all@example.com',
	];

	protected function get_success_statuses(): array {
		return ['valid'];
	}

	public function get_allowed_statuses() {
		return [
			'valid',
			'invalid',
			'catch-all',
			'unknown',
			'spamtrap',
			'abuse',
			'do_not_mail',
		];
	}


	public function validate( $value, $field_id = null ) {
		if ( rgblank( $value ) ) {
			return null;
		}

		$response = $this->get_api_response( $value );
		return $this->create_validation_result( $value, $response );
	}

	/**
	 * Check if an email is a valid sandbox email
	 *
	 * @param string $email The email to check
	 * @return bool
	 */
	protected function is_valid_sandbox_email( $email ) {
		return in_array( strtolower( $email ), $this->sandbox_emails );
	}

	public function get_error_messages() {
		return [
			'invalid'     => __( 'Email address is invalid.', 'gp-email-validator' ),
			'catch-all'   => __( 'Domain appears to accept all email addresses.', 'gp-email-validator' ),
			'unknown'     => __( 'Email address could not be verified.', 'gp-email-validator' ),
			'spamtrap'    => __( 'Email address is a known spam trap.', 'gp-email-validator' ),
			'abuse'       => __( 'Email address is associated with abuse.', 'gp-email-validator' ),
			'do_not_mail' => __( 'Email address should not be mailed to.', 'gp-email-validator' ),
		];
	}

	/**
	 * Get sub-status messages for additional validation details
	 *
	 * @return array<string, string>
	 */
	public function get_sub_status_messages(): array {
		return [
			'alternate'                   => __( 'Alternate email found.', 'gp-email-validator' ),
			'antispam_system'             => __( 'Email blocked by anti-spam system.', 'gp-email-validator' ),
			'greylisted'                  => __( 'Email temporarily greylisted.', 'gp-email-validator' ),
			'mail_server_temporary_error' => __( 'Temporary mail server error.', 'gp-email-validator' ),
			'forcible_disconnect'         => __( 'Mail server forcibly disconnected.', 'gp-email-validator' ),
			'mail_server_did_not_respond' => __( 'Mail server did not respond.', 'gp-email-validator' ),
			'timeout_exceeded'            => __( 'Connection timeout exceeded.', 'gp-email-validator' ),
			'failed_smtp_connection'      => __( 'SMTP connection failed.', 'gp-email-validator' ),
			'mailbox_quota_exceeded'      => __( 'Mailbox quota exceeded.', 'gp-email-validator' ),
			'exception_occurred'          => __( 'Exception occurred during validation.', 'gp-email-validator' ),
			'possible_trap'               => __( 'Possible spam trap detected.', 'gp-email-validator' ),
			'role_based'                  => __( 'Role-based email address.', 'gp-email-validator' ),
			'global_suppression'          => __( 'Email globally suppressed.', 'gp-email-validator' ),
			'mailbox_not_found'           => __( 'Mailbox not found.', 'gp-email-validator' ),
			'no_dns_entries'              => __( 'No DNS entries found.', 'gp-email-validator' ),
			'failed_syntax_check'         => __( 'Email syntax check failed.', 'gp-email-validator' ),
			'possible_typo'               => __( 'Possible typo in email address.', 'gp-email-validator' ),
			'unroutable_ip_address'       => __( 'Unroutable IP address.', 'gp-email-validator' ),
			'leading_period_removed'      => __( 'Leading period removed.', 'gp-email-validator' ),
			'does_not_accept_mail'        => __( 'Domain does not accept mail.', 'gp-email-validator' ),
			'alias_address'               => __( 'Alias email address.', 'gp-email-validator' ),
			'role_based_catch_all'        => __( 'Role-based catch-all address.', 'gp-email-validator' ),
			'disposable'                  => __( 'Disposable email address.', 'gp-email-validator' ),
			'toxic'                       => __( 'Known toxic email pattern.', 'gp-email-validator' ),
		];
	}

	protected function get_technical_details( $result ): array {
		$result_metadata = $result->get_metadata();

		return [
			sprintf( '• Status: %s', ucfirst( $result_metadata['status'] ) ),
			sprintf( '• Sub Status: %s', ucfirst( str_replace( '_', ' ', $result_metadata['sub_status'] ) ) ),
			sprintf( '• Domain Age: %d days', $result_metadata['domain_age_days'] ),
			sprintf( '• Free Email: %s', $result_metadata['free_email'] ? 'Yes' : 'No' ),
			sprintf( '• MX Found: %s', $result_metadata['mx_found'] ? 'Yes' : 'No' ),
			sprintf( '• SMTP Provider: %s', $result_metadata['smtp_provider'] ),
			sprintf( '• Processed At: %s', $result_metadata['processed_at'] ),
		];
	}

	/**
	 * Define settings for the ZeroBounce validator
	 *
	 * @return array
	 */
	public function get_settings() {
		return [
			[
				'name'          => 'zerobounce_mode',
				'tooltip'       => __( 'Choose between live and sandbox mode. Use sandbox for testing which locks the list of emails that can be submitted to the defined sandbox emails provided by ZeroBounce.', 'gp-email-validator' ),
				'label'         => __( 'Mode', 'gp-email-validator' ),
				'type'          => 'radio',
				'default_value' => 'test',
				'horizontal'    => true,
				'choices'       => [
					[
						'value' => 'live',
						'label' => __( 'Live', 'gp-email-validator' ),
					],
					[
						'value' => 'test',
						'label' => __( 'Sandbox', 'gp-email-validator' ),
					],
				],
			],
			[
				'name'    => 'zerobounce_rejected_results',
				'label'   => __( 'Rejection Criteria', 'gp-email-validator' ),
				'tooltip' => __( 'Select which ZeroBounce results should cause the email to be rejected.', 'gp-email-validator' ),
				'type'    => 'checkbox',
				'choices' => [
					[
						'name'    => 'zerobounce_result_catch_all',
						'label'   => __( 'Catch-all', 'gp-email-validator' ),
						'tooltip' => __( 'Domains that accept all email addresses.', 'gp-email-validator' ),
					],
					[
						'name'    => 'zerobounce_result_unknown',
						'label'   => __( 'Unknown', 'gp-email-validator' ),
						'tooltip' => __( 'Addresses that could not be verified.', 'gp-email-validator' ),
					],
					[
						'name'          => 'zerobounce_result_invalid',
						'label'         => __( 'Invalid', 'gp-email-validator' ),
						'tooltip'       => __( 'Addresses that are invalid or do not exist.', 'gp-email-validator' ),
						'default_value' => 1,
					],
					[
						'name'          => 'zerobounce_result_do_not_mail',
						'label'         => __( 'Do Not Mail', 'gp-email-validator' ),
						'tooltip'       => __( 'Addresses that should not be mailed to.', 'gp-email-validator' ),
						'default_value' => 1,
					],
					[
						'name'          => 'zerobounce_result_spamtrap',
						'label'         => __( 'Spam Trap', 'gp-email-validator' ),
						'tooltip'       => __( 'Known spam trap addresses.', 'gp-email-validator' ),
						'default_value' => 1,
					],
					[
						'name'          => 'zerobounce_result_abuse',
						'label'         => __( 'Abuse', 'gp-email-validator' ),
						'tooltip'       => __( 'Addresses associated with abuse.', 'gp-email-validator' ),
						'default_value' => 1,
					],
				],
			],
			[
				'name'       => 'zerobounce_api_key',
				'tooltip'    => __( 'API key from your ZeroBounce account.', 'gp-email-validator' ),
				'label'      => __( 'API Key', 'gp-email-validator' ),
				'type'       => 'text',
				'input_type' => 'password',
			],
		];
	}

	/**
	 * Get the API response for an email address
	 *
	 * @param string $value The email address to validate
	 * @return array
	 */
	protected function get_api_response( $value ) {
		$mode    = gp_email_validator()->get_plugin_setting( 'zerobounce_mode' );
		$api_key = gp_email_validator()->get_plugin_setting( 'zerobounce_api_key' );

		// If in test mode, validate against sandbox emails
		if ( $mode === 'test' ) {
			if ( ! $this->is_valid_sandbox_email( $value ) ) {
				return [
					'status'     => 'invalid',
					'sub_status' => 'not_valid_sandbox_email',
					'reason'     => 'Email is not a valid sandbox email.',
				];
			}
		}

		if ( empty( $api_key ) ) {
			return [
				'error' => __( 'API key not configured.', 'gp-email-validator' ),
			];
		}

		$url = add_query_arg([
			'email'      => urlencode( $value ),
			'api_key'    => $api_key,
			'ip_address' => '', // Optional, but parameter required
		], $this->api_endpoint);

		$response = wp_remote_get( $url );

		if ( is_wp_error( $response ) ) {
			return [
				'error' => $response->get_error_message(),
			];
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( ! $data ) {
			return [
				'error' => __( 'Invalid response from ZeroBounce API', 'gp-email-validator' ),
			];
		}

		return $data;
	}

	public function get_rejection_settings(): array {
		$settings = [];

		// Get rejected results from settings
		$results = ['invalid', 'catch-all', 'unknown', 'spamtrap', 'abuse', 'do_not_mail'];
		foreach ( $results as $result ) {
			$settings[ $result ] = (bool) gp_email_validator()->get_plugin_setting( "zerobounce_result_{$result}" );
		}

		return $settings;
	}
}
