interface ValidationResponse {
	success: boolean;
	data: {
		suggestion: string;
	};
}

/**
 * Handles email validation and suggestion management for Gravity Forms email fields.
 * Provides email suggestions when users enter potentially misspelled email addresses.
 */
class EmailValidator {
	// Cache validation results to avoid unnecessary API calls
	private validationCache = new Map<string, ValidationResponse>();

	// Track which email addresses have had their suggestions accepted, so we don't show them again
	private acceptedSuggestions = new Set<string>();

	// Store localized strings (as well as the AJAX URL and nonce)
	private strings = window.gp_email_validator_strings;

	constructor() {
		// Add a single event listener at the document level for all email fields
		document.addEventListener('blur', this.handleBlur.bind(this), true);
	}

	private getSuggestionsContainer(field: HTMLElement): HTMLElement {
		// Look for existing suggestions container or create a new one
		let suggestions = field.querySelector(
			'.gpev-suggestions'
		) as HTMLElement;

		if (!suggestions) {
			suggestions = document.createElement('div');
			suggestions.className = 'gpev-suggestions';
			suggestions.appendChild(document.createElement('p'));
			field.appendChild(suggestions);
		}

		return suggestions;
	}

	private handleBlur(event: FocusEvent): void {
		const input = event.target as HTMLInputElement;
		if (!this.isEmailInput(input)) return;

		const field = input.closest('.gpev-field');
		if (!field) return;

		// For email confirmation fields, ensure both inputs match before showing suggestions
		if (!this.validateEmailConfirmation(field as HTMLElement)) return;

		if (!input.value) {
			this.hideSuggestions(field as HTMLElement);
			return;
		}

		this.checkEmailSuggestion(input);
	}

	private isEmailInput(element: Element): element is HTMLInputElement {
		return (
			element instanceof HTMLInputElement &&
			element.closest('.gpev-field') !== null
		);
	}

	private validateEmailConfirmation(field: HTMLElement): boolean {
		const inputs = Array.from(
			field.querySelectorAll('input[type="email"]')
		) as HTMLInputElement[];

		// Single email field doesn't need confirmation
		if (inputs.length <= 1) return true;

		// For confirmation fields, all inputs must match the last one
		const lastInput = inputs[inputs.length - 1];
		return !inputs.some(
			(input) => input !== lastInput && input.value !== lastInput.value
		);
	}

	public async checkEmailSuggestion(input: HTMLInputElement): Promise<void> {
		const field = input.closest('.gpev-field') as HTMLElement;
		const email = input.value;

		// Don't show suggestions for emails that were previously accepted
		if (this.acceptedSuggestions.has(email)) return;

		const suggestions = this.getSuggestionsContainer(field);
		// Skip if we're already showing suggestions for this email
		if (suggestions.dataset.currentEmail === email) return;

		// Use cached result if available
		if (this.validationCache.has(email)) {
			this.handleValidationResponse(
				this.validationCache.get(email)!,
				input,
				field,
				suggestions
			);
			return;
		}

		try {
			const response = await this.fetchSuggestion(email);

			// Only update if the input hasn't changed while fetching
			if (input.value === email) {
				this.handleValidationResponse(
					response,
					input,
					field,
					suggestions
				);
			}
		} catch (error) {
			// Silently handle any errors to avoid breaking form functionality
			this.hideSuggestions(field);
		}
	}

	private async fetchSuggestion(email: string): Promise<ValidationResponse> {
		const formData = new FormData();
		formData.append('action', 'gpev_get_suggestion');
		formData.append('nonce', this.strings.nonce);
		formData.append('email', email);

		const response = await fetch(this.strings.ajaxUrl, {
			method: 'POST',
			body: formData,
		});

		if (!response.ok) {
			throw new Error('GP Email Validator: Network response was not ok');
		}

		const data = await response.json();
		this.validationCache.set(email, data);
		return data;
	}

	private handleValidationResponse(
		response: ValidationResponse,
		input: HTMLInputElement,
		field: HTMLElement,
		suggestions: HTMLElement
	): void {
		if (!response.success || !response.data.suggestion) {
			this.hideSuggestions(field);
			return;
		}

		const paragraph = suggestions.querySelector('p');
		if (!paragraph) return;

		// Update suggestion content and display
		paragraph.innerHTML = '';
		paragraph.appendChild(
			this.createSuggestionContent(
				response.data.suggestion,
				input,
				field,
				suggestions
			)
		);

		suggestions.style.display = 'block';
		suggestions.dataset.currentEmail = input.value;
	}

	private createSuggestionContent(
		suggestion: string,
		input: HTMLInputElement,
		field: HTMLElement,
		suggestions: HTMLElement
	): DocumentFragment {
		// Create "Did you mean [suggestion]?" text with clickable suggestion
		const fragment = document.createDocumentFragment();

		const text = document.createElement('span');
		text.textContent = this.strings.didYouMean || 'Did you mean ';
		fragment.appendChild(text);

		const link = document.createElement('a');
		link.href = '#';
		link.textContent = suggestion;
		link.addEventListener('click', (e) => {
			e.preventDefault();
			this.acceptSuggestion(suggestion, input.value, field, suggestions);
		});
		fragment.appendChild(link);

		fragment.appendChild(document.createTextNode('?'));

		return fragment;
	}

	private acceptSuggestion(
		suggestion: string,
		originalEmail: string,
		field: HTMLElement,
		suggestions: HTMLElement
	): void {
		// Update all email inputs in the field (for confirmation fields)
		const inputs = field.querySelectorAll(
			'input[type="email"]'
		) as NodeListOf<HTMLInputElement>;

		inputs.forEach((input) => (input.value = suggestion));

		this.hideSuggestions(field);
		this.acceptedSuggestions.add(suggestion);
		// Also add the original email to accepted suggestions
		this.acceptedSuggestions.add(originalEmail);
	}

	private hideSuggestions(field: HTMLElement): void {
		const suggestions = field.querySelector(
			'.gpev-suggestions'
		) as HTMLElement;
		if (suggestions) {
			suggestions.style.display = 'none';
			delete suggestions.dataset.currentEmail;
		}
	}
}

// Initialize the validator when the DOM is ready
document.addEventListener('DOMContentLoaded', () => {
	const validator = new EmailValidator();
	window.gpevGetSuggestions = (input: HTMLInputElement) => {
		validator.checkEmailSuggestion(input);
	};
});
