/**
 * GravityBoard Admin Conditional Logic
 *
 * Adds Assignee field to Gravity Forms notification conditional logic UI.
 */
// Ensure the global object and the fields array exist.
window.gform = window.gform || {};
window.gform.conditionalLogicFields = window.gform.conditionalLogicFields || {};
gravityBoardConditionalLogic = gravityBoardConditionalLogic || {};

/**
 * Add the custom Assignee field to the conditional logic fields dropdown.
 * Use the unique fieldId and fieldLabel for each feed.
 */
gform.addFilter(
	'gform_conditional_logic_fields',
	function (options, form, selectedFieldId) {
		if (
			!gravityBoardConditionalLogic.feeds ||
			!gravityBoardConditionalLogic.fields
		) {
			return options;
		}

		const feedField = gravityBoardConditionalLogic.fields.feed;
		const assigneeField = gravityBoardConditionalLogic.fields.assignee;
		const createdByField = gravityBoardConditionalLogic.fields.createdBy;

		if (
			feedField &&
			gravityBoardConditionalLogic.feeds &&
			gravityBoardConditionalLogic.feeds.length > 0
		) {
			options.push({
				label: feedField.label,
				value: feedField.inputType,
				operators: [], // Defined in gform_conditional_logic_operators filter.
			});
		}

		if (
			assigneeField &&
			gravityBoardConditionalLogic.users &&
			gravityBoardConditionalLogic.users.length > 0
		) {
			options.push({
				label: assigneeField.label,
				value: assigneeField.inputType,
				operators: [], // Defined in gform_conditional_logic_operators filter.
			});
		}

		if (
			createdByField &&
			'toplevel_page_gf_edit_forms' === window.adminpage
		) {
			options.push({
				label: createdByField.label,
				value: createdByField.inputType,
				operators: [], // Defined in gform_conditional_logic_operators filter.
			});
		}

		return options;
	}
);

/**
 * Change the input type to 'select' for the Assignee field.
 * Checks if the selectedFieldId matches the current feed's fieldId.
 */
gform.addFilter(
	'gform_conditional_logic_values_input',
	function (
		str,
		currentObjectType,
		ruleIndex,
		selectedFieldId,
		selectedValue
	) {
		if (
			!gravityBoardConditionalLogic.feeds ||
			!gravityBoardConditionalLogic.fields
		) {
			return str;
		}

		const feedInputType = gravityBoardConditionalLogic.fields.feed
			? gravityBoardConditionalLogic.fields.feed.inputType
			: undefined;
		const assigneeInputType = gravityBoardConditionalLogic.fields.assignee
			? gravityBoardConditionalLogic.fields.assignee.inputType
			: undefined;
		const createdByInputType = gravityBoardConditionalLogic.fields.createdBy
			? gravityBoardConditionalLogic.fields.createdBy.inputType
			: undefined;

		switch (selectedFieldId) {
			case feedInputType:
				if (!feedInputType) break; // Should not match if feedInputType is undefined

				// Format feeds into the expected dropdown format.
				// gravityBoardConditionalLogic.feeds is checked at the top of the function
				const feedOptions = gravityBoardConditionalLogic.feeds.map(
					(feed) => {
						return {
							value: feed.id,
							text: feed.name,
						};
					}
				);
				return GetRuleValuesDropDown(
					feedOptions,
					currentObjectType,
					ruleIndex,
					selectedValue,
					false
				);

			case assigneeInputType:
			case createdByInputType:
				// Ensure the selectedFieldId actually matches a defined inputType
				if (selectedFieldId === assigneeInputType && !assigneeInputType)
					break;
				if (
					selectedFieldId === createdByInputType &&
					!createdByInputType
				)
					break;

				const userOptions = [];
				if (Array.isArray(gravityBoardConditionalLogic.users)) {
					gravityBoardConditionalLogic.users.forEach((user) => {
						if (
							!userOptions.some(
								(option) => option.value === user.value
							)
						) {
							userOptions.push(user);
						}
					});
				}
				return GetRuleValuesDropDown(
					userOptions,
					currentObjectType,
					ruleIndex,
					selectedValue,
					false
				);
		}

		return str;
	}
);

/**
 * Add operators for the Assignee field.
 */
gform.addFilter(
	'gform_conditional_logic_operators',
	function (operators, objectType, fieldIdValue) {
		if (
			!gravityBoardConditionalLogic.feeds ||
			!gravityBoardConditionalLogic.fields ||
			!gravityBoardConditionalLogic.strings
		) {
			return operators;
		}

		const feedInputType = gravityBoardConditionalLogic.fields.feed
			? gravityBoardConditionalLogic.fields.feed.inputType
			: undefined;
		const assigneeInputType = gravityBoardConditionalLogic.fields.assignee
			? gravityBoardConditionalLogic.fields.assignee.inputType
			: undefined;
		const createdByInputType = gravityBoardConditionalLogic.fields.createdBy
			? gravityBoardConditionalLogic.fields.createdBy.inputType
			: undefined;

		// Note: Gravity Forms JS filters often pass the field's 'value' (our fieldId) as the third argument.
		if (
			(feedInputType && fieldIdValue === feedInputType) ||
			(assigneeInputType && fieldIdValue === assigneeInputType) ||
			(createdByInputType && fieldIdValue === createdByInputType)
		) {
			return {
				is: gravityBoardConditionalLogic.strings.is,
				isnot: gravityBoardConditionalLogic.strings.isnot,
			};
		}
		return operators;
	}
);
