/* global wp, jQuery, ClipboardJS */
(function ($) {
	$(function () {
		let mediaFrame;

		const $imagePreview = $('#gravityboard-background-image-preview');
		const $imageIdInput = $('#board_background_image_id');
		const $uploadButton = $('#gravityboard-upload-background-image-button');
		const $removeButton = $('#gravityboard-remove-background-image-button');

		if (!$uploadButton.length) {
			console.error(
				'GravityBoard: Upload button (#gravityboard-upload-background-image-button) not found!'
			);
		}
		if (!$imageIdInput.length) {
			console.error(
				'GravityBoard: Image ID input (#gravityboard-background-image-id) not found!'
			);
		}
		if (!$imagePreview.length) {
			console.error(
				'GravityBoard: Image preview div (#gravityboard-background-image-preview) not found!'
			);
		}

		// Unsaved Changes Detection for New Feeds
		function initializeUnsavedChangesDetection() {
			let hasUnsavedChanges = false;
			let isSubmitting = false;

			// Track changes to form fields
			const $form = $('#gform-settings');
			if ($form.length) {
				// Monitor input changes
				$form.on(
					'change keyup input',
					'input, select, textarea',
					function () {
						// Ignore certain fields that shouldn't trigger unsaved changes
						const $field = $(this);

						// Skip if field has onChange or onClick attributes (likely system fields)
						if ($field.attr('onChange') || $field.attr('onClick')) {
							return;
						}

						// Skip hidden fields and system fields
						if (
							$field.attr('type') === 'hidden' ||
							$field.hasClass('gform-settings-ignore-unsaved')
						) {
							return;
						}

						// Skip the feed ID field
						if ($field.attr('name') === 'gf_feed_id') {
							return;
						}

						hasUnsavedChanges = true;
					}
				);

				// Monitor WordPress color picker changes
				$form.on(
					'wpcolorpicker:change',
					'.gf-color-picker',
					function () {
						hasUnsavedChanges = true;
					}
				);

				// Also listen for iris color picker changes (fallback for older WP versions)
				$form.on('irischange', '.gf-color-picker', function () {
					hasUnsavedChanges = true;
				});

				// Clear unsaved changes flag when form is submitted
				$form.on('submit', function () {
					isSubmitting = true;
					hasUnsavedChanges = false;
				});

				// Also clear when save button is clicked (in case form submission is handled differently)
				$form.on(
					'click',
					'input[type="submit"], button[type="submit"], .gform-button--primary',
					function () {
						isSubmitting = true;
						hasUnsavedChanges = false;
					}
				);
			}

			// Set up beforeunload handler
			$(window).on(
				'beforeunload.gravityboard-feed-settings',
				function (event) {
					if (hasUnsavedChanges && !isSubmitting) {
						const message =
							window.gravityboard_admin_feed_settings_strings
								?.unsaved_changes ||
							'You have unsaved changes. Are you sure you want to leave this page?';
						event.originalEvent.returnValue = message; // For older browsers
						return message;
					}
				}
			);

			// Clean up on page unload
			$(window).on('unload.gravityboard-feed-settings', function () {
				$(window).off(
					'beforeunload.gravityboard-feed-settings unload.gravityboard-feed-settings'
				);
			});
		}

		// Initialize unsaved changes detection
		initializeUnsavedChangesDetection();

		// CSS Dimension Field Validation
		function validateCssDimension(value) {
			// Get validation pattern from PHP configuration
			const fieldConfig =
				gravityboard_admin_feed_settings_strings?.field_validation
					?.board_min_height;
			const pattern =
				fieldConfig?.pattern ||
				'^(\\d+(?:\\.\\d+)?)(px|%|vh|vw|em|rem)$';
			const regex = new RegExp(pattern, 'i');
			return regex.test(value);
		}

		function showValidationError($field, message) {
			$field.addClass('gfield_error');
			$field.closest('.gfield_setting').addClass('gfield_error');

			// Remove existing error message
			$field.siblings('.gform-settings-validation__error').remove();

			// Add error message
			$field.after(
				'<div class="gform-settings-validation__error">' +
					message +
					'</div>'
			);
		}

		function clearValidationError($field) {
			$field
				.removeClass('gfield_error')
				.closest('.gfield_setting')
				.removeClass('gfield_error')
				.end()
				.closest('.gform-settings-input__container--invalid')
				.removeClass('gform-settings-input__container--invalid')
				.end()
				.siblings('.gform-settings-validation__error')
				.remove();
		}

		// Real-time validation for CSS dimension fields
		$(document).on('input blur', '.css-dimension-field', function () {
			const $field = $(this);
			const value = $field.val();
			const isValid = value.length === 0 || validateCssDimension(value);

			if (!isValid) {
				const message =
					$field.data('validation-message') ||
					'Please enter a valid CSS dimension (e.g., 400px, 50vh, 100%)';
				showValidationError($field, message);
			} else {
				clearValidationError($field);
			}
		});

		// CSS Dimension field validation
		const $minHeightField = $('input[name="board_min_height"]');
		if ($minHeightField.length) {
			// Add event listeners
			$minHeightField.on('input blur', function () {
				const $field = $(this);
				const value = $field.val().trim();

				// Clear previous validation state
				clearValidationError($field);

				if (value && !validateCssDimension(value)) {
					// Get validation message from PHP configuration
					const fieldConfig =
						gravityboard_admin_feed_settings_strings
							?.field_validation?.board_min_height;
					const message =
						fieldConfig?.message ||
						'Please enter a valid CSS dimension (e.g., 400px, 50vh, 100%)';
					showValidationError($field, message);
				}
			});
		}

		// Function to update preview and button visibility
		const updatePreview = (id, url) => {
			$imageIdInput.val(id);
			if (url) {
				$imagePreview
					.css({
						'background-image': 'url(' + url + ')',
						'background-size': 'cover',
						'background-position': 'center',
						'min-height': '200px',
						'min-width': '266px',
						border: '1px solid #ccc',
					})
					.show();
				$removeButton.show();
			} else {
				$imagePreview
					.css({
						'background-image': '',
						'min-height': '',
						border: '',
					})
					.hide(); // Hide preview area if no image
				$removeButton.hide();
			}
		};

		// Media Uploader
		$uploadButton.on('click', function (e) {
			e.preventDefault();

			// If the frame already exists, reopen it
			if (mediaFrame) {
				mediaFrame.open();
				return;
			}

			// Create a new media frame
			mediaFrame = wp.media({
				library: {
					type: 'image',
				},
				multiple: false, // Only allow single image selection
			});

			// When an image is selected in the media frame
			mediaFrame.on('select', function () {
				const attachment = mediaFrame
					.state()
					.get('selection')
					.first()
					.toJSON();

				// Get the medium size URL for preview, or fall back to full
				const previewUrl =
					attachment.sizes?.medium?.url || attachment.url;

				updatePreview(attachment.id, previewUrl);

				// Announce the change for screen readers using localized string
				if (
					window.gravityboard_admin_feed_settings_strings
						?.image_selected
				) {
					wp.a11y.speak(
						window.gravityboard_admin_feed_settings_strings
							.image_selected
					);
				}
			});

			// Finally, open the media frame
			mediaFrame.open();
		});

		// Remove Image
		$removeButton.on('click', function (e) {
			e.preventDefault();
			updatePreview('', '');

			// Announce the change for screen readers using localized string
			if (
				window.gravityboard_admin_feed_settings_strings?.image_removed
			) {
				wp.a11y.speak(
					window.gravityboard_admin_feed_settings_strings
						.image_removed
				);
			}
		});

		// Initial check on page load - the preview div's style is set by PHP now.
		// We just need to ensure the remove button visibility is correct.
		if (!$imageIdInput.val()) {
			$removeButton.hide();
		} else {
			$removeButton.show();
		}

		if (
			typeof ClipboardJS !== 'undefined' &&
			$('.gk-gravityboard-shortcode input.shortcode').length
		) {
			new ClipboardJS('.gk-gravityboard-shortcode input.shortcode', {
				text: function (trigger) {
					return $(trigger).val();
				},
			});

			$('.gk-gravityboard-shortcode input.shortcode').on(
				'click',
				function (e) {
					e.preventDefault();
					var $el = $(this)
						.closest('.gk-gravityboard-shortcode')
						.find('.copied');
					$el.show();
					// Immediate accessibility announcement
					if (window.wp && wp.a11y) {
						wp.a11y.speak('Shortcode copied to clipboard');
					}
					// Quick visual feedback without accessibility delay
					setTimeout(function () {
						$el.fadeOut();
					}, 150);
				}
			);
		}

		// Color Picker Initialization for Lane Background Colors
		function initializeColorPickers() {
			$('.gf-color-picker').each(function () {
				// Initialize only if not already initialized to prevent conflicts
				if (!$(this).data('wpWpColorPicker')) {
					$(this).wpColorPicker();
				}
			});
		}

		// Initial call to initialize any color pickers already on the page
		initializeColorPickers();

		// Re-initialize if new color pickers are added to the DOM or become visible
		// due to Gravity Forms conditional logic.
		$(document).on(
			'gform_post_conditional_logic_field_action',
			function (
				event,
				formId,
				action,
				targetId,
				defaultFieldValues,
				isInit
			) {
				// Only re-initialize if a section is shown to avoid unnecessary processing
				if (action === 'show') {
					// Use minimal timeout for DOM readiness without accessibility impact
					setTimeout(function () {
						// Selectors for color pickers that might be inside the conditionally shown section
						// This targets any .gf-color-picker, which is the class we added.
						if (
							$(targetId).find('.gf-color-picker').length ||
							$(targetId).hasClass('gf-color-picker')
						) {
							initializeColorPickers();
						}
						// Also, our specific lane color section might be the target itself.
						// The section fields are generated by get_lane_background_color_fields()
						// and the section has a dependency. GF wraps sections in divs like #gform_setting_row_SECTION_NAME
						// The fields themselves are directly .gf-color-picker
						// We check if the target of the conditional logic contains our pickers
						if (
							$('#gform_fields')
								.find(targetId)
								.find('.gf-color-picker').length > 0
						) {
							initializeColorPickers();
						}
					}, 0);
				}
			}
		);

		// Specifically for our lane color pickers, they depend on the 'Lane Field' selection.
		// The 'Lane Field' setting has the name 'card_fields_lane_field'.
		// Its input element in the DOM will likely have an ID like 'gform_setting_card_fields_lane_field'
		// or can be targeted by its name attribute within the settings form.
		const laneFieldSelector =
			'select[name="meta[card_fields_lane_field]"], select#gform_setting_card_fields_lane_field';

		$(document).on('change', laneFieldSelector, function () {
			// Minimal delay for conditional field rendering
			setTimeout(initializeColorPickers, 0);
		});

		// Also initialize on load if the lane field is already populated and its section is visible.
		// This ensures pickers are active if the page loads with a lane field already selected.
		// The visibility of the section is handled by GF conditional logic + our dependency in PHP.
		if ($(laneFieldSelector).val() && $(laneFieldSelector).val() !== '') {
			setTimeout(initializeColorPickers, 0); // Immediate initialization for accessibility
		}

		// Tab URL Hash Functionality
		function initializeTabHashNavigation() {
			const $tabNavigation = $('.gform-settings-tabs__navigation');
			const $tabs = $tabNavigation.find('a[data-tab]');
			const $tabInput = $('input[name="gform_settings_tab"]');

			if (!$tabs.length) {
				return; // No tabs found, exit early
			}

			// Function to activate a tab by updating the hidden input and triggering GF's display logic
			function activateTab(dataTab) {
				const $targetTab = $tabs.filter('[data-tab="' + dataTab + '"]');
				if ($targetTab.length && $tabInput.length) {
					// Set the hidden input value that GF uses
					$tabInput.val(dataTab);

					// Remove active class from all tabs and containers
					$(
						'.gform-settings-tabs__navigation .active, .gform-settings-tabs__container.active'
					).removeClass('active');

					// Add active class to target tab and container
					$targetTab.addClass('active');
					$(
						'.gform-settings-tabs__container[data-tab="' +
							dataTab +
							'"]'
					).addClass('active');
				}
			}

			// Listen to GF's tab clicks to update URL hash
			$tabs.on('click', function () {
				const dataTab = $(this).data('tab');

				// Update URL hash without scrolling
				if (history.pushState) {
					history.pushState(null, null, '#' + dataTab);
				} else {
					// Fallback for older browsers
					location.hash = dataTab;
				}
			});

			// On page load, check for hash and activate corresponding tab
			function activateTabFromHash() {
				const hash = window.location.hash.replace('#', '');
				if (hash) {
					const $targetTab = $tabs.filter(
						'[data-tab="' + hash + '"]'
					);

					if ($targetTab.length) {
						// Immediate tab activation for accessibility
						activateTab(hash);
					}
				}
			}

			// Handle browser back/forward navigation
			$(window).on('hashchange', function () {
				activateTabFromHash();
			});

			// Activate tab on initial page load
			activateTabFromHash();
		}

		// Initialize tab hash navigation
		initializeTabHashNavigation();

		// Configure Select2 events for permission field indicators
		function initPermissionFieldPlaceholders() {
			const $permissionFields = $('select[id$="_role"]:not([id*="description"])');
			const messages = window.gravityboard_admin_feed_settings_strings || {};
			const defaultPlaceholder = messages.disabled_role_message || '';

			$permissionFields.each(function() {
				const $field = $(this);

				// Skip if already processed
				if ($field.data('gravityboard-disabled-role-initialized')) {
					return;
				}

				const fieldId = $field.attr('id');
				const placeholder = $field.attr('placeholder') || defaultPlaceholder;

				// Wait for Select2 to be initialized with retry limit
				const maxRetries = 50; // 50 attempts = ~50ms total wait
				let retryCount = 0;
				
				const checkSelect2 = () => {
					if ($field.data('select2')) {
						const $header = $field.parents('.gform-settings-field');
						const $container = $field.closest('.gform-settings-input__container');

						// Create visual indicator element
						const $indicator = $('<div>', {
							id: fieldId + '-empty-indicator',
							class: 'gravityboard-permission-indicator',
							html: '<span>' + placeholder + '</span>',
							'aria-hidden': 'true' // Hide from screen readers (we'll use aria-describedby instead)
						});

						// Create screen reader announcement element
						const $srAnnouncement = $('<div>', {
							id: fieldId + '-sr-status',
							class: 'gravityboard-sr-only',
							'aria-live': 'polite',
							'aria-atomic': 'true'
						});

						// Insert elements before the field container
						$container.before($indicator);
						$container.before($srAnnouncement);

						// Update aria-describedby to include our screen reader element
						const existingAriaDescribedBy = $field.attr('aria-describedby') || '';
						const newAriaDescribedBy = existingAriaDescribedBy
							? existingAriaDescribedBy + ' ' + fieldId + '-sr-status'
							: fieldId + '-sr-status';
						$field.attr('aria-describedby', newAriaDescribedBy);

						// Function to update indicator visibility and screen reader announcements
						const updateIndicator = () => {
							const selectedValues = $field.val() || [];
							$header.toggleClass('gravityboard-role-disabled', selectedValues.length === 0);
							if (selectedValues.length === 0) {
								// Show visual indicator
								$indicator.show();
								// Announce to screen readers
								$srAnnouncement.text(placeholder);
							} else {
								// Hide visual indicator
								$indicator.hide();
								// Clear screen reader announcement
								$srAnnouncement.text('');
							}
						};

						// Listen to Select2 dropdown close event only
						$field.on('select2:close', updateIndicator);

						// Also listen to regular change events as fallback for non-Select2 usage
						$field.on('change', function() {
							// Only update if Select2 is not active (fallback for regular selects)
							if (!$field.data('select2')) {
								updateIndicator();
							}
						});

						// Initial state check
						updateIndicator();

						// Mark as processed
						$field.data('gravityboard-disabled-role-initialized', true);
					} else {
						// Check retry limit to prevent infinite recursion
						retryCount++;
						if (retryCount < maxRetries) {
							// Retry with 1ms delay
							setTimeout(checkSelect2, 1);
						} else {
							console.warn('Select2 failed to initialize for field:', fieldId);
						}
					}
				};

				checkSelect2();
			});
		}

		// Initialize permission field placeholders immediately for accessibility
		setTimeout(initPermissionFieldPlaceholders, 0);

		// Also reinitialize when conditional logic shows new fields
		$(document).on('gform_post_conditional_logic_field_action', function(event, formId, action, targetId) {
			if (action === 'show') {
				setTimeout(initPermissionFieldPlaceholders, 0);
			}
		});
	});
})(jQuery);
