jQuery(document).ready(function($) {
	const strings = gravityboard_entry_detail_checklist_strings;
	const restUrl = strings.rest_url;
	const restNonce = strings.rest_nonce;
	const $metabox = $('.gk-gravityboard-checklists-metabox');
	const ANNOUNCEMENT_DELAY = 3000;

	$metabox.on('change', '.gk-gravityboard-item-checkbox', function() {
		const $checkbox = $(this);
		const $item = $checkbox.closest('.gk-gravityboard-checklist-item');
		const $text = $checkbox.siblings('.gk-gravityboard-item-text');
		const $container = $checkbox.closest('.gk-gravityboard-checklist-container');
		const $feedSection = $checkbox.closest('.gk-gravityboard-board-feed-checklists');
		const $status = $('#gk-checklist-status-' + $checkbox.data('entry-id'));

		const entryId = $checkbox.data('entry-id');
		const feedId = $checkbox.data('feed-id');
		const checklistId = $checkbox.data('checklist-id');
		const itemId = $checkbox.data('item-id');
		const isComplete = $checkbox.prop('checked');

		// Optimistic UI update
		if (isComplete) {
			$text.addClass('completed');
		} else {
			$text.removeClass('completed');
		}

		// Disable checkbox during update
		$checkbox.prop('disabled', true);

		// Announce to screen readers
		$status.text(strings.updatingChecklistItem);

		// Make REST API call using jQuery AJAX
		$.ajax({
			url: restUrl + `boards/${feedId}/cards/${entryId}/checklists/${checklistId}/items/${itemId}`,
			method: 'PUT',
			contentType: 'application/json',
			beforeSend: function(xhr) {
				xhr.setRequestHeader('X-WP-Nonce', restNonce);
			},
			data: JSON.stringify({
				is_complete: isComplete
			}),
			success: function(response) {
				if (response && response.checklists) {
					// Update progress for all checklists based on response
					updateProgressDisplay(response.checklists, $feedSection);

					$status.text(strings.checklistItemUpdated);

					// Clear announcement after delay
					setTimeout(() => $status.text(''), ANNOUNCEMENT_DELAY);
				} else {
					// Handle unexpected response structure as an error
					$status.text(strings.errorUpdatingChecklistItem + ' Please try again.');

					// Clear error message after longer delay
					setTimeout(() => $status.text(''), ANNOUNCEMENT_DELAY * 2);

					// Log the issue for debugging
					console.warn('GravityBoard: Unexpected API response structure when updating checklist item:', response);
				}
			},
			error: function(xhr) {
				// Revert optimistic update
				if (isComplete) {
					$text.removeClass('completed');
					$checkbox.prop('checked', false);
				} else {
					$text.addClass('completed');
					$checkbox.prop('checked', true);
				}

				let errorMessage = strings.errorUpdatingChecklistItem;
			if (xhr.responseJSON && xhr.responseJSON.message) {
				errorMessage = xhr.responseJSON.message;
			}

			$status.text(errorMessage);

			// Clear error message after delay
			setTimeout(() => $status.text(''), 5000);
			},
			complete: function() {
				$checkbox.prop('disabled', false);
			}
		});
	});

	// Handle "Add an item" functionality
	$metabox.on('click', '.gk-gravityboard-show-add-btn', function() {
		const $button = $(this);
		const $section = $button.closest('.gk-gravityboard-add-item-section');
		const $form = $section.find('.gk-gravityboard-add-item-form');
		const $input = $form.find('.gk-gravityboard-add-item-input');

		$button.hide();
		$form.show();
		$input.focus();
	});

	$metabox.on('click', '.gk-gravityboard-cancel-add-btn', function() {
		const $button = $(this);
		const $section = $button.closest('.gk-gravityboard-add-item-section');
		const $form = $section.find('.gk-gravityboard-add-item-form');
		const $showBtn = $section.find('.gk-gravityboard-show-add-btn');
		const $input = $form.find('.gk-gravityboard-add-item-input');

		$input.val('');
		$form.hide();
		$showBtn.show().focus();
	});

	$metabox.on('click', '.gk-gravityboard-add-item-btn', function() {
		const $button = $(this);
		const $section = $button.closest('.gk-gravityboard-add-item-section');
		const $input = $section.find('.gk-gravityboard-add-item-input');
		const $feedSection = $button.closest('.gk-gravityboard-board-feed-checklists');
		const $status = $feedSection.find('.gk-checklist-status-sr');

		const entryId = $section.data('entry-id');
		const feedId = $section.data('feed-id');
		const checklistId = $section.data('checklist-id');
		const itemText = $input.val().trim();

		if (!itemText) {
			$input.focus();
			return;
		}

		// Disable form during request
		$input.prop('disabled', true);
		$button.prop('disabled', true);

		$status.text(strings.addingChecklistItem);

		// Add item via REST API
		$.ajax({
			url: restUrl + `boards/${feedId}/cards/${entryId}/checklists/${checklistId}/items`,
			method: 'POST',
			contentType: 'application/json',
			beforeSend: function(xhr) {
				xhr.setRequestHeader('X-WP-Nonce', restNonce);
			},
			data: JSON.stringify({
				checklist_id: checklistId,
				label: itemText
			}),
			success: function(response) {
				if (response && response.item && response.checklists) {
					// Add the new item to the DOM
					const newItem = response.item;
					const $checklistContainer = $section.closest('.gk-gravityboard-checklist-container');
					const $itemsList = $checklistContainer.find('.gk-gravityboard-checklist-items');

					// Create new item HTML
					const newItemHtml = createChecklistItemHtml(newItem, entryId, feedId, checklistId);

					// Add to the list
					if ($itemsList.length === 0) {
						// Create the list if it doesn't exist (was empty before)
						const $noItemsMsg = $checklistContainer.find('.gk-gravityboard-no-items');
						$noItemsMsg.replaceWith('<ul class="gk-gravityboard-checklist-items">' + newItemHtml + '</ul>');
					} else {
						$itemsList.append(newItemHtml);
					}

					// Update progress for all checklists
					updateProgressDisplay(response.checklists, $feedSection);

					$status.text(strings.checklistItemAdded);

					// Clear announcement after delay
					setTimeout(() => $status.text(''), ANNOUNCEMENT_DELAY);

					// Clear form and hide it
					$input.val('');
					$section.find('.gk-gravityboard-add-item-form').hide();
					$section.find('.gk-gravityboard-show-add-btn').show();
				} else {
					// Handle unexpected response structure as an error
					$status.text(strings.errorAddingChecklistItem);

					// Clear error message after longer delay
					setTimeout(() => $status.text(''), ANNOUNCEMENT_DELAY * 2);

					// Log the issue for debugging but don't reload the page
					console.warn('GravityBoard: Unexpected API response structure when adding checklist item:', response);
				}
			},
			error: function(xhr) {
				let errorMessage = strings.errorAddingChecklistItem;
				if (xhr.responseJSON && xhr.responseJSON.message) {
					errorMessage = xhr.responseJSON.message;
				}

				$status.text(errorMessage);

				// Clear error message after delay
				setTimeout(() => $status.text(''), ANNOUNCEMENT_DELAY);
			},
			complete: function() {
				$input.prop('disabled', false);
				$button.prop('disabled', false);
			}
		});
	});

	// Handle Enter and Escape keys in add item input
	$metabox.on('keydown', '.gk-gravityboard-add-item-input', function(e) {
		if (e.key === 'Enter') {
			e.preventDefault();
			$(this).closest('.gk-gravityboard-add-item-section').find('.gk-gravityboard-add-item-btn').click();
		} else if (e.key === 'Escape') {
			e.preventDefault();
			$(this).closest('.gk-gravityboard-add-item-section').find('.gk-gravityboard-cancel-add-btn').click();
		}
	});

	/**
	 * Create HTML for a new checklist item
	 */
	function createChecklistItemHtml(item, entryId, feedId, checklistId) {
		const checked = item.is_complete ? 'checked="checked"' : '';
		const completedClass = item.is_complete ? ' completed' : '';

		return `
			<li class="gk-gravityboard-checklist-item" data-item-id="${item.id}">
				<label class="gk-gravityboard-item-display">
					<input type="checkbox"
						   class="gk-gravityboard-item-checkbox"
						   data-entry-id="${entryId}"
						   data-feed-id="${feedId}"
						   data-checklist-id="${checklistId}"
						   data-item-id="${item.id}"
						   ${checked}>
					<span class="gk-gravityboard-item-text${completedClass}">${escapeHtml(item.label)}</span>
				</label>
			</li>
		`;
	}

	/**
	 * Escape HTML to prevent XSS
	 */
	function escapeHtml(text) {
		const div = document.createElement('div');
		div.textContent = text;
		return div.innerHTML;
	}

	/**
	 * Update progress display for all checklists
	 */
	function updateProgressDisplay(checklistsData, $feedSection) {
		if (!checklistsData || !Array.isArray(checklistsData)) {
			return;
		}

		checklistsData.forEach(function(checklistData) {
			const $checklistContainer = $feedSection.find('[data-checklist-id="' + checklistData.id + '"]');
			if ($checklistContainer.length) {
				const progress = calculateProgress(checklistData.items || []);

				const $progressText = $checklistContainer.find('.gk-gravityboard-progress-text');
				const $progressFill = $checklistContainer.find('.gk-gravityboard-progress-fill');

				$progressText.text( strings.progressText.replace( '{completed}', progress.completed ).replace( '{total}', progress.total ).replace( '{percentage}', progress.percentage ) );
				$progressFill.css('width', progress.percentage + '%');
			}
		});
	}

	/**
	 * Calculate progress from items array
	 */
	function calculateProgress(items) {
		const total = items.length;
		const completed = items.filter(item => item.is_complete).length;
		const percentage = total > 0 ? Math.round((completed / total) * 100) : 0;

		return {
			total: total,
			completed: completed,
			percentage: percentage
		};
	}
});