<?php

/**
 * @package GravityBoard
 * @subpackage Admin
 * @since 1.0.0
 */

namespace GravityKit\GravityBoard\Admin;

use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Helpers;
use GFForms;
/**
 * Admin class.
 *
 * @since 1.0.0
 */
class Admin {

	/**
	 * The instance of the Admin class.
	 *
	 * @since 1.0.0
	 *
	 * @var Admin
	 */
	private static $instance;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		self::init();
	}

	/**
	 * Get the instance of the Admin class.
	 *
	 * @since 1.0.0
	 *
	 * @return Admin The instance of the Admin class.
	 */
	public static function get_instance() {
		static $instance;
		if ( ! $instance ) {
			$instance = new self();
		}
		return $instance;
	}

	/**
	 * Initialize the Admin class.
	 *
	 * @since 1.0.0
	 */
	public static function init() {
		add_filter( 'gform_noconflict_scripts', [ self::class, 'add_noconflict_scripts' ] );
		add_filter( 'gform_toolbar_menu', [ self::class, 'gform_toolbar_menu' ], 10, 2 );
		add_action( 'gform_form_actions', [ self::class, 'gform_toolbar_menu' ], 10, 2 );
		add_filter( 'gform_predefined_choices', [ self::class, 'gform_predefined_choices' ] );
		add_filter( 'gform_notification_settings_fields', [ self::class, 'gform_notification_settings_fields' ], 10, 3 );
	}

	/**
	 * Add GravityBoard notification settings fields to the Gravity Forms notification settings.
	 *
	 * @since 1.0.0
	 *
	 * @param array $fields Notification settings fields.
	 * @param array $notification Notification data.
	 * @param array $form Form data.
	 *
	 * @return array Modified array of notification settings fields.
	 */
	public static function gform_notification_settings_fields( $fields, $notification, $form ) {

		if ( ! isset( $fields[0]['fields'] ) ) {
			return $fields;
		}

		foreach ( $fields[0]['fields'] as &$field ) {
			if ( 'toEmail' === $field['name'] ) {
				$field['class'] = 'merge-tag-support mt-position-right mt-hide_all_fields';
			}
		}

		return $fields;
	}

	/**
	 * Add GravityBoard scripts to Gravity Forms no-conflict list.
	 *
	 * @since 1.0.0
	 *
	 * @param array $scripts Array of script handles.
	 *
	 * @return array Modified array of script handles.
	 */
	public static function add_noconflict_scripts( $scripts ) {
		$scripts[] = 'gravityboard-app';
		$scripts[] = 'gravityboard_admin_conditional_logic_script';
		$scripts[] = 'gravityboard_admin_feed_settings';
		$scripts[] = 'gravityboard_entry_detail_mentions';

		return $scripts;
	}

	/**
	 * Add a GravityView menu to the Form Toolbar with connected views
	 *
	 * @since 1.0.0
	 *
	 * @param array    $menu_items      Menu items, as set in GFForms::top_toolbar()
	 * @param int|null $current_form_id ID of the current Gravity form.
	 *
	 * @return array Modified array
	 */
	public static function gform_toolbar_menu( $menu_items = array(), $current_form_id = null ) {

		// Don't show on Trashed forms.
		if ( 'trash' === rgget( 'filter' ) && 'gf_edit_forms' === rgget( 'page' ) ) {
			return $menu_items;
		}

		$feeds = Feed::get_instance()->get_active_feeds( $current_form_id );

		// Remove the current feed from the list of feeds.
		$current_feed_id = Feed::get_instance()->get_current_feed_id();
		if ( ! empty( $current_feed_id ) ) {
			foreach ( $feeds as $key => $feed ) {
				if ( $feed['id'] === $current_feed_id ) {
					unset( $feeds[ $key ] );
				}
			}
		}

		$priority = -1;

		if ( 'form_list' === GFForms::get_page() ) {
			$priority = 750;
		}

		$capabilities = [
			'gravityforms_edit_forms',
			'gravityforms_create_form',
			'gravityforms_preview_forms',
		];

		// Short circuit to create just a top item when no calendars exist.
		if ( empty( $feeds ) ) {

			$menu_items['gravityboard'] = array(
				'label'        => esc_attr__( 'Create a Board', 'gk-gravityboard' ),
				'title'        => esc_attr__( 'Create a Board using this form as a data source', 'gk-gravityboard' ),
				'url'          => Feed::get_edit_feed_url( 0, $current_form_id ),
				'menu_class'   => 'gf_form_toolbar_settings',
				'priority'     => $priority,
				'capabilities' => $capabilities,
			);

			return $menu_items;
		}

		$sub_menu_items = array();
		$form           = Helpers::get_form( $current_form_id );

		// Create the submenu items array.
		foreach ( (array) $feeds as $feed ) {

			$label = Feed::get_instance()->get_board_name( $feed, $form );

			$sub_menu_items[] = array(
				'label' => $label,
				'url'   => Feed::get_edit_feed_url( $feed['id'], $current_form_id ),
				'icon'  => Feed::get_instance()->get_menu_icon(),
			);
		}

		// Add a final item to create a calendar.
		$sub_menu_items[] = array(
			'label'        => esc_attr__( 'Create a Board', 'gk-gravityboard' ),
			'link_class'   => 'gv-board-create',
			'icon'         => '<i class="dashicons dashicons-plus"></i>',
			'title'        => esc_attr__( 'Create a Board using this form as a data source', 'gk-gravityboard' ),
			'url'          => Feed::get_edit_feed_url( 0, $current_form_id ),
			'capabilities' => $capabilities,
		);

		// Create the top menu to hold the submenu items.
		$menu_items['gravityboard'] = array(
			'label'          => __( 'Boards', 'gk-gravityboard' ),
			'icon'           => Feed::get_instance()->get_menu_icon(),
			'title'          => __( 'Boards using this form as a data source', 'gk-gravityboard' ),
			'url'            => add_query_arg( 'id', $current_form_id, admin_url( Feed::get_view_boards_url() ) ),
			'menu_class'     => 'gf_form_toolbar_settings',
			'sub_menu_items' => $sub_menu_items,
			'priority'       => $priority,
			'capabilities'   => $capabilities,
		);

		return $menu_items;
	}

	/**
	 * Add GravityBoard predefined choices to the Gravity Forms form editor.
	 *
	 * @since 1.0.0
	 *
	 * @param array $existing_choices Existing choices.
	 *
	 * @return array Modified array of choices.
	 */
	public static function gform_predefined_choices( $existing_choices ) {

		$predefined_choices = [];

		$predefined_choices[ esc_html__( 'Project Management', 'gk-gravityboard' ) ] = [
			esc_html__( 'Backlog', 'gk-gravityboard' ),
			esc_html__( 'To Do', 'gk-gravityboard' ),
			esc_html__( 'In Progress', 'gk-gravityboard' ),
			esc_html__( 'In Review', 'gk-gravityboard' ),
			esc_html__( 'Done', 'gk-gravityboard' ),
		];

		$predefined_choices[ esc_html__( 'Software Development', 'gk-gravityboard' ) ] = [
			esc_html__( 'Backlog', 'gk-gravityboard' ),
			esc_html__( 'Sprint Planning', 'gk-gravityboard' ),
			esc_html__( 'In Progress', 'gk-gravityboard' ),
			esc_html__( 'Code Review', 'gk-gravityboard' ),
			esc_html__( 'QA', 'gk-gravityboard' ),
			esc_html__( 'Done', 'gk-gravityboard' ),
		];

		$predefined_choices[ esc_html__( 'Personal Productivity', 'gk-gravityboard' ) ] = [
			esc_html__( 'Inbox', 'gk-gravityboard' ),
			esc_html__( 'Next', 'gk-gravityboard' ),
			esc_html__( 'Doing', 'gk-gravityboard' ),
			esc_html__( 'Waiting', 'gk-gravityboard' ),
			esc_html__( 'Done', 'gk-gravityboard' ),
		];

		$predefined_choices[ esc_html__( 'Marketing Campaign', 'gk-gravityboard' ) ] = [
			esc_html__( 'Ideas', 'gk-gravityboard' ),
			esc_html__( 'Planned', 'gk-gravityboard' ),
			esc_html__( 'Creating', 'gk-gravityboard' ),
			esc_html__( 'Scheduled', 'gk-gravityboard' ),
			esc_html__( 'Live', 'gk-gravityboard' ),
			esc_html__( 'Analyzing', 'gk-gravityboard' ),
		];

		$predefined_choices[ esc_html__( 'Product Roadmap', 'gk-gravityboard' ) ] = [
			esc_html__( 'Now', 'gk-gravityboard' ),
			esc_html__( 'Next', 'gk-gravityboard' ),
			esc_html__( 'Later', 'gk-gravityboard' ),
		];

		$predefined_choices[ esc_html__( 'Recruiting Pipeline', 'gk-gravityboard' ) ] = [
			esc_html__( 'Sourcing', 'gk-gravityboard' ),
			esc_html__( 'Applied', 'gk-gravityboard' ),
			esc_html__( 'Interviewing', 'gk-gravityboard' ),
			esc_html__( 'Offer Extended', 'gk-gravityboard' ),
			esc_html__( 'Hired / Rejected', 'gk-gravityboard' ),
		];

		$predefined_choices[ esc_html__( 'Content Creation', 'gk-gravityboard' ) ] = [
			esc_html__( 'Ideas', 'gk-gravityboard' ),
			esc_html__( 'Drafting', 'gk-gravityboard' ),
			esc_html__( 'Editing', 'gk-gravityboard' ),
			esc_html__( 'Scheduled', 'gk-gravityboard' ),
			esc_html__( 'Published', 'gk-gravityboard' ),
		];

		$predefined_choices[ esc_html__( 'Event Planning', 'gk-gravityboard' ) ] = [
			esc_html__( 'To Do', 'gk-gravityboard' ),
			esc_html__( 'In Progress', 'gk-gravityboard' ),
			esc_html__( 'Completed', 'gk-gravityboard' ),
			esc_html__( 'Post-Mortem', 'gk-gravityboard' ),
		];

		// Do this here instead of across each choice in order to simplify translation.
		foreach ( $predefined_choices as $key => $value ) {
			// translators: This is a label for a preset choice. %s is the name of the preset.
			$label                        = sprintf( esc_html__( '%s (GravityBoard)', 'gk-gravityboard' ), $key );
			$predefined_choices[ $label ] = $value;
			unset( $predefined_choices[ $key ] );
		}

		return array_merge( $existing_choices, $predefined_choices );
	}
}
