<?php
/**
 * Admin Board View
 *
 * @package GravityKit\GravityBoard
 * @since 1.0
 */

namespace GravityKit\GravityBoard\Admin;

use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Renderer;
use GravityKit\GravityBoard\Helpers;
use GFCommon;
use WP_Error;

/**
 * Class Board_View
 *
 * Handles displaying boards in the WordPress admin.
 */
class Board_View {

	const ADMIN_VIEW_PAGE_SLUG = 'gk-gravityboard-view';

	/**
	 * Initialize the board view.
	 *
	 * @since 1.0
	 */
	public static function init() {
		add_action( 'admin_menu', [ __CLASS__, 'register_admin_page' ] );
	}

	/**
	 * Register the admin page.
	 *
	 * @since 1.0
	 */
	public static function register_admin_page() {
		// Add hidden page that's not in the menu.
		add_submenu_page(
			'gravityboard', // This parent doesn't exist, but it's needed to make the page hidden.
			esc_html__( 'Board View', 'gk-gravityboard' ),
			esc_html__( 'Board View', 'gk-gravityboard' ),
			'read', // Changed to 'read' to allow access to more users as the permission check is done inside render_board_view.
			self::ADMIN_VIEW_PAGE_SLUG,
			[ __CLASS__, 'render_board_view' ]
		);
	}

	/**
	 * Get the validated feed from the URL.
	 *
	 * @since 1.0
	 *
	 * @return array|WP_Error The feed array, or a WP_Error if the feed is not found or the user does not have permission.
	 */
	private static function get_validated_feed_from_url() {
		$feed_id = (int) rgget( 'feed_id' );

		if ( empty( $feed_id ) ) {
			return new WP_Error( 'invalid_board_id', esc_html__( 'Invalid board ID.', 'gk-gravityboard' ) );
		}

		$feed_instance = Feed::get_instance();
		$feed          = $feed_instance->get_feed( $feed_id );

		if ( ! $feed ) {
			return new WP_Error( 'board_not_found', esc_html__( 'Board not found.', 'gk-gravityboard' ) );
		}

		if ( ! Helpers::user_has_permission( 'view_board', $feed ) ) {
			return new WP_Error( 'permission_denied', esc_html__( 'You do not have permission to view this board.', 'gk-gravityboard' ) );
		}

		return $feed;
	}

	/**
	 * Render the board view page.
	 *
	 * @since 1.0
	 */
	public static function render_board_view() {

		add_filter(
			'gk/gravityboard/board-settings',
			function ( $board_settings ) {
				$board_settings['use_full_height'] = 1;
				return $board_settings;
			},
			10,
			1
		);

		// Get the feed ID from the URL.
		$feed = self::get_validated_feed_from_url();

		if ( is_wp_error( $feed ) ) {
			echo wpautop( esc_html( $feed->get_error_message() ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			return;
		}

		// Get the form.
		$form = Helpers::get_form( $feed['form_id'] );

		if ( ! $form ) {
			echo wpautop( esc_html__( 'Form not found.', 'gk-gravityboard' ) ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			return;
		}

		// Enqueue necessary styles and scripts.
		Renderer::get_instance()->enqueue_styles( $feed['id'] );

		// Handle feed name display.
		$feed_name = Feed::get_instance()->get_board_name( $feed, $form );

		// Add admin-specific styles.
		self::enqueue_admin_styles();

		// Render the admin header.
		?>
		<div class="wrap gk-gravityboard-admin-view">
			<div class="gk-gravityboard-admin-view-header">
				<h1 class="wp-heading-inline screen-reader-text"><?php echo esc_html( $feed_name ); ?></h1>

				<div class="gk-gravityboard-admin-view-header-actions">
					<ul>
						<li>
							<a href="<?php echo esc_url( Feed::get_view_boards_url() ); ?>" class="page-title-action button-secondary">
								<i class="gform-icon gform-icon--list"></i>
								<?php esc_html_e( 'All Boards', 'gk-gravityboard' ); ?>
							</a>
						</li>
						<?php
						if ( GFCommon::current_user_can_any( 'gravityforms_edit_forms' ) ) {
							?>
						<li>
							<a href="<?php echo esc_url( Feed::get_edit_feed_url( $feed['id'] ) ); ?>" class="page-title-action button-secondary">
								<i class="gform-icon gform-icon--cog"></i>
								<?php esc_html_e( 'Edit Board', 'gk-gravityboard' ); ?>
							</a>
						</li>
							<?php
						}
						?>
					</ul>
				</div>
			</div>

			<hr class="wp-header-end">

			<?php echo do_shortcode( '[gravityboard id="' . esc_attr( $feed['id'] ) . '"]' ); ?>
		</div>
		<?php
	}

	/**
	 * Enqueue admin-specific styles.
	 *
	 * @since 1.0
	 */
	private static function enqueue_admin_styles() {
		$css = '
			.gk-gravityboard-admin-view-header-actions {
				display: flex;
				gap: var(--gravityboard-spacing-unit, 8px);
				align-items: center;
				flex-wrap: wrap;
				flex-direction: row;
				justify-content: flex-start;
			}
			.gk-gravityboard-admin-view-header-actions ul {
				display: flex;
				gap: var(--gravityboard-spacing-unit, 8px);
				align-items: center;
				flex-wrap: wrap;
				flex-direction: row;
			}
			.gk-gravityboard-admin-view-header-actions ul li a,
			.gk-gravityboard-admin-view-header-actions ul li a:active {
				margin: 0!important;
				display: inline-flex!important;
				align-items: center!important;
				gap: var(--gravityboard-spacing-unit, 8px);
				padding: var(--gravityboard-spacing-xs) var(--gravityboard-spacing-md)!important;
			}
			.gk-gravityboard-admin-view-header-actions ul li i {
				font-size: 18px!important;
				width: 18px!important;
				min-height: 18px!important;
				vertical-align: middle;
			}
			.gk-gravityboard-admin-view-header {
				position: relative;
				padding: var(--gravityboard-spacing-unit, 8px) 0 var(--gravityboard-spacing-md, 16px) 0;
			}
			:root {
				--gravityboard-height: calc( 100vh - 65px - 32px - ( var(--gravityboard-spacing-lg) * 2 ) - 100px );
			}
		';

		wp_add_inline_style( Renderer::CSS_ASSET_NAME, $css );
	}
}
