<?php
/**
 * GravityBoard REST Controller
 *
 * @package GravityBoard
 *
 * @since   1.0
 */

namespace GravityKit\GravityBoard;

use Exception;
use GF_Query;
use GF_Query_Condition;
use GravityKit\GravityBoard\Assignees\Assignees;
use GravityKit\GravityBoard\QueryFilters\QueryFilters;
use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use GFAPI;

defined( 'ABSPATH' ) || exit;

/**
 * Class Rest
 *
 * Handles all GravityBoard operations through the WP REST API.
 *
 * @since 1.0.0
 */
class Ajax {

	/**
	 * REST namespace.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	const REST_NAMESPACE = 'gravityboard/v1';
	const NAMESPACE      = 'gravityboard/v1';

	const NONCE_ACTION = 'gravityboard_nonce';

	/**
	 * Singleton instance.
	 *
	 * @since 1.0.0
	 *
	 * @var Rest|null
	 */
	private static $instance = null;

	/**
	 * Get singleton instance.
	 *
	 * @since 1.0.0
	 *
	 * @return Ajax
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Rest constructor – registers the routes.
	 */
	private function __construct() {
		add_action( 'rest_api_init', [ $this, 'register_routes' ] );
	}

	/**
	 * Register all REST routes.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function register_routes() {
		/*
		 * Entries (Cards)
		 * ---------------
		 */
		register_rest_route(
			self::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/cards',
			[
				[
					'methods'             => WP_REST_Server::READABLE,
					'permission_callback' => [ $this, 'can_view_board' ],
					'callback'            => [ $this, 'get_cards' ],
					'args'                => [
						'feed_id' => [
							'required'          => true,
							'validate_callback' => function ( $param ) {
								return is_numeric( $param );
							},
						],
					],
				],
				[
					'methods'             => WP_REST_Server::CREATABLE,
					'permission_callback' => [ $this, 'can_add_card' ],
					'callback'            => [ $this, 'add_card' ],
				],
			]
		);

		/*
		 * Entries (Cards) with feed_id in URL
		 * -----------------------------------
		 */
		register_rest_route(
			self::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)',
			[
				[
					'methods'             => WP_REST_Server::EDITABLE,
					'permission_callback' => [ $this, 'can_edit_card' ],
					'callback'            => [ $this, 'update_card' ],
				],
				[
					'methods'             => WP_REST_Server::DELETABLE,
					'permission_callback' => [ $this, 'can_delete_card' ],
					'callback'            => [ $this, 'delete_card' ],
				],
			]
		);

		register_rest_route(
			self::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/move',
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'permission_callback' => [ $this, 'can_move_card' ],
				'callback'            => [ $this, 'move_card' ],
			]
		);

		/*
		 * Lanes
		 * -----
		 */
		register_rest_route(
			self::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/lanes',
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'permission_callback' => [ $this, 'can_modify_lane' ],
				'callback'            => [ $this, 'add_lane' ],
			]
		);

		register_rest_route(
			self::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/lanes/(?P<lane_id>\d+)',
			[
				[
					'methods'             => WP_REST_Server::EDITABLE,
					'permission_callback' => [ $this, 'can_modify_lane' ],
					'callback'            => [ $this, 'update_lane' ],
				],
				[
					'methods'             => WP_REST_Server::DELETABLE,
					'permission_callback' => [ $this, 'can_modify_lane' ],
					'callback'            => [ $this, 'delete_lane' ],
				],
			]
		);

		register_rest_route(
			self::NAMESPACE,
			'/boards/(?P<feed_id>\d+)/lanes/(?P<lane_id>\d+)/move',
			[
				'methods'             => WP_REST_Server::EDITABLE,
				'permission_callback' => [ $this, 'can_modify_lane' ],
				'callback'            => [ $this, 'move_lane' ],
			]
		);

		/**
		 * Allow other plugins to register routes.
		 *
		 * @since 1.0.0
		 *
		 * @param Ajax $this The Ajax instance.
		 */
		do_action( 'gk/gravityboard/ajax/register-routes', $this );
	}

	/**
	 * Ensure the current user can view the board.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_view_board( WP_REST_Request $request ): bool {
		$feed = $this->get_feed_from_request( $request );
		return Feed::get_instance()->user_has_permission( 'view_board', $feed );
	}

	/**
	 * Ensure the current user can add a card.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_add_card( WP_REST_Request $request ): bool {
		$feed = $this->get_feed_from_request( $request );
		return Feed::get_instance()->user_has_permission( 'add_card', $feed );
	}

	/**
	 * Ensure the current user can edit a card.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_edit_card( WP_REST_Request $request ): bool {
		$feed = $this->get_feed_from_request( $request );
		return Feed::get_instance()->user_has_permission( 'edit_card', $feed );
	}

	/**
	 * Ensure the current user can move a card.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_move_card( WP_REST_Request $request ): bool {
		$feed = $this->get_feed_from_request( $request );
		return Feed::get_instance()->user_has_permission( 'move_card', $feed );
	}

	/**
	 * Ensure the current user can delete a card.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_delete_card( WP_REST_Request $request ): bool {

		$feed = $this->get_feed_from_request( $request );
		return Feed::get_instance()->user_has_permission( 'delete_card', $feed );
	}

	/**
	 * Ensure the current user can add, edit, move or delete lanes.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_modify_lane( WP_REST_Request $request ): bool {
		$feed = $this->get_feed_from_request( $request );
		return Feed::get_instance()->user_has_permission( 'modify_lane', $feed );
	}

	/**
	 * Attachment permissions.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_add_attachment( WP_REST_Request $request ): bool {
		return $this->can_edit_card( $request );
	}

	/**
	 * Check if the current user can delete an attachment.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool True if the user has permission, false otherwise.
	 */
	public function can_delete_attachment( WP_REST_Request $request ): bool {
		return $this->can_edit_card( $request );
	}

	/**
	 * Fetch cards for a given feed.
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function get_cards( WP_REST_Request $request ) {
		$feed          = $this->get_feed_from_request( $request );
		$lane_field_id = Feed::get_instance()->get_lane_field_id( $feed );
		if ( ! $lane_field_id ) {
			return new WP_Error( 'lane_field_not_set', __( 'Lane field not configured.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$form_id    = $feed['form_id'];
		$form       = Helpers::get_form( $form_id );
		$lane_field = GFAPI::get_field( $form, $lane_field_id );
		if ( ! $lane_field ) {
			return new WP_Error( 'lane_field_not_found', __( 'Lane field not found.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		// Build lanes.
		$lanes = Lane::build_lanes_from_field( $lane_field, $feed );

		// Search criteria & conditional logic.
		$search_criteria = [ 'status' => 'active' ];
		$paging          = [
			'offset'    => 0,
			'page_size' => Feed::MAX_ENTRIES_PER_BOARD,
		];

		$conditional_logic = Feed::get_conditional_logic( $feed['meta'] ?? [] );
		$query             = new GF_Query( $form_id, $search_criteria, null, $paging );

		if ( is_array( $conditional_logic ) ) {
			try {
				$conditions = QueryFilters::create()
					->with_form( $form )
					->with_filters( $conditional_logic )
					->get_query_conditions();
			} catch ( Exception $e ) {
				return new WP_Error( 'invalid_conditions', __( 'Invalid conditional logic.', 'gk-gravityboard' ), [ 'status' => 400 ] );
			}

			if ( $conditions ) {
				$query_parts = $query->_introspect();
				$query->where( GF_Query_Condition::_and( $query_parts['where'], $conditions ) );
			}
		}

		$entries = $query->get();
		if ( is_wp_error( $entries ) ) {
			return $entries;
		}

		foreach ( $entries as $entry ) {
			$lane_value                   = rgar( $entry, (string) $lane_field_id );
			$lane_id                      = Helpers::get_lane_id_from_value( $lanes, $lane_value );
			$lanes[ $lane_id ]['cards'][] = Card::build_data_from_entry( $entry, $feed );
		}

		foreach ( $lanes as &$lane ) {
			$lane['cards'] = $this->sort_cards_by_position( $lane['cards'] );
		}

		// Remove "(Not Set)" lane if it has no cards.
		if ( isset( $lanes['-1'] ) && 0 === count( $lanes['-1']['cards'] ) ) {
			unset( $lanes['-1'] );
		}

		/**
		 * Filter lanes data before returning.
		 */
		$lanes = apply_filters( 'gk/gravityboard/data/lanes', $lanes, $entries, $feed );

		return rest_ensure_response(
			[
				'lanes'       => array_values( $lanes ),
				'max_entries' => Feed::MAX_ENTRIES_PER_BOARD,
			]
		);
	}

	/**
	 * Add a new card (Gravity Forms entry).
	 *
	 * @param WP_REST_Request $request Request object.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function add_card( WP_REST_Request $request ) {
		$feed     = $this->get_feed_from_request( $request );
		$lane_id  = (int) $request->get_param( 'lane_id' );
		$position = (int) $request->get_param( 'position' );

		// Create card using Card class.
		$created_entry = Card::create_from_request( $request, $feed, $lane_id, $position );

		if ( is_wp_error( $created_entry ) ) {
			return $created_entry;
		}

		if ( $created_entry instanceof ErrorCollection ) {
			return $created_entry->as_wp_error();
		}

		/**
		 * Runs when a card (entry) has been created, before the notification is triggered.
		 *
		 * @since 1.0
		 *
		 * @param array           $created_entry The created entry data.
		 * @param array           $feed The feed data.
		 * @param WP_REST_Request $request The REST request object.
		 */
		do_action( 'gk/gravityboard/card/added', $created_entry, $feed, $request );

		Notifications::get_instance()->maybe_trigger_notifications(
			$created_entry,
			'gk/gravityboard/card/added',
			$feed
		);

		// Refresh entry data after action hooks to ensure any meta added by hooks (like checklists) is included.
		$refreshed_entry = Helpers::get_card( $created_entry['id'] );

		return rest_ensure_response(
			[
				'message' => __( 'Card added successfully.', 'gk-gravityboard' ),
				'card'    => Card::build_data_from_entry( $refreshed_entry, $feed ),
				'lane_id' => (string) $lane_id,
			]
		);
	}

	/**
	 * Update a card.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 *
	 * @return WP_REST_Response|WP_Error The response or error.
	 */
	public function update_card( WP_REST_Request $request ) {
		$entry = $this->get_entry_from_request( $request );

		if ( is_wp_error( $entry ) ) {
			return $entry;
		}

		$feed = $this->get_feed_from_request( $request );

		// Update card using Card class.
		$result = Card::update_from_request( $request, $entry, $feed );
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		$changes       = $result['changes'];
		$updated_entry = $result['updated_entry'];

		/**
		 * Allow plugins to add additional changes to the card.
		 *
		 * @param array $changes The changes to the card, each element is a string of the field or meta key that was updated.
		 * @param WP_REST_Request $request The REST request object.
		 * @param array $entry The entry data.
		 * @param array $feed The feed data.
		 */
		$additional_changes = apply_filters( 'gk/gravityboard/card/update-card', [], $request, $entry, $feed );

		if ( ! empty( $additional_changes ) ) {
			foreach ( $additional_changes as $additional_change ) {
				if ( is_wp_error( $additional_change ) ) {
					return $additional_change;
				}
			}

			$changes = array_unique( array_merge( $changes, $additional_changes ) );
		}

		if ( empty( $changes ) ) {
			return rest_ensure_response(
				[
					'message'        => __( 'No changes detected.', 'gk-gravityboard' ),
					'updated_fields' => [],
					'card'           => Card::build_data_from_entry( $entry, $feed ),
				]
			);
		}

		/**
		 * Runs when a card (entry) has been updated, before the notification is triggered.
		 *
		 * @since 1.0
		 *
		 * @param array $updated_entry The updated entry data.
		 * @param array $feed The feed data.
		 * @param array $changes The changes to the card, each element is a string of the field or meta key that was updated.
		 */
		do_action( 'gk/gravityboard/card/edited', $updated_entry, $feed, $changes );

		Notifications::get_instance()->maybe_trigger_notifications(
			$updated_entry,
			'gk/gravityboard/card/edited',
			$feed
		);

		return rest_ensure_response(
			[
				'message'        => __( 'Card updated successfully.', 'gk-gravityboard' ),
				'updated_fields' => $changes,
				'card'           => Card::build_data_from_entry( $updated_entry, $feed ),
			]
		);
	}

	/**
	 * Move/re-order a card.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error The response or error.
	 */
	public function move_card( WP_REST_Request $request ) {
		$entry = $this->get_entry_from_request( $request );
		if ( is_wp_error( $entry ) ) {
			return $entry;
		}

		$entry_id      = $entry['id'];
		$feed          = $this->get_feed_from_request( $request );
		$feed_id       = $feed['id'];
		$lane_field_id = Feed::get_instance()->get_lane_field_id( $feed );

		$target_lane_id   = (int) $request->get_param( 'target_lane_id' );
		$source_lane_id   = $request->get_param( 'source_lane_id', null );
		$ordered_ids_json = $request->get_param( 'ordered_card_ids' );
		$ordered_ids      = json_decode( $ordered_ids_json, true );

		// Check for JSON decode errors.
		if ( json_last_error() !== JSON_ERROR_NONE ) {
			return new WP_Error( 'invalid_json', __( 'Invalid JSON format for ordered card IDs.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$ordered_ids = array_map( 'intval', (array) $ordered_ids );

		$form       = Helpers::get_form( $feed['form_id'] );
		$lane_field = GFAPI::get_field( $form, $lane_field_id );

		if ( ! isset( $lane_field->choices[ $target_lane_id ] ) ) {
			return new WP_Error( 'lane_not_found', __( 'Target lane not found.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$is_lane_change = null !== $source_lane_id && (string) $source_lane_id !== (string) $target_lane_id;
		if ( $is_lane_change ) {
			$new_lane_value = $lane_field->choices[ $target_lane_id ]['value'];

			// Validate and update lane.
			$validation_result = Helpers::validate_field_value( $lane_field_id, $new_lane_value, $form, $entry );
			if ( is_wp_error( $validation_result ) ) {
				return $validation_result;
			}

			$result = GFAPI::update_entry_field( $entry_id, $lane_field_id, $new_lane_value );
			if ( true !== $result ) {
				return new WP_Error( 'update_failed', __( 'Failed to update lane.', 'gk-gravityboard' ), [ 'status' => 500 ] );
			}
		}

		$order_changed = $this->process_card_order_changes( $ordered_ids, $feed_id );

		$updated_entry = Helpers::get_card( $entry_id );

		if ( $is_lane_change ) {

			/**
			 * Runs when a card (entry) has been moved to a new lane, before the notification is triggered.
			 *
			 * @since 1.0
			 *
			 * @param array $updated_entry The updated entry data.
			 * @param array $feed The feed data.
			 * @param int   $source_lane_id The ID of the source lane.
			 * @param int   $target_lane_id The ID of the target lane.
			 */
			do_action( 'gk/gravityboard/card/changed-lane', $updated_entry, $feed, $source_lane_id, $target_lane_id );

			Notifications::get_instance()->maybe_trigger_notifications(
				$updated_entry,
				'gk/gravityboard/card/changed-lane',
				$feed
			);
		} elseif ( null !== $order_changed ) {

			/**
			 * Runs when a card (entry) has been sorted, before the notification is triggered.
			 *
			 * @since 1.0
			 *
			 * @param array $updated_entry The updated entry data.
			 * @param array $feed The feed data.
			 * @param int   $source_lane_id The ID of the lane with cards that were sorted.
			 * @param array $ordered_ids The ordered card IDs.
			 */
			do_action( 'gk/gravityboard/card/sorted', $updated_entry, $feed, $source_lane_id, $ordered_ids );

			Notifications::get_instance()->maybe_trigger_notifications(
				$updated_entry,
				'gk/gravityboard/card/sorted',
				$feed
			);
		}

		return rest_ensure_response(
			[
				'card_id'          => $entry_id,
				'lane_id'          => (string) $target_lane_id,
				'is_lane_change'   => (bool) $is_lane_change,
				'order_changed'    => $order_changed,
				'ordered_card_ids' => array_map( 'strval', $ordered_ids ),
			]
		);
	}

	/**
	 * Delete a card.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error The response or error.
	 */
	public function delete_card( WP_REST_Request $request ) {
		$entry = $this->get_entry_from_request( $request );
		if ( is_wp_error( $entry ) ) {
			return $entry;
		}

		$entry_id = $entry['id'];
		$feed     = $this->get_feed_from_request( $request );

		$deletion_mode         = rgars( $feed, 'meta/deletion_mode', 'trash' );
		$trigger_notifications = rgars( $feed, 'meta/trigger_notifications', '1' ) === '1';
		$disable_hook          = ! $trigger_notifications;

		if ( 'trash' === $deletion_mode ) {
			$result = GFAPI::update_entry_property( $entry_id, 'status', 'trash', false, $disable_hook );
		} else {
			if ( $disable_hook ) {
				remove_all_actions( 'gform_delete_entry' );
				remove_all_actions( 'gform_delete_lead' );
			}
			$result = GFAPI::delete_entry( $entry_id );
		}

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		/**
		 * Runs when a card (entry) has been trashed or deleted, before the notification is triggered.
		 *
		 * @since 1.0
		 *
		 * @param array  $entry The entry data.
		 * @param array  $feed The feed data.
		 * @param string $deletion_mode The deletion mode ("trash" or "delete").
		 */
		do_action( 'gk/gravityboard/card/deleted', $entry, $feed, $deletion_mode );

		Notifications::get_instance()->maybe_trigger_notifications(
			$entry,
			'gk/gravityboard/card/deleted',
			$feed
		);

		return rest_ensure_response( [ 'deleted' => true ] );
	}

	/**
	 * Add a new lane.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error The response or error.
	 */
	public function add_lane( WP_REST_Request $request ) {
		$feed = $this->get_feed_from_request( $request );

		$result = Lane::create_from_request( $request, $feed );
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		Notifications::get_instance()->maybe_trigger_notifications(
			$result,
			'gk/gravityboard/lane/added',
			$feed
		);

		return rest_ensure_response( $result );
	}

	/**
	 * Update an existing lane.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error The response or error.
	 */
	public function update_lane( WP_REST_Request $request ) {
		$feed    = $this->get_feed_from_request( $request );
		$lane_id = (int) $request->get_param( 'lane_id' );

		$result = Lane::update_from_request( $request, $feed, $lane_id );
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		Notifications::get_instance()->maybe_trigger_notifications(
			$result,
			'gk/gravityboard/lane/edited',
			$feed
		);

		return rest_ensure_response( $result );
	}

	/**
	 * Move/re-order a lane.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error The response or error.
	 */
	public function move_lane( WP_REST_Request $request ) {
		$feed       = $this->get_feed_from_request( $request );
		$new_pos    = (int) $request->get_param( 'position' );
		$current_id = (int) $request->get_param( 'lane_id' );

		$result = Lane::move( $feed, $current_id, $new_pos );
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		Notifications::get_instance()->maybe_trigger_notifications(
			$result,
			'gk/gravityboard/lane/moved',
			$feed
		);

		return rest_ensure_response( $result );
	}

	/**
	 * Delete a lane.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return WP_REST_Response|WP_Error The response or error.
	 */
	public function delete_lane( WP_REST_Request $request ) {
		$feed    = $this->get_feed_from_request( $request );
		$lane_id = (int) $request->get_param( 'lane_id' );

		$result = Lane::delete( $feed, $lane_id );
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		Notifications::get_instance()->maybe_trigger_notifications(
			$result,
			'gk/gravityboard/lane/deleted',
			$feed
		);

		return rest_ensure_response( $result );
	}

	/**
	 * Sort cards by position.
	 *
	 * @param array $cards The cards to sort.
	 * @return array The sorted cards.
	 */
	private function sort_cards_by_position( array $cards ): array {
		usort(
			$cards,
			static function ( $a, $b ) {
				return ( $a['position'] ?? PHP_INT_MAX ) - ( $b['position'] ?? PHP_INT_MAX );
			}
		);

		return $cards;
	}

	/**
	 * Update card positions when order changes.
	 *
	 * @param array $ordered_ids Ordered card IDs in lane.
	 * @param int   $feed_id     Feed ID.
	 *
	 * @return bool|null True if errors occurred, null if order unchanged, false on success.
	 */
	private function process_card_order_changes( array $ordered_ids, int $feed_id ) {
		$current = [];

		foreach ( $ordered_ids as $id ) {
			$current[ $id ] = (int) gform_get_meta( $id, Feed::get_entry_meta_position_key( $feed_id ) );
		}

		$changed = false;
		foreach ( $ordered_ids as $index => $id ) {
			if ( ! isset( $current[ $id ] ) || $current[ $id ] !== $index ) {
				$changed = true;
				break;
			}
		}

		if ( ! $changed ) {
			return null;
		}

		$errors = false;
		foreach ( $ordered_ids as $index => $id ) {
			$ok = gform_update_meta( $id, Feed::get_entry_meta_position_key( $feed_id ), $index );
			if ( false === $ok ) {
				$errors = true;
			}
		}

		return $errors;
	}

	/*
	 * ==================================================
	 * Helpers to hydrate objects from request
	 * ==================================================
	 */

	/**
	 * Get feed from request.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return array The feed data.
	 * @throws \Exception If feed not found or not active.
	 */
	public function get_feed_from_request( WP_REST_Request $request ): array {
		$feed_id = (int) $request->get_param( 'feed_id' );

		// Validate feed ID is positive integer.
		if ( $feed_id <= 0 ) {
			throw new \Exception( esc_html__( 'Invalid feed ID.', 'gk-gravityboard' ), 400 );
		}

		$feed = Feed::get_instance()->get_feed( $feed_id );

		if ( empty( $feed ) || ! $feed['is_active'] ) {
			// Use generic error message to prevent information disclosure.
			throw new \Exception( esc_html__( 'Feed not found.', 'gk-gravityboard' ), 404 );
		}

		return $feed;
	}

	/**
	 * Get entry from request.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return array|WP_Error The entry data or error.
	 */
	public function get_entry_from_request( WP_REST_Request $request ) {
		$entry_id = (int) $request->get_param( 'entry_id' );

		// Validate entry ID is positive integer.
		if ( $entry_id <= 0 ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$entry = Helpers::get_card( $entry_id );

		if ( is_wp_error( $entry ) ) {
			return $entry;
		}

		return $entry;
	}
}
