<?php
/**
 * Assignees class
 *
 * @package GravityBoard
 * @since 1.0
 */

namespace GravityKit\GravityBoard\Assignees;

use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Helpers;
use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Assignees class
 *
 * @package GravityBoard
 * @since 1.0
 */
class Assignees {

	/**
	 * The instance of the Assignees class.
	 *
	 * @since 1.0
	 * @var Assignees
	 */
	private static $instance = null;

	/**
	 * The assignee user ID for the current notification.
	 *
	 * @since 1.0
	 * @var int|null
	 */
	private $notification_assignee_user_id = null;

	/**
	 * Entry meta key for storing assignee data
	 *
	 * @since 1.0
	 * @var string
	 */
	const ASSIGNEE_META_KEY = 'gravityboard_assignees';

	/**
	 * REST API parameter key for assignees.
	 *
	 * @since 1.0
	 * @var string
	 */
	const REST_PARAM = 'assignees';

	/**
	 * Get the instance of the Assignees class.
	 *
	 * @since 1.0
	 * @return Assignees The instance of the Assignees class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.0
	 */
	public function __construct() {

		add_filter( 'gk/gravityboard/card/data', [ $this, 'filter_card_data' ], 10, 3 );

		add_filter( 'gform_entry_meta', [ $this, 'filter_entry_meta' ], 10, 2 );
		add_filter( 'gk/gravityboard/card/update-card', [ $this, 'filter_update_card' ], 10, 4 );
		add_action( 'gk/gravityboard/card/added', [ $this, 'action_card_added' ], 10, 3 );

		// Hook for triggering notifications after assignee updates.
		add_action( 'gk/gravityboard/assignees/added', [ $this, 'handle_assignee_change' ], 10, 4 );
		add_action( 'gk/gravityboard/assignees/removed', [ $this, 'handle_assignee_change' ], 10, 4 );
		add_action( 'gk/gravityboard/assignees/updated', [ $this, 'handle_assignee_change' ], 10, 4 );

		// Hook to format assignee display in Gravity Forms entries table.
		add_filter( 'gform_entries_field_value', [ $this, 'filter_entries_field_value' ], 10, 4 );
	}

	/**
	 * Filter the card data.
	 *
	 * @since 1.1
	 *
	 * @param array $card_data The card data.
	 * @param array $entry The entry data.
	 * @param array $feed The feed data.
	 *
	 * @return array The filtered card data.
	 */
	public function filter_card_data( $card_data, $entry, $feed ) {
		$enable_assignees = (bool) rgars( $feed, 'meta/enable_assignees', false );
		$can_view         = Feed::get_instance()->user_has_permission( 'view_assignees', $feed );

		$assignees = [];
		if ( $enable_assignees && $can_view ) {
			$assignees = self::get_assignees_details( $entry['id'] );
		}

		$card_data['assignees'] = $assignees;

		return $card_data;
	}

	/**
	 * Handles assignee change notifications.
	 *
	 * @param int   $user_id The ID of the user who was assigned.
	 * @param int   $entry_id The ID of the entry that was updated.
	 * @param array $feed The feed associated with the board.
	 * @param array $form The form associated with the entry.
	 *
	 * @return void
	 */
	public function handle_assignee_change( $user_id, $entry_id, $feed, $form ) {
		// Get the current action to determine notification type.
		$current_action = current_action();

		// Check if Gravity Forms is active and required classes exist.
		if ( ! class_exists( 'GFAPI' ) || ! class_exists( 'GFCommon' ) || ! function_exists( 'rgar' ) ) {
			return;
		}

		// Make sure we have a valid form ID from the feed.
		if ( empty( $feed['form_id'] ) ) {
			return;
		}

		// Get form.
		$form = Helpers::get_form( $feed['form_id'] );
		if ( empty( $form ) ) {
			return;
		}

		// Check if assignees are enabled for this feed.
		$assignees_enabled = Feed::get_instance()->get_setting( 'enable_assignees', '1', $feed );
		if ( ! $assignees_enabled ) {
			return;
		}

		// Get the entry.
		$entry = Helpers::get_card( $entry_id );

		if ( is_wp_error( $entry ) ) {
			Feed::get_instance()->log_error( 'Error getting entry for sending notifications: ' . $entry_id . ' - ' . $entry->get_error_message() );
			return;
		}

		// Prepare for notification.
		$event = $current_action; // Using the hook name as the event.

		// Get notifications that are configured for this event.
		$notifications = \GFCommon::get_notifications_to_send( $event, $form, $entry );

		// Filter the assignee user ID for the merge tags.
		$assignee_user_id = function () use ( $user_id ) {
			return $user_id;
		};

		// Add the filter to the merge tags.
		add_filter( 'gk/gravityboard/merge-tags/assignee-user-id', $assignee_user_id );

		// Send notifications - with our filter in place, conditional logic will be handled automatically.
		foreach ( $notifications as $notification ) {
			\GFCommon::send_notification( $notification, $form, $entry );
		}

		remove_filter( 'gk/gravityboard/merge-tags/assignee-user-id', $assignee_user_id );
	}

	/**
	 * Action the card added.
	 *
	 * @since 1.0
	 * @param array           $entry The entry data.
	 * @param array           $feed The feed data.
	 * @param WP_REST_Request $request The REST request object.
	 *
	 * @return void
	 */
	public function action_card_added( $entry, $feed, $request ) {
		$assignees_json = $request->get_param( self::REST_PARAM );

		$assignees = json_decode( $assignees_json, true );

		if ( is_null( $assignees ) || empty( $assignees ) || empty( $entry['id'] ) || empty( $feed['form_id'] ) ) {
			return;
		}

		$form = Helpers::get_form( $feed['form_id'] );

		if ( is_wp_error( $form ) ) {
			Feed::get_instance()->log_error( 'Failed to get form for card added: ' . $entry['id'] );
			return;
		}

		$ok = self::update_assignees( $entry['id'], $assignees, true, $feed, $form );

		if ( false === $ok ) {
			Feed::get_instance()->log_error( 'Failed to add assignees for card added: ' . $entry['id'] . ' - ' . $assignees_json );
		}
	}

	/**
	 * Filter the update card.
	 *
	 * @since 1.0
	 * @param array           $changes The changes to the card.
	 * @param WP_REST_Request $request The REST request object.
	 * @param array           $entry The entry data.
	 * @param array           $feed The feed data.
	 *
	 * @return array|WP_Error The changes to the card, or a WP_Error if the assignees failed to update.
	 */
	public function filter_update_card( $changes, $request, $entry, $feed ) {
		if ( null === $request->get_param( self::REST_PARAM ) ) {
			return $changes;
		}

		$ids  = array_map( 'absint', (array) $request->get_param( self::REST_PARAM ) );
		$form = Helpers::get_form( $feed['form_id'] );
		$ok   = self::update_assignees( $entry['id'], $ids, true, $feed, $form );

		if ( false === $ok ) {
			return new WP_Error( 'assignee_update_failed', __( 'Failed to update assignees.', 'gk-gravityboard' ), [ 'status' => 500 ] );
		}

		if ( true === $ok ) {
			$changes[] = self::REST_PARAM;
		}

		return $changes;
	}

	/**
	 * Filter the entry meta.
	 *
	 * @since 1.0
	 * @param array $entry_meta The entry meta.
	 * @return array The entry meta.
	 */
	public function filter_entry_meta( $entry_meta ) {
		$entry_meta[ self::ASSIGNEE_META_KEY ] = [
			'label'             => esc_html__( 'GravityBoard Assignees', 'gk-gravityboard' ),
			'is_numeric'        => false,
			'is_default_column' => false,
			'filter'            => [
				'operators' => [ 'contains', 'ncontains' ],
			],
		];

		return $entry_meta;
	}

	/**
	 * Get the assignees for an entry.
	 *
	 * @since 1.0
	 * @param int  $entry_id The ID of the entry.
	 * @param bool $validate_user_exists Whether to validate that the user exists.
	 * @return array The assignees for the entry.
	 */
	public static function get_assignees( $entry_id, $validate_user_exists = true ) {
		$assignees = \gform_get_meta( $entry_id, self::ASSIGNEE_META_KEY );

		$assignees = json_decode( $assignees, true );

		if ( is_null( $assignees ) ) {
			return [];
		}

		if ( ! is_array( $assignees ) ) {
			return [];
		}

		return self::sanitize_assignees( $assignees, $validate_user_exists );
	}

	/**
	 * Get the assignees details for an entry.
	 *
	 * @since 1.0
	 * @param int $entry_id The ID of the entry.
	 *
	 * @return array{
	 *     @type int $id The ID of the user.
	 *     @type string $text The text of the user.
	 *     @type string $avatar The avatar of the user.
	 *     @type string $email The email of the user.
	 * } The assignees details for the entry.
	 */
	public static function get_assignees_details( $entry_id ) {
		$assignees = self::get_assignees( $entry_id, true );

		if ( empty( $assignees ) ) {
			return [];
		}

		$formatted_assignees = [];

		foreach ( $assignees as $assignee_id ) {
			$user = get_userdata( $assignee_id );

			if ( ! $user || ! $user->exists() ) {
				continue;
			}

			$formatted_assignees[] = [
				'id'     => $user->ID,
				'text'   => ! empty( $user->display_name ) ? $user->display_name : $user->user_login,
				'avatar' => get_avatar_url( $user->ID, [ 'size' => 64 ] ),
				'email'  => $user->user_email,
			];
		}

		return $formatted_assignees;
	}

	/**
	 * Helper method to validate and sanitize assignees.
	 *
	 * @since 1.0
	 * @param array $assignees The assignees to validate.
	 * @param bool  $validate_user_exists Whether to validate that the user exists.
	 *
	 * @return array Assignees, sorted and unique.
	 */
	public static function sanitize_assignees( $assignees, $validate_user_exists = true ) {
		$assignees = array_map( 'intval', $assignees );
		$assignees = array_unique( $assignees );
		$assignees = array_values( $assignees );
		$assignees = array_filter( $assignees );
		sort( $assignees );

		if ( $validate_user_exists ) {
			$assignees = array_filter(
				$assignees,
				function ( $user_id ) {
					return false !== get_user_by( 'id', $user_id );
				}
			);
		}

		return $assignees;
	}

	/**
	 * Update the assignees for an entry.
	 *
	 * @since 1.0
	 * @param int   $entry_id The ID of the entry.
	 * @param array $assignees The assignees to update. Will be cast to an array of integers.
	 * @param bool  $trigger_hooks Whether to trigger hooks.
	 * @param array $feed The feed associated with the entry.
	 * @param array $form The form associated with the entry.
	 *
	 * @return bool|null True if the assignees were updated, false otherwise, null if no change was made.
	 */
	public static function update_assignees( $entry_id, $assignees, $trigger_hooks = true, $feed = [], $form = [] ) {

		// Validate and sanitize the assignees.
		$assignees = self::sanitize_assignees( $assignees, true );

		$changes = self::get_changes( $entry_id, $assignees, false );

		if ( null === $changes ) {
			return null;
		}

		// This is a hack to allow for searching for assignees using "contains" in Gravity Forms: we need to add at least two.
		// additional values to the array to ensure that IDs are ALWAYS preceded and followed by a comma.
		// That way, the number in between commas will be the full user ID, not a partial match.
		// So we add 0 values to pad the array and then remove them when retrieving the assignees (array_filter handles this).
		$assignees = array_merge( [ 0 ], $assignees, [ 0 ] );

		$result = \gform_update_meta( $entry_id, self::ASSIGNEE_META_KEY, wp_json_encode( $assignees ) );

		if ( false === $result ) {
			return false;
		}

		if ( $trigger_hooks ) {
			self::trigger_hooks( $changes, $entry_id, $assignees, $feed, $form );
		}

		return true;
	}

	/**
	 * Get the changes for the assignees.
	 *
	 * @since 1.0
	 * @param int   $entry_id The ID of the entry.
	 * @param array $assignees The assignees to get changes for.
	 * @param bool  $validate_user_exists Whether to validate that the user exists.
	 *
	 * @return array|null The changes for the assignees, or null if no changes were made.
	 */
	public static function get_changes( $entry_id, $assignees, $validate_user_exists = true ) {
		$assignees         = self::sanitize_assignees( $assignees, $validate_user_exists );
		$current_assignees = self::get_assignees( $entry_id, $validate_user_exists );

		if ( $current_assignees === $assignees ) {
			return null;
		}

		return [
			'added'   => array_diff( $assignees, $current_assignees ),
			'removed' => array_diff( $current_assignees, $assignees ),
		];
	}

	/**
	 * Trigger hooks for the assignees.
	 *
	 * @since 1.0
	 * @param array $changes The changes for the assignees.
	 * @param int   $entry_id The ID of the entry.
	 * @param array $assignees The assignees to trigger hooks for.
	 * @param array $feed The feed associated with the entry.
	 * @param array $form The form associated with the entry.
	 *
	 * @return void
	 */
	private static function trigger_hooks( $changes, $entry_id, $assignees, $feed, $form ) {

		if ( null === $changes ) {
			Feed::get_instance()->log_debug( 'GK Assignees - update_entry_assignees - No change detected for Entry ID: ' . $entry_id );
			return;
		}

		if ( ! isset( $changes['added'] ) && ! isset( $changes['removed'] ) ) {
			Feed::get_instance()->log_debug( 'GK Assignees - update_entry_assignees - Invalid changes for Entry ID: ' . $entry_id );
			return;
		}

		$added_ids   = $changes['added'];
		$removed_ids = $changes['removed'];

		foreach ( $added_ids as $user_id ) {
			/**
			 * Fired when a user is assigned to an entry via the GravityBoard interface.
			 *
			 * @since 1.0
			 *
			 * @param int   $user_id The ID of the user who was assigned.
			 * @param int   $entry_id The ID of the entry that was updated.
			 * @param array $feed The feed associated with the board.
			 * @param array $form The form associated with the entry.
			 */
			do_action( 'gk/gravityboard/assignees/added', (int) $user_id, $entry_id, $feed, $form );
		}

		foreach ( $removed_ids as $user_id ) {
			/**
			 * Fired when a user is unassigned from an entry via the GravityBoard interface.
			 *
			 * @since 1.0
			 *
			 * @param int   $user_id The ID of the user who was unassigned.
			 * @param int   $entry_id The ID of the entry that was updated.
			 * @param array $feed The feed associated with the board.
			 * @param array $form The form associated with the entry.
			 */
			do_action( 'gk/gravityboard/assignees/removed', (int) $user_id, $entry_id, $feed, $form );
		}

		/**
		 * Fired after entry assignees are successfully updated.
		 *
		 * @since 1.0
		 *
		 * @param int   $entry_id The ID of the entry that was updated.
		 * @param array $assignees The assignees for the entry.
		 * @param array $feed The feed associated with the board.
		 * @param array $form The form associated with the entry.
		 */
		do_action( 'gk/gravityboard/assignees/updated', (int) $entry_id, $assignees, $feed, $form );
	}

	/**
	 * Filter the entry list field value to properly display assignees.
	 *
	 * @since 1.0
	 *
	 * @param string $value The current value of the field.
	 * @param int    $form_id The ID of the form.
	 * @param string $field_id The ID of the field.
	 * @param array  $entry The entry object.
	 *
	 * @return string The filtered value of the field.
	 */
	public function filter_entries_field_value( $value, $form_id, $field_id, $entry ) {
		if ( self::ASSIGNEE_META_KEY !== $field_id ) {
			return $value;
		}

		$assignees_details = self::get_assignees_details( $entry['id'] );

		if ( empty( $assignees_details ) ) {
			return '<span class="screen-reader-text">' . esc_html__( 'No assignees', 'gk-gravityboard' ) . '</span></span>';
		}

		$template = '<a href="{url}" title="{title_attr}">{name}</a>';

		foreach ( $assignees_details as $assignee ) {
			$names[] = strtr(
				$template,
				[
					'{url}'        => esc_url_raw( get_edit_user_link( $assignee['id'] ) ),
					'{avatar}'     => esc_url_raw( $assignee['avatar'] ),
					// translators: %s is the name of the assignee.
					'{title_attr}' => sprintf( esc_html__( 'Go to %s\'s user profile', 'gk-gravityboard' ), esc_attr( $assignee['text'] ) ),
					'{name}'       => esc_attr( $assignee['text'] ),
				]
			);
		}

		return implode( ', ', $names );
	}
}
