<?php
/**
 * GravityBoard Assignees Merge Tags
 *
 * @package GravityKit
 * @subpackage GravityBoard
 * @since 1.0
 */

namespace GravityKit\GravityBoard\Assignees;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * GravityBoard Assignees Merge Tags Class
 *
 * @since 1.0
 */
class MergeTags {



	/**
	 * Instance of this class.
	 *
	 * @since 1.0
	 * @var MergeTags
	 */
	private static $instance = null;

	/**
	 * Initialize the class
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	public function __construct() {
		add_filter( 'gform_replace_merge_tags', [ $this, 'replace_merge_tags' ], 10, 7 );
		add_filter( 'gform_custom_merge_tags', [ $this, 'add_custom_merge_tags' ], 10, 2 );
	}

	/**
	 * Return an instance of this class.
	 *
	 * @since 1.0
	 *
	 * @return MergeTags A single instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Add custom merge tags to the merge tag dropdown in the form editor
	 *
	 * @since 1.0
	 *
	 * @param array $merge_tags Array of merge tags.
	 * @return array Modified array of merge tags
	 */
	public function add_custom_merge_tags( $merge_tags ) {

		$merge_tags[] = [
			'label' => esc_html__( 'GravityBoard: Card Assignees: User Emails', 'gk-gravityboard' ),
			'tag'   => '{gravityboard_assignees:emails}',
		];

		$merge_tags[] = [
			'label' => esc_html__( 'GravityBoard: Card Assignees: Display Names', 'gk-gravityboard' ),
			'tag'   => '{gravityboard_assignees:names}',
		];

		$merge_tags[] = [
			'label' => esc_html__( 'Card Assignees: List with Avatars', 'gk-gravityboard' ),
			'tag'   => '{gravityboard_assignees:list}',
		];

		$merge_tags[] = [
			'label' => esc_html__( 'GravityBoard: Card Assignee: User Email', 'gk-gravityboard' ),
			'tag'   => '{gravityboard_assignee_email}',
		];

		$merge_tags[] = [
			'label' => esc_html__( 'GravityBoard: Card Assignee: Display Name', 'gk-gravityboard' ),
			'tag'   => '{gravityboard_assignee:display_name}',
		];

		return $merge_tags;
	}

	/**
	 * Replace custom merge tags with their values
	 *
	 * @since 1.0
	 *
	 * @param string $text The text to replace merge tags in.
	 * @param array  $form The form object.
	 * @param array  $entry The entry object.
	 * @param bool   $url_encode Whether to URL encode the value.
	 * @param bool   $esc_html Whether to escape HTML in the value.
	 * @param bool   $nl2br Whether to convert newlines to <br> tags.
	 * @param string $format The format to apply to the value.
	 * @return string The text with merge tags replaced
	 */
	public function replace_merge_tags( $text, $form, $entry, $url_encode, $esc_html, $nl2br, $format ) {
		if ( empty( $entry ) || empty( $form ) ) {
			return $text;
		}

		// Check if the text contains any assignees tags.
		if ( strpos( $text, '{gravityboard_' ) === false ) {
			return $text;
		}

		$text = $this->replace_assignees_tags( $text, $entry, $url_encode, $esc_html, $format );

		$text = $this->replace_assignee_tags( $text, $entry, $url_encode, $esc_html, $format );

		return $text;
	}

	/**
	 * Replace {gravityboard_assignee:property} merge tags.
	 *
	 * @since 1.0
	 *
	 * @param string $text The text to replace merge tags in.
	 * @param array  $entry The entry object.
	 * @param bool   $url_encode Whether to URL encode the value.
	 * @param bool   $esc_html Whether to escape HTML in the value.
	 * @param string $format The format requested.
	 * @return string The text with merge tags replaced.
	 */
	private function replace_assignee_tags( $text, $entry, $url_encode, $esc_html, $format ) {

		/**
		 * Filter the assignee user.
		 *
		 * @internal Do not rely on this filter; it is designed for internal use only.
		 * @used-by Assignees::handle_assignee_change()
		 *
		 * @since 1.0
		 *
		 * @param int|null $assignee_user_id The assignee user ID.
		 * @param array   $entry The entry object.
		 */
		$assignee_user_id = apply_filters( 'gk/gravityboard/merge-tags/assignee-user-id', null, $entry );

		if ( is_null( $assignee_user_id ) || ! is_int( $assignee_user_id ) ) {
			return $text;
		}

		$assignee_user = new \WP_User( $assignee_user_id );

		if ( ! $assignee_user || ! $assignee_user->exists() ) {
			return $text;
		}

		preg_match_all( '/\{gravityboard_assignee:(.*?)\}/', $text, $matches, PREG_SET_ORDER );
		foreach ( $matches as $match ) {
			$full_tag = $match[0];
			$property = $match[1];

			// Prevent leaking hashed passwords.
			$value = 'user_pass' == $property ? '' : $assignee_user->get( $property );
			$value = $url_encode ? rawurlencode( $value ) : $value;

			$text = str_replace( $full_tag, $value, $text );
		}

		$text = str_replace( '{gravityboard_assignee_email}', $assignee_user->user_email, $text );

		return $text;
	}

	/**
	 * Replace {gravityboard_assignees:property} merge tags.
	 *
	 * @since 1.0
	 *
	 * @param string $text The text to replace merge tags in.
	 * @param array  $entry The entry object.
	 * @param bool   $url_encode Whether to URL encode the value.
	 * @param bool   $esc_html Whether to escape HTML in the value.
	 * @param string $format The format requested.
	 * @return string The text with merge tags replaced.
	 */
	private function replace_assignees_tags( $text, $entry, $url_encode, $esc_html, $format ) {

		$assignee_details = Assignees::get_assignees_details( $entry['id'] );

		if ( empty( $assignee_details ) || ! is_array( $assignee_details ) ) {
			// Replace all assignees tags with empty string.
			$text = preg_replace( '/{gravityboard_assignees:(emails|names|list)}/', '', $text );
			return $text;
		}

		// Emails.
		if ( false !== strpos( $text, '{gravityboard_assignees:emails}' ) ) {

			$emails     = wp_list_pluck( $assignee_details, 'email' );
			$email_list = implode( ', ', $emails );

			if ( $url_encode ) {
				$email_list = rawurlencode( $email_list );
			}

			if ( $esc_html ) {
				$email_list = esc_html( $email_list );
			}

			$text = str_replace( '{gravityboard_assignees:emails}', $email_list, $text );
		}

		// Names.
		if ( false !== strpos( $text, '{gravityboard_assignees:names}' ) ) {

			$name_list = wp_list_pluck( $assignee_details, 'text' );

			$name_list = implode( ', ', $name_list );

			if ( $url_encode ) {
				$name_list = rawurlencode( $name_list );
			}

			if ( $esc_html ) {
				$name_list = esc_html( $name_list );
			}

			$text = str_replace( '{gravityboard_assignees:names}', $name_list, $text );
		}

		// List with avatars (HTML format).
		if ( false !== strpos( $text, '{gravityboard_assignees:list}' ) ) {
			$list_output = '';

			if ( ! empty( $assignee_details ) ) {

				if ( 'html' === $format ) {
					$list_output = '<ul class="gk-gravityboard-assignees-list">';
				}

				foreach ( $assignee_details as $assignee_detail ) {
					$avatar = $assignee_detail['avatar'];
					$name   = $assignee_detail['text'];

					if ( $esc_html ) {
						$name = esc_html( $name );
					} elseif ( $url_encode ) {
						$name = rawurlencode( $name );
					}

					if ( 'html' === $format ) {
						$list_output .= sprintf(
							'<li class="gk-gravityboard-assignee-item">
								<img src="%s" alt="%s" class="gk-gravityboard-assignee-avatar" style="width: 24px; height: 24px; border-radius: 50%%;" />
								<span class="gk-gravityboard-assignee-name">%s</span>
							</li>',
							$avatar,
							$name,
							$name
						);
					} else {
						$list_output .= '- ' . $name . PHP_EOL;
					}
				}

				if ( 'html' === $format ) {
					$list_output .= '</ul>';
				}
			}

			if ( $url_encode ) {
				$list_output = rawurlencode( $list_output );
			}

			if ( $esc_html ) {
				$list_output = esc_html( $list_output );
			}

			$text = str_replace( '{gravityboard_assignees:list}', $list_output, $text );
		}

		return $text;
	}
}
