<?php
/**
 * GravityBoard Attachments
 *
 * @package GravityBoard
 */

namespace GravityKit\GravityBoard\Attachments;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use GF_Entry_List_Table;
use GravityKit\GravityBoard\Helpers;
use GravityKit\GravityBoard\Ajax;
use GravityKit\GravityBoard\Feed;
use GravityKit\GravityBoard\Notifications;
use WP_Error;
use GFFormsModel;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Class Ajax
 *
 * Handles AJAX requests for GravityBoard operations.
 *
 * @since 1.0
 */
class Attachments {

	/**
	 * Instance of the class.
	 *
	 * @var Attachments
	 */
	public static $instance;

	/**
	 * The meta key used to store attachments on an entry.
	 *
	 * @since 1.1
	 *
	 * @var string
	 */
	const ATTACHMENT_META_KEY = 'gravityboard_attachments';

	/**
	 * The directory slug for GravityBoard attachments within a form's upload folder.
	 *
	 * @since 1.3
	 *
	 * @var string
	 */
	const ATTACHMENTS_DIR_SLUG = 'gravityboard';

	/**
	 * The prefix for attachment IDs.
	 *
	 * @since 1.3
	 *
	 * @var string
	 */
	const ATTACHMENT_ID_PREFIX = 'attachment_';

	/**
	 * Get the instance of the class.
	 *
	 * @since 1.0
	 *
	 * @return Attachments The instance of the class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.0
	 */
	public function __construct() {
		add_filter( 'gform_entry_meta', [ $this, 'filter_entry_meta' ], 10, 2 );
		add_filter( 'gk/gravityboard/data/lanes', [ $this, 'add_attachments_to_lanes' ], 10, 3 );
		add_action( 'gk/gravityboard/ajax/register-routes', [ $this, 'register_routes' ] );

		// Hook to format attachment display in Gravity Forms entries table.
		add_filter( 'gform_entries_field_value', [ $this, 'filter_entries_field_value' ], 10, 4 );
	}

	/**
	 * Filter the entry meta.
	 *
	 * @since 1.0
	 *
	 * @param array $entry_meta The entry meta.
	 *
	 * @return array The entry meta.
	 */
	public function filter_entry_meta( $entry_meta ) {
		$entry_meta[ self::ATTACHMENT_META_KEY ] = [
			'label'             => esc_html__( 'GravityBoard Attachments', 'gk-gravityboard' ),
			'is_numeric'        => false,
			'is_default_column' => false,
			'filter'            => [
				'operators' => [ 'is', 'isnot', 'contains' ],
			],
		];

		return $entry_meta;
	}

	/**
	 * Filter the entry list field value to properly display attachments.
	 *
	 * This method formats attachment data for display in the Gravity Forms entries table.
	 * It shows attachment counts and filenames in a user-friendly format, and includes
	 * hidden search data to make attachments searchable by filename.
	 *
	 * @since 1.0
	 *
	 * @param string $value The current value of the field.
	 * @param int    $form_id The ID of the form.
	 * @param string $field_id The ID of the field.
	 * @param array  $entry The entry object.
	 *
	 * @return string The filtered value of the field.
	 */
	public function filter_entries_field_value( $value, $form_id, $field_id, $entry ) {
		if ( self::ATTACHMENT_META_KEY !== $field_id ) {
			return $value;
		}

		$attachments = self::get_attachments( $entry['id'] );

		if ( is_wp_error( $attachments ) || empty( $attachments ) ) {
			return '<span class="screen-reader-text">' . esc_html__( 'No attachments', 'gk-gravityboard' ) . '</span>';
		}

		$attachment_count = count( $attachments );
		$attachment_names = array_map(
            function ( $attachment ) {
                return $attachment['name'] ?? __( 'Unnamed file', 'gk-gravityboard' );
            },
            $attachments
        );

		// For search purposes, we include the filenames in a hidden span.
		$search_data = '<span style="display:none;">' . esc_html( implode( ' ', $attachment_names ) ) . '</span>';

		$text = sprintf(
			/* translators: %d: number of attachments. */
			esc_html( _n( '📎 %d file', '📎 %d files', $attachment_count, 'gk-gravityboard' ) ),
			$attachment_count
		) . $search_data;

		$entry_list_table = new GF_Entry_List_Table();
		$url              = $entry_list_table->get_detail_url( $entry ) . '#' . self::ATTACHMENT_META_KEY;

		return sprintf( '<a href="%s">%s</a>', $url, $text );
	}

	/**
	 * Register REST API routes for attachments.
	 *
	 *  Attachments
	 *    • POST   /gravityboard/v1/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/attachments
	 *    • DELETE /gravityboard/v1/boards/(?P<feed_id>\d+)/cards/(?P<entry_id>\d+)/attachments/(?P<attachment_id>\d+)
	 *
	 * @since 1.3
	 */
	public function register_routes() {
		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\\d+)/cards/(?P<entry_id>\\d+)/attachments',
			[
				'methods'             => 'POST',
				'callback'            => [ $this, 'rest_upload_attachment' ],
				'permission_callback' => [ $this, 'can_upload_attachment' ],
			]
		);

		register_rest_route(
			Ajax::NAMESPACE,
			'/boards/(?P<feed_id>\\d+)/cards/(?P<entry_id>\\d+)/attachments/(?P<attachment_id>[\\w-]+)',
			[
				'methods'             => 'DELETE',
				'callback'            => [ $this, 'rest_delete_attachment' ],
				'permission_callback' => [ $this, 'can_delete_attachment' ],
			]
		);
	}

	/**
	 * Permission check for uploading an attachment.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool|WP_Error True if permission is granted, otherwise WP_Error.
	 */
	public function can_upload_attachment( WP_REST_Request $request ) {
		$feed = Ajax::get_instance()->get_feed_from_request( $request );
		if ( ! $feed || is_wp_error( $feed ) ) {
			return new WP_Error( 'gk_gravityboard_feed_not_found', esc_html__( 'Feed not found.', 'gk-gravityboard' ), [ 'status' => 404 ] );
		}
		if ( ! Helpers::user_has_permission( 'add_attachments', $feed ) ) {
			return new WP_Error( 'gk_gravityboard_permission_denied', esc_html__( 'You do not have permission to add attachments.', 'gk-gravityboard' ), [ 'status' => 403 ] );
		}
		return true;
	}

	/**
	 * Permission check for deleting an attachment.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 * @return bool|WP_Error True if permission is granted, otherwise WP_Error.
	 */
	public function can_delete_attachment( WP_REST_Request $request ) {
		$feed = Ajax::get_instance()->get_feed_from_request( $request );
		if ( ! $feed || is_wp_error( $feed ) ) {
			return new WP_Error( 'gk_gravityboard_feed_not_found', esc_html__( 'Feed not found.', 'gk-gravityboard' ), [ 'status' => 404 ] );
		}
		if ( ! Helpers::user_has_permission( 'delete_attachments', $feed ) ) {
			return new WP_Error( 'gk_gravityboard_permission_denied', esc_html__( 'You do not have permission to delete attachments.', 'gk-gravityboard' ), [ 'status' => 403 ] );
		}
		return true;
	}

	/**
	 * REST API handler for uploading an attachment to a card.
	 *
	 * @since 1.3
	 * @param WP_REST_Request $request The REST request.
	 * @return WP_REST_Response|WP_Error
	 */
	public function rest_upload_attachment( WP_REST_Request $request ) {
		$feed     = Ajax::get_instance()->get_feed_from_request( $request );
		$entry    = Ajax::get_instance()->get_entry_from_request( $request ); // Fetches entry using entry_id from request.
		$entry_id = $entry['id'];
		$form_id  = (int) $feed['form_id'];

		$files = $request->get_file_params();
		if ( empty( $files['file'] ) ) {
			return new WP_Error( 'gk_gravityboard_no_file_uploaded', esc_html__( 'No file uploaded.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$file = $files['file'];

		$uploaded_file = self::handle_entry_attachment_upload( $file, $form_id, $entry_id, $feed );

		if ( is_wp_error( $uploaded_file ) ) {
			return new WP_Error( 'gk_gravityboard_upload_failed', $uploaded_file->get_error_message(), [ 'status' => 500 ] );
		}

		$attachment                = self::prepare_attachment_data( $uploaded_file );
		$updated_attachments_array = self::add_attachment( $entry_id, $feed, $attachment );

		if ( is_wp_error( $updated_attachments_array ) ) {
			return new WP_Error( 'gk_gravityboard_meta_update_failed', $updated_attachments_array->get_error_message(), [ 'status' => 500 ] );
		}

		$client_safe_attachment = $attachment;
		unset( $client_safe_attachment['file_path'] );

		return new WP_REST_Response(
			[
				'message'           => esc_html__( 'File uploaded successfully.', 'gk-gravityboard' ),
				'attachment'        => $client_safe_attachment,
				'attachments_count' => count( $updated_attachments_array ),
			],
			200
		);
	}

	/**
	 * REST API handler for deleting an attachment from a card.
	 *
	 * @since 1.3
	 * @param WP_REST_Request $request The REST request.
	 * @return WP_REST_Response|WP_Error
	 */
	public function rest_delete_attachment( WP_REST_Request $request ) {
		$feed                    = Ajax::get_instance()->get_feed_from_request( $request );
		$entry                   = Ajax::get_instance()->get_entry_from_request( $request );
		$entry_id                = $entry['id'];
		$attachment_id_to_delete = sanitize_text_field( $request->get_param( 'attachment_id' ) );

		if ( empty( $attachment_id_to_delete ) ) {
			return new WP_Error( 'gk_gravityboard_attachment_id_required', esc_html__( 'Attachment ID is required.', 'gk-gravityboard' ), [ 'status' => 400 ] );
		}

		$delete_result = self::delete_attachment( $entry_id, $feed, $attachment_id_to_delete );

		if ( is_wp_error( $delete_result ) ) {
			$error_code    = $delete_result->get_error_code();
			$error_message = $delete_result->get_error_message();
			$status_code   = 500; // Default server error.

			if ( 'no_attachments' === $error_code ) {
				$status_code = 404; // Not found.
			} elseif ( 'file_delete_failed' === $error_code || 'meta_update_failed' === $error_code || 'invalid_file_path' === $error_code ) {
				$status_code = 500; // Internal server error.
			}

			return new WP_Error( 'gk_gravityboard_' . $error_code, $error_message, [ 'status' => $status_code ] );
		}

		return new WP_REST_Response(
			[
				'message'               => esc_html__( 'Attachment deleted successfully.', 'gk-gravityboard' ),
				'attachments_count'     => count( $delete_result['updated_attachments'] ),
				'deleted_attachment_id' => $attachment_id_to_delete,
			],
			200
		);
	}

	/**
	 * Add attachments to the lanes.
	 *
	 * @param array $lanes The lanes data.
	 * @param array $entry The entry data.
	 * @param array $feed The feed data.
	 * @return array The lanes data.
	 */
	public function add_attachments_to_lanes( $lanes, $entry, $feed ) {

		// Only show notes if enabled and user has permission.
		$enable_attachments   = (bool) rgars( $feed, 'meta/enable_card_attachments', false );
		$can_view_attachments = Helpers::user_has_permission( 'view_attachments', $feed );

		if ( ! $enable_attachments || ! $can_view_attachments ) {
			return $lanes;
		}

		foreach ( $lanes as &$lane ) {
			foreach ( $lane['cards'] as &$card ) {
				$card = $this->add_attachments( $card );
			}
		}
		return $lanes;
	}

	/**
	 * Add attachments to the card.
	 *
	 * @param array $card The card data.
	 *
	 * @return array The card data.
	 */
	public function add_attachments( $card ) {
		$card['attachments']       = [];
		$card['attachments_count'] = 0;

		$entry_attachments = self::get_attachments( $card['id'] );

		if ( is_array( $entry_attachments ) ) {

			// We don't need to send the full file_path to the client.
			$client_safe_attachments   = array_map(
				function ( $att ) {
					unset( $att['file_path'] );
					return $att;
				},
				$entry_attachments
			);
			$card['attachments']       = $client_safe_attachments;
			$card['attachments_count'] = count( $client_safe_attachments );
		}

		return $card;
	}

	/**
	 * Get attachments for an entry.
	 *
	 * @since 1.1
	 *
	 * @param int $entry_id The entry ID.
	 * @return array|WP_Error Array of attachment objects, an empty array if none, or WP_Error on failure.
	 */
	public static function get_attachments( $entry_id ) {
		// Validate entry_id is a positive integer.
		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID', 'gk-gravityboard' ) );
		}

		// Verify the entry exists.
		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) ) {
			/** @var WP_Error $entry */
			return $entry;
		}

		$attachments = gform_get_meta( $entry_id, self::ATTACHMENT_META_KEY );
		return is_array( $attachments ) ? $attachments : [];
	}

	/**
	 * Get an attachment by ID.
	 *
	 * @since 1.1
	 *
	 * @param int    $entry_id      The entry ID.
	 * @param array  $feed          The feed array.
	 * @param string $attachment_id The attachment ID.
	 *
	 * @return array|null|WP_Error The attachment object, or null if not found, or WP_Error on failure.
	 */
	public static function get_attachment_by_id( $entry_id, $feed, $attachment_id ) {
		// Validate entry_id is a positive integer.
		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID', 'gk-gravityboard' ) );
		}

		// Verify the entry exists.
		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) ) {
			/** @var WP_Error $entry */
			return $entry;
		}

		// Check if current user has permission to view attachments.
		if ( ! Helpers::user_has_permission( 'view_attachments', $feed, $entry['user_id'] ?? null ) ) {
			return new WP_Error( 'missing_permission', esc_html__( 'Current user does not have permission to view attachments', 'gk-gravityboard' ) );
		}

		// Sanitize the attachment ID.
		$attachment_id = sanitize_text_field( $attachment_id );
		if ( empty( $attachment_id ) ) {
			return new WP_Error( 'invalid_attachment_id', esc_html__( 'Invalid attachment ID', 'gk-gravityboard' ) );
		}

		$attachments = self::get_attachments( $entry_id );

		if ( empty( $attachments ) ) {
			return null;
		}

		foreach ( $attachments as $attachment ) {
			if ( isset( $attachment['id'] ) && $attachment['id'] === $attachment_id ) {
				return $attachment;
			}
		}

		return null;
	}

	/**
	 * Update attachments for an entry.
	 *
	 * @since 1.1
	 *
	 * @param int   $entry_id          The entry ID.
	 * @param array $feed              The feed data.
	 * @param array $attachments_array The array of attachment objects to save.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public static function update_attachments( $entry_id, $feed, $attachments_array ) {
		// Validate entry_id is a positive integer.
		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID', 'gk-gravityboard' ) );
		}

		// Verify the entry exists.
		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) ) {
			/** @var WP_Error $entry */
			return $entry;
		}

		// Check if current user has permission to update attachments.
		if ( ! Helpers::user_has_permission( 'add_attachments', $feed, $entry['user_id'] ?? null ) ) {
			return new WP_Error( 'missing_permission', esc_html__( 'Current user does not have permission to update attachments', 'gk-gravityboard' ) );
		}

		// Ensure attachments_array is an array.
		if ( ! is_array( $attachments_array ) ) {
			$attachments_array = [];
		}

		// Sanitize each attachment in the array.
		$sanitized_attachments = [];
		foreach ( $attachments_array as $attachment ) {
			$sanitized_attachment = self::sanitize_attachment( $attachment );

			// Ensure required fields are present.
			if ( ! empty( $sanitized_attachment['id'] ) ) {
				$sanitized_attachments[] = $sanitized_attachment;
			}
		}

		// Update the meta with sanitized attachments.
		$result = gform_update_meta( $entry_id, self::ATTACHMENT_META_KEY, $sanitized_attachments );

		if ( $result ) {

			/**
			 * Runs when attachments have been updated, before the notification is triggered.
			 *
			 * @since 1.0
			 *
			 * @param array $sanitized_attachments The sanitized attachments array.
			 * @param array $entry The entry array.
			 * @param array $feed The feed array.
			 */
			do_action( 'gk/gravityboard/attachments/updated', $sanitized_attachments, $entry, $feed );

			Notifications::get_instance()->maybe_trigger_notifications(
				$entry,
				'gk/gravityboard/attachments/updated',
				$feed
			);

			return true;
		}

		return new WP_Error( 'update_failed', esc_html__( 'Failed to update attachment metadata', 'gk-gravityboard' ) );
	}

	/**
	 * Add a single attachment to an entry's attachment meta.
	 *
	 * @since 1.1
	 *
	 * @param int   $entry_id        The entry ID.
	 * @param array $feed            The feed array.
	 * @param array $attachment_data The data for the new attachment.
	 * @return array|WP_Error The updated array of all attachments on success, WP_Error on failure.
	 */
	public static function add_attachment( $entry_id, $feed, $attachment_data ) {
		// Validate entry_id is a positive integer.
		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID', 'gk-gravityboard' ) );
		}

		// Verify the entry exists.
		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) ) {
			/** @var WP_Error $entry */
			return $entry;
		}

		// Check if current user has permission to add attachments.
		if ( ! Helpers::user_has_permission( 'add_attachments', $feed, $entry['user_id'] ?? null ) ) {
			return new WP_Error( 'missing_permission', esc_html__( 'Current user does not have permission to add attachments', 'gk-gravityboard' ) );
		}

		// Validate attachment_data is an array and has required fields.
		if ( ! is_array( $attachment_data ) || empty( $attachment_data['id'] ) ) {
			return new WP_Error( 'invalid_attachment_data', esc_html__( 'Invalid attachment data', 'gk-gravityboard' ) );
		}

		// Sanitize the attachment data.
		$sanitized_data = self::sanitize_attachment( $attachment_data, $feed );

		// Ensure required fields are present after sanitization.
		if ( empty( $sanitized_data['id'] ) ) {
			return new WP_Error( 'missing_attachment_id', esc_html__( 'Missing attachment ID', 'gk-gravityboard' ) );
		}

		// Get existing attachments and add the new one.
		$attachments = self::get_attachments( $entry_id );

		// Handle potential WP_Error from get_attachments.
		if ( is_wp_error( $attachments ) ) {
			return $attachments;
		}

		// Check for duplicate attachment IDs.
		foreach ( $attachments as $attachment ) {
			if ( isset( $attachment['id'] ) && $attachment['id'] === $sanitized_data['id'] ) {
				return new WP_Error( 'duplicate_attachment_id', esc_html__( 'Attachment with this ID already exists', 'gk-gravityboard' ) );
			}
		}

		$attachments[] = $sanitized_data;

		// Update attachments with the new array.
		$update_result = self::update_attachments( $entry_id, $feed, $attachments );

		if ( is_wp_error( $update_result ) ) {
			return $update_result;
		}

		if ( $update_result ) {

			/**
			 * Runs when attachments have been added, before the notification is triggered.
			 *
			 * @since 1.0
			 *
			 * @param array $attachments The attachments array.
			 * @param array $entry The entry array.
			 * @param array $feed The feed array.
			 */
			do_action( 'gk/gravityboard/attachments/added', $attachments, $entry, $feed );

			Notifications::get_instance()->maybe_trigger_notifications(
				$entry,
				'gk/gravityboard/attachments/added',
				$feed
			);

			return $attachments;
		}

		return new WP_Error( 'attachment_update_failed', esc_html__( 'Failed to update attachments', 'gk-gravityboard' ) );
	}

	/**
	 * Delete a single attachment from an entry's attachment meta.
	 *
	 * @since 1.1
	 *
	 * @param int    $entry_id                The entry ID.
	 * @param array  $feed                    The feed array.
	 * @param string $attachment_id_to_delete The unique ID of the attachment to delete.
	 * @return array|null|WP_Error An array containing ['updated_attachments' => array, 'deleted_file_path' => string|null] on success,
	 *                    or null if the attachment was not found or update failed.
	 */
	public static function delete_attachment( $entry_id, $feed, $attachment_id_to_delete ) {
		// Validate entry_id is a positive integer.
		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', 'Invalid entry ID' );
		}

		// Verify the entry exists.
		$entry = Helpers::get_card( $entry_id );
		if ( is_wp_error( $entry ) ) {
			/** @var WP_Error $entry */
			return $entry;
		}

		// Check if current user has permission to delete attachments.
		if ( ! Helpers::user_has_permission( 'delete_attachments', $feed, $entry['user_id'] ?? null ) ) {
			return new WP_Error( 'missing_permission', esc_html__( 'Current user does not have permission to delete attachments', 'gk-gravityboard' ) );
		}

		// Sanitize the attachment ID to delete.
		$attachment_id_to_delete = sanitize_text_field( $attachment_id_to_delete );
		if ( empty( $attachment_id_to_delete ) ) {
			return new WP_Error( 'invalid_attachment_id', esc_html__( 'Invalid attachment ID', 'gk-gravityboard' ) );
		}

		// Get existing attachments.
		$attachment = self::get_attachment_by_id( $entry_id, $feed, $attachment_id_to_delete );
		if ( is_wp_error( $attachment ) ) {
			return $attachment;
		}

		if ( empty( $attachment ) ) {
			return new WP_Error( 'no_attachments', esc_html__( 'The attachment was not found.', 'gk-gravityboard' ) );
		}

		$deleted_file_path = rgar( $attachment, 'file_path' );
		$file_url          = rgar( $attachment, 'url', '' );

		// Check if this is a file that belongs to the current site.
		if ( ! $deleted_file_path || strpos( $deleted_file_path, wp_get_upload_dir()['basedir'] ) !== 0 ) {
			return new WP_Error( 'invalid_file_path', esc_html__( 'The attachment file path is invalid.', 'gk-gravityboard' ) );
		}

		// Get all attachments for this entry.
		$attachments = self::get_attachments( $entry_id );

		if ( is_wp_error( $attachments ) ) {
			return $attachments;
		}

		// Filter out the attachment to be deleted.
		$updated_attachments = array_filter(
			$attachments,
			function ( $att ) use ( $attachment_id_to_delete ) {
				return $att['id'] !== $attachment_id_to_delete;
			}
		);

		// Delete the physical file using WordPress Filesystem.
		$file_deleted = Helpers::delete_file( $deleted_file_path, $entry_id, $file_url );

		if ( ! $file_deleted ) {
			return new WP_Error( 'file_delete_failed', esc_html__( 'Failed to delete the attachment file.', 'gk-gravityboard' ) );
		}

		// Update the attachments meta.
		$updated = self::update_attachments( $entry_id, $feed, $updated_attachments );
		if ( is_wp_error( $updated ) ) {
			return $updated;
		}

		if ( $updated ) {

			/**
			 * Runs when attachments have been removed, before the notification is triggered.
			 *
			 * @since 1.0
			 *
			 * @param array $updated_attachments The updated attachments array.
			 * @param array $entry The entry array.
			 * @param array $feed The feed array.
			 * @param string $attachment_id_to_delete The ID of the attachment that was deleted.
			 */
			do_action( 'gk/gravityboard/attachments/removed', $updated_attachments, $entry, $feed, $attachment_id_to_delete );

			Notifications::get_instance()->maybe_trigger_notifications(
				$entry,
				'gk/gravityboard/attachments/removed',
				$feed
			);

			return [
				'updated_attachments' => $updated_attachments,
				'deleted_file_path'   => $deleted_file_path,
				'file_deleted'        => $file_deleted,
			];
		}

		return new WP_Error( 'meta_update_failed', esc_html__( 'Failed to update attachments meta.', 'gk-gravityboard' ) );
	}

	/**
	 * Prepare attachment data.
	 *
	 * @since 1.3
	 *
	 * @param array $uploaded_file The uploaded file array.
	 *
	 * @return array The prepared attachment data.
	 */
	public static function prepare_attachment_data( $uploaded_file ) {
		$original_filename = basename( $uploaded_file['file'] );

		$current_user_id = get_current_user_id();
		$attachment_data = [
			'name'      => $original_filename,
			'url'       => $uploaded_file['url'],
			'type'      => $uploaded_file['type'],
			'file_path' => $uploaded_file['file'],
			'user_id'   => (int) $current_user_id,
			'timestamp' => time(),
			'id'        => uniqid( self::ATTACHMENT_ID_PREFIX ), // Generate a unique ID for this attachment.
		];

		return $attachment_data;
	}

	/**
	 * Sanitizes attachment data.
	 *
	 * @since 1.1
	 *
	 * @param array $attachment_data The raw attachment data.
	 * @param array $feed The feed data.
	 *
	 * @return array Sanitized attachment data.
	 */
	private static function sanitize_attachment( $attachment_data, $feed = [] ) {
		if ( ! is_array( $attachment_data ) ) {
			return [];
		}

		$sanitized_data = [];
		$allowed_keys   = [
			'id',
			'name',
			'type',
			'size',
			'url',
			'file_path',
			'date_created',
			'user_id',
			'timestamp',
			'feed_id',
		];

		foreach ( $allowed_keys as $key ) {
			if ( isset( $attachment_data[ $key ] ) ) {
				switch ( $key ) {
					case 'id':
					case 'feed_id':
					case 'timestamp':
					case 'type':
					case 'date_created':
						$sanitized_data[ $key ] = sanitize_text_field( $attachment_data[ $key ] );
						break;
					case 'name':
						$sanitized_data[ $key ] = sanitize_file_name( $attachment_data[ $key ] );
						break;
					case 'url':
						$sanitized_data[ $key ] = esc_url_raw( $attachment_data[ $key ] );
						break;
					case 'file_path':
						$sanitized_data[ $key ] = wp_normalize_path( $attachment_data[ $key ] );
						break;
					case 'user_id':
					case 'size':
						$sanitized_data[ $key ] = absint( $attachment_data[ $key ] );
						break;
				}
			}
		}

		// Add current user ID and timestamp if not provided.
		if ( ! isset( $sanitized_data['user_id'] ) ) {
			$sanitized_data['user_id'] = get_current_user_id();
		}

		if ( ! isset( $sanitized_data['date_created'] ) ) {
			$sanitized_data['date_created'] = current_time( 'mysql', 1 );
		}

		if ( ! isset( $sanitized_data['feed_id'] ) ) {
			$sanitized_data['feed_id'] = $feed['id'] ?? null;
		}

		return $sanitized_data;
	}

	/**
	 * Handles the actual file upload process for an entry attachment.
	 *
	 * This method sets up a custom upload directory, manipulates the 'upload_dir'
	 * filter to direct wp_handle_upload to the correct path, performs the upload,
	 * and then restores the original filter state.
	 *
	 * @since 1.3
	 *
	 * @param array $file_array The specific file array from $_FILES (e.g., $_FILES['file']).
	 * @param int   $form_id    The ID of the form.
	 * @param int   $entry_id   The ID of the entry.
	 * @param array $feed       The feed configuration.
	 * @return array{file: string, url: string, type: string}|WP_Error An array on success,
	 *                        or WP_Error on failure.
	 */
	public static function handle_entry_attachment_upload( $file_array, $form_id, $entry_id, $feed = [] ) {
		if ( empty( $file_array ) || ! isset( $file_array['tmp_name'] ) ) {
			return new WP_Error( 'no_file_provided', esc_html__( 'No file data provided for upload.', 'gk-gravityboard' ) );
		}

		// Get the target directory for this entry's attachments.
		$upload_dir_parts = self::get_or_create_entry_attachments_upload_dir( $form_id, $entry_id );
		if ( is_wp_error( $upload_dir_parts ) ) {
			return $upload_dir_parts; // Pass the error up.
		}

		$target_path = $upload_dir_parts['path'];
		$target_url  = $upload_dir_parts['url'];

		// Store the original upload_dir filter state if it exists.
		$original_upload_dir_filters = null;
		if ( has_filter( 'upload_dir' ) ) {
			$original_upload_dir_filters = $GLOBALS['wp_filter']['upload_dir']; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
			// Clear any existing upload_dir filters to avoid conflicts, if we have something to restore.
			remove_all_filters( 'upload_dir' );
		}

		// Get basedir before the filter to avoid recursion in the callback.
		$basedir_global = wp_get_upload_dir()['basedir'];

		$upload_dir_filter_callback = function ( $param ) use ( $target_path, $target_url, $basedir_global ) {
			$param['path']   = untrailingslashit( $target_path );
			$param['url']    = $target_url;
			$param['subdir'] = str_replace( $basedir_global, '', $target_path ); // Subdir relative to WP uploads base.
			return $param;
		};
		add_filter( 'upload_dir', $upload_dir_filter_callback, 10, 1 );

		// Ensure wp_handle_upload is available.
		if ( ! function_exists( 'wp_handle_upload' ) ) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}

		$file_type      = $file_array['type'];
		$file_extension = pathinfo( $file_array['name'], PATHINFO_EXTENSION );

		// Check if the feed allows all file types.
		$allow_all_file_types = (bool) rgars( $feed, 'meta/allow_all_attachment_types', false );

		if ( $allow_all_file_types ) {
			add_filter(
				'upload_mimes',
				function ( $mimes ) use ( $file_type, $file_extension ) {
					// Get the file extension from the file array.
					$mimes[ $file_extension ] = $file_type;
					return $mimes;
				},
				10000,
				1
			);

			$allow_any_filetype_check = function ( $data, $file, $filename ) use ( $file_type, $file_extension ) {
				// If filetype check fails, bypass it by faking a type.
				if ( empty( $data['ext'] ) || empty( $data['type'] ) ) {
					return [
						'ext'             => $file_extension,
						'type'            => $file_type,
						'proper_filename' => $filename,
					];
				}

				return $data;
			};
			add_filter( 'wp_check_filetype_and_ext', $allow_any_filetype_check, 10, 4 );
		}

		$upload_overrides = [
			'test_form' => false,
		];
		$uploaded_file    = wp_handle_upload( $file_array, $upload_overrides );

		// Remove our specific filter.
		remove_filter( 'upload_dir', $upload_dir_filter_callback, 10 );

		// Restore original filters if they existed.
		if ( $original_upload_dir_filters ) {
			$GLOBALS['wp_filter']['upload_dir'] = $original_upload_dir_filters; // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		}

		if ( isset( $uploaded_file['error'] ) ) {
			return new WP_Error( 'upload_failed', $uploaded_file['error'] );
		}

		return $uploaded_file;
	}

	/**
	 * Get the GravityBoard attachments directory path and URL for a specific entry,
	 * and ensure the directory exists.
	 *
	 * @since 1.3
	 *
	 * @param int $form_id  The ID of the form.
	 * @param int $entry_id The ID of the entry.
	 * @return array|WP_Error An array with 'path' and 'url' keys for the attachments directory, or WP_Error on failure.
	 */
	public static function get_or_create_entry_attachments_upload_dir( $form_id, $entry_id ) {
		$form_id = absint( $form_id );
		if ( empty( $form_id ) ) {
			return new WP_Error( 'invalid_form_id', esc_html__( 'Invalid form ID for attachments directory.', 'gk-gravityboard' ) );
		}

		$entry_id = absint( $entry_id );
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'invalid_entry_id', esc_html__( 'Invalid entry ID for attachments directory.', 'gk-gravityboard' ) );
		}

		$base_dir_parts = self::get_form_base_upload_dir_parts( $form_id );

		if ( is_wp_error( $base_dir_parts ) ) {
			return $base_dir_parts;
		}

		$entry_specific_sub_dir = wp_hash( $entry_id );

		$target_path = trailingslashit( $base_dir_parts['path'] . self::ATTACHMENTS_DIR_SLUG ) . trailingslashit( $entry_specific_sub_dir );
		$target_url  = trailingslashit( $base_dir_parts['url'] . self::ATTACHMENTS_DIR_SLUG ) . $entry_specific_sub_dir;

		if ( ! wp_mkdir_p( $target_path ) ) {
			// translators: %s: Directory path.
			$error_message = sprintf( esc_html__( 'Could not create attachment directory: %s.', 'gk-gravityboard' ), $target_path );

			Feed::get_instance()->log_error( __METHOD__ . ' ' . $error_message );

			return new \WP_Error( 'gravityboard_attachment_dir_creation_failed', $error_message );
		}

		return [
			'path' => $target_path,
			'url'  => $target_url,
		];
	}

	/**
	 * Get the base upload directory path and URL for a given form.
	 *
	 * @since 1.3
	 *
	 * @param int $form_id The ID of the form.
	 * @return array|WP_Error An array with 'path' and 'url' keys, or WP_Error on failure.
	 */
	private static function get_form_base_upload_dir_parts( $form_id ) {
		$form_id = absint( $form_id );
		if ( empty( $form_id ) ) {
			return new WP_Error( 'invalid_form_id', esc_html__( 'Invalid form ID provided.', 'gk-gravityboard' ) );
		}

		// Pass a dummy filename as it might be used for uniqueness checks by GF,.
		// but we are primarily interested in the directory path it returns.
		$upload_parts = GFFormsModel::get_file_upload_path( $form_id, 'temp.txt' );

		if ( empty( $upload_parts ) || empty( $upload_parts['path'] ) || empty( $upload_parts['url'] ) ) {
			return new \WP_Error(
				'gravityboard_attachment_gf_upload_path_parts_failed',
				// translators: %d: Form ID.
				sprintf( esc_html__( 'Could not get Gravity Forms file upload path parts for form ID: %d.', 'gk-gravityboard' ), $form_id )
			);
		}

		// The path returned by get_file_upload_path is for the file itself, so we need its directory.
		$base_path = trailingslashit( dirname( $upload_parts['path'] ) );
		$base_url  = trailingslashit( dirname( $upload_parts['url'] ) );

		return [
			'path' => $base_path,
			'url'  => $base_url,
		];
	}
}
